import React, { useState, useEffect } from 'react';
import { useForm } from 'react-hook-form';
import { toast } from 'sonner';
import { 
  Globe, Phone, Mail, Clock, CheckCircle, FileText, Shield, 
  HelpCircle, ArrowRight, Users, MapPin, AlertCircle, Info,
  BookOpen, Target, Award, Workflow, Timer, MessageSquare
} from 'lucide-react';
import { createNRIAssistanceRequest, NRIAssistanceRequest } from '../../api/nriAssistance';

export function NRIAssistance() {
  const [activeTab, setActiveTab] = useState('overview');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [properties, setProperties] = useState([]);
  const [propertyValidationError, setPropertyValidationError] = useState('');
  
  const { register, handleSubmit, formState: { errors }, reset, watch } = useForm<NRIAssistanceRequest>();
  const watchedPropertyId = watch('propertyId');
  const watchedPropertyArea = watch('propertyArea');
  const watchedPropertyCity = watch('propertyCity');

  useEffect(() => {
    fetchProperties();
  }, []);

  useEffect(() => {
    if (watchedPropertyId && watchedPropertyArea && watchedPropertyCity) {
      validatePropertyDetails();
    } else {
      setPropertyValidationError('');
    }
  }, [watchedPropertyId, watchedPropertyArea, watchedPropertyCity]);

  const fetchProperties = async () => {
    try {
      const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
      const response = await fetch(`${API_BASE}/properties/`);
      if (response.ok) {
        const data = await response.json();
        setProperties(data.properties || data || []);
      }
    } catch (error) {
      console.error('Error fetching properties:', error);
    }
  };

  const validatePropertyDetails = () => {
    const property = properties.find(p => 
      String(p.id) === String(watchedPropertyId) || 
      String(p.property_id) === String(watchedPropertyId) ||
      p.id === parseInt(watchedPropertyId) ||
      p.property_id === parseInt(watchedPropertyId)
    );
    
    if (!property) {
      setPropertyValidationError('Property ID not found');
      return;
    }
    
    const areaMatch = property.locality?.toLowerCase().includes(watchedPropertyArea?.toLowerCase()) ||
                     property.area?.toLowerCase().includes(watchedPropertyArea?.toLowerCase()) ||
                     watchedPropertyArea?.toLowerCase().includes(property.locality?.toLowerCase()) ||
                     watchedPropertyArea?.toLowerCase().includes(property.area?.toLowerCase());
    const cityMatch = property.city?.toLowerCase().includes(watchedPropertyCity?.toLowerCase()) ||
                     watchedPropertyCity?.toLowerCase().includes(property.city?.toLowerCase());
    
    if (!areaMatch) {
      setPropertyValidationError('Area/Locality does not match the property');
      return;
    }
    
    if (!cityMatch) {
      setPropertyValidationError('City does not match the property');
      return;
    }
    
    setPropertyValidationError('');
  };

  const onSubmit = async (data: NRIAssistanceRequest) => {
    // Check if user is logged in
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const accessToken = localStorage.getItem('access_token');
    const loginType = localStorage.getItem('login_type');
    
    console.log('Login type:', loginType);
    console.log('User object:', user);
    
    // Check if user is logged in at all
    if (!accessToken || !user.email) {
      toast.error('Please login to submit NRI assistance requests.');
      return;
    }
    
    // User is logged in, allow NRI assistance request (User = buyer)
    
    // Check property validation
    if (propertyValidationError) {
      toast.error('Please fix property details errors before submitting.');
      return;
    }
    
    setIsSubmitting(true);
    try {
      await createNRIAssistanceRequest(data);
      toast.success('Request submitted successfully! You will receive a confirmation email shortly.');
      reset();
      setActiveTab('overview');
    } catch (error) {
      console.error('Error submitting request:', error);
      toast.error('Failed to submit request. Please try again.');
    } finally {
      setIsSubmitting(false);
    }
  };

  const services = [
    {
      title: 'Property Investment Guidance',
      description: 'Expert advice on property investment opportunities, market analysis, and legal compliance for NRIs',
      icon: MapPin,
      features: ['Market Research', 'Investment Analysis', 'Legal Compliance', 'Documentation Support']
    },
    {
      title: 'Legal & Documentation Support',
      description: 'Comprehensive assistance with legal procedures, documentation, and regulatory compliance',
      icon: FileText,
      features: ['FEMA Compliance', 'POA Documentation', 'Legal Procedures', 'Regulatory Guidance']
    },
    {
      title: 'Banking & Financial Services',
      description: 'Support with NRE/NRO accounts, remittances, and financial planning for property investments',
      icon: Shield,
      features: ['NRE/NRO Accounts', 'Remittance Guidance', 'Tax Planning', 'Financial Advisory']
    },
    {
      title: 'Property Verification',
      description: 'Thorough verification of property documents, ownership, and legal status before investment',
      icon: CheckCircle,
      features: ['Title Verification', 'Document Check', 'Legal Status', 'Ownership Validation']
    }
  ];

  const processSteps = [
    {
      step: 1,
      title: 'Submit Request',
      description: 'Fill out our assistance request form with your specific requirements and contact details',
      icon: MessageSquare,
      timeframe: '5 minutes'
    },
    {
      step: 2,
      title: 'Initial Review',
      description: 'Our team reviews your request and assigns a dedicated assistance officer within 24 hours',
      icon: Users,
      timeframe: '24 hours'
    },
    {
      step: 3,
      title: 'Consultation',
      description: 'Detailed consultation with our experts to understand your needs and provide guidance',
      icon: Phone,
      timeframe: '2-3 days'
    },
    {
      step: 4,
      title: 'Resolution',
      description: 'Complete assistance with documentation, procedures, and follow-up until resolution',
      icon: CheckCircle,
      timeframe: '5-10 days'
    }
  ];

  const faqs = [
    {
      question: 'What is the NRI Assistance Desk?',
      answer: 'The NRI Assistance Desk is a dedicated support service for Non-Resident Indians seeking help with property investments, legal procedures, documentation, and other India-related matters.'
    },
    {
      question: 'What types of issues can you help with?',
      answer: 'We assist with property investment guidance, legal documentation, FEMA compliance, banking procedures, property verification, and general advisory services for NRIs.'
    },
    {
      question: 'How long does it take to resolve requests?',
      answer: 'Resolution time varies based on complexity. Simple queries are resolved within 2-3 days, while complex documentation or legal matters may take 5-10 days.'
    },
    {
      question: 'Is there a fee for the assistance service?',
      answer: 'Basic consultation and guidance are provided free of charge. Specialized services like document verification or legal assistance may have associated fees.'
    },
    {
      question: 'Can you help with property purchases in any Indian city?',
      answer: 'Yes, we provide assistance for property investments across all major Indian cities and have local expertise in various markets.'
    }
  ];

  return (
    <div className="min-h-screen bg-gray-50 pt-20">
      <div className="max-w-7xl mx-auto px-4 lg:px-8">
        {/* Header */}
        <div className="bg-white shadow-sm border border-gray-200 rounded-lg px-6 py-8 mb-8">
          <div className="text-center">
            <div className="flex justify-center mb-4">
              <div className="bg-blue-100 p-4 rounded-full">
                <Globe className="w-12 h-12 text-blue-600" />
              </div>
            </div>
            <h1 className="text-3xl font-bold text-gray-900 mb-4">NRI Assistance Desk</h1>
            <p className="text-lg text-gray-600 max-w-3xl mx-auto">
              Comprehensive support for Non-Resident Indians navigating property investments, 
              legal procedures, and documentation in India. Your trusted partner for seamless transactions.
            </p>
          </div>
        </div>

        {/* Navigation Tabs */}
        <div className="bg-white shadow-sm border border-gray-200 rounded-lg mb-8">
          <div className="flex flex-wrap border-b border-gray-200">
            {[
              { id: 'overview', label: 'Services Overview', icon: Info },
              { id: 'process', label: 'How It Works', icon: Workflow },
              { id: 'request', label: 'Submit Request', icon: MessageSquare },
              { id: 'faq', label: 'FAQ', icon: HelpCircle }
            ].map((tab) => {
              const Icon = tab.icon;
              return (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id)}
                  className={`flex items-center px-6 py-4 text-sm font-medium border-b-2 transition-colors ${
                    activeTab === tab.id
                      ? 'border-blue-500 text-blue-600 bg-blue-50'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  <Icon className="w-4 h-4 mr-2" />
                  {tab.label}
                </button>
              );
            })}
          </div>
        </div>

        {/* Tab Content */}
        {activeTab === 'overview' && (
          <div className="space-y-8">
            {/* Services Grid */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8">
              <h2 className="text-2xl font-bold text-gray-900 mb-6 text-center">Our Services</h2>
              <div className="grid md:grid-cols-2 gap-6">
                {services.map((service, index) => {
                  const Icon = service.icon;
                  return (
                    <div key={index} className="border border-gray-200 rounded-lg p-6 hover:shadow-md transition-shadow">
                      <div className="flex items-center mb-4">
                        <div className="bg-blue-100 p-3 rounded-lg mr-4">
                          <Icon className="w-6 h-6 text-blue-600" />
                        </div>
                        <h3 className="text-lg font-semibold text-gray-900">{service.title}</h3>
                      </div>
                      <p className="text-gray-600 mb-4">{service.description}</p>
                      <div className="space-y-2">
                        {service.features.map((feature, idx) => (
                          <div key={idx} className="flex items-center text-sm text-gray-700">
                            <CheckCircle className="w-4 h-4 text-green-500 mr-2" />
                            {feature}
                          </div>
                        ))}
                      </div>
                    </div>
                  );
                })}
              </div>
            </div>

            {/* Why Choose Us */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8">
              <h2 className="text-2xl font-bold text-gray-900 mb-6 text-center">Why Choose Our NRI Assistance?</h2>
              <div className="grid md:grid-cols-3 gap-6">
                <div className="text-center">
                  <div className="bg-green-100 p-4 rounded-full w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                    <Award className="w-8 h-8 text-green-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">Expert Guidance</h3>
                  <p className="text-gray-600">Experienced professionals with deep knowledge of Indian property laws and NRI regulations</p>
                </div>
                <div className="text-center">
                  <div className="bg-blue-100 p-4 rounded-full w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                    <Shield className="w-8 h-8 text-blue-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">Secure Process</h3>
                  <p className="text-gray-600">End-to-end secure handling of your documents and personal information</p>
                </div>
                <div className="text-center">
                  <div className="bg-purple-100 p-4 rounded-full w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                    <Clock className="w-8 h-8 text-purple-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">Quick Response</h3>
                  <p className="text-gray-600">Fast turnaround times with dedicated support throughout the process</p>
                </div>
              </div>
            </div>
          </div>
        )}

        {activeTab === 'process' && (
          <div className="space-y-8">
            {/* Process Steps */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8">
              <h2 className="text-2xl font-bold text-gray-900 mb-8 text-center">How Our Assistance Process Works</h2>
              <div className="space-y-8">
                {processSteps.map((step, index) => {
                  const Icon = step.icon;
                  return (
                    <div key={index} className="flex items-start space-x-6">
                      <div className="flex-shrink-0">
                        <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                          <span className="text-blue-600 font-bold">{step.step}</span>
                        </div>
                      </div>
                      <div className="flex-1">
                        <div className="flex items-center space-x-3 mb-2">
                          <Icon className="w-5 h-5 text-gray-600" />
                          <h3 className="text-lg font-semibold text-gray-900">{step.title}</h3>
                          <span className="bg-green-100 text-green-800 text-xs px-2 py-1 rounded-full">
                            {step.timeframe}
                          </span>
                        </div>
                        <p className="text-gray-600">{step.description}</p>
                        {index < processSteps.length - 1 && (
                          <div className="mt-4 ml-6">
                            <ArrowRight className="w-5 h-5 text-gray-400" />
                          </div>
                        )}
                      </div>
                    </div>
                  );
                })}
              </div>
            </div>

            {/* Contact Information */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8">
              <h2 className="text-2xl font-bold text-gray-900 mb-6 text-center">Get In Touch</h2>
              <div className="grid md:grid-cols-3 gap-6">
                <div className="text-center">
                  <div className="bg-blue-100 p-4 rounded-full w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                    <Phone className="w-8 h-8 text-blue-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">Phone Support</h3>
                  <p className="text-gray-600">+91-80-4567-8900</p>
                  <p className="text-sm text-gray-500">Mon-Fri, 9 AM - 6 PM IST</p>
                </div>
                <div className="text-center">
                  <div className="bg-green-100 p-4 rounded-full w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                    <Mail className="w-8 h-8 text-green-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">Email Support</h3>
                  <p className="text-gray-600">nri-support@newageland.com</p>
                  <p className="text-sm text-gray-500">Response within 24 hours</p>
                </div>
                <div className="text-center">
                  <div className="bg-purple-100 p-4 rounded-full w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                    <MessageSquare className="w-8 h-8 text-purple-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">Live Chat</h3>
                  <p className="text-gray-600">Available on website</p>
                  <p className="text-sm text-gray-500">Mon-Fri, 9 AM - 6 PM IST</p>
                </div>
              </div>
            </div>
          </div>
        )}

        {activeTab === 'request' && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8">
            <h2 className="text-2xl font-bold text-gray-900 mb-6 text-center">Submit Assistance Request</h2>
            <div className="max-w-2xl mx-auto">
              <form className="space-y-6" onSubmit={handleSubmit(onSubmit)}>
                <div className="grid md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Full Name *</label>
                    <input
                      type="text"
                      {...register('fullName', { required: 'Full name is required' })}
                      className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                        errors.fullName ? 'border-red-300' : 'border-gray-300'
                      }`}
                      placeholder="Enter your full name"
                    />
                    {errors.fullName && (
                      <p className="text-red-500 text-sm mt-1">{errors.fullName.message}</p>
                    )}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Email Address *</label>
                    <input
                      type="email"
                      {...register('email', { 
                        required: 'Email is required',
                        pattern: {
                          value: /^[A-Z0-9._%+-]+@[A-Z0-9.-]+\.[A-Z]{2,}$/i,
                          message: 'Invalid email address'
                        }
                      })}
                      className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                        errors.email ? 'border-red-300' : 'border-gray-300'
                      }`}
                      placeholder="Enter your email"
                    />
                    {errors.email && (
                      <p className="text-red-500 text-sm mt-1">{errors.email.message}</p>
                    )}
                  </div>
                </div>
                
                <div className="grid md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Phone Number *</label>
                    <input
                      type="tel"
                      {...register('phone', { required: 'Phone number is required' })}
                      className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                        errors.phone ? 'border-red-300' : 'border-gray-300'
                      }`}
                      placeholder="Enter your phone number"
                    />
                    {errors.phone && (
                      <p className="text-red-500 text-sm mt-1">{errors.phone.message}</p>
                    )}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Country of Residence *</label>
                    <select 
                      {...register('countryOfResidence', { required: 'Country is required' })}
                      className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                        errors.countryOfResidence ? 'border-red-300' : 'border-gray-300'
                      }`}
                    >
                      <option value="">Select your country</option>
                      <option value="USA">United States</option>
                      <option value="UK">United Kingdom</option>
                      <option value="Canada">Canada</option>
                      <option value="Australia">Australia</option>
                      <option value="UAE">United Arab Emirates</option>
                      <option value="Singapore">Singapore</option>
                      <option value="Other">Other</option>
                    </select>
                    {errors.countryOfResidence && (
                      <p className="text-red-500 text-sm mt-1">{errors.countryOfResidence.message}</p>
                    )}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Assistance Category *</label>
                  <select 
                    {...register('assistanceCategory', { required: 'Category is required' })}
                    className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.assistanceCategory ? 'border-red-300' : 'border-gray-300'
                    }`}
                  >
                    <option value="">Select category</option>
                    <option value="Property">Property Investment</option>
                    <option value="Legal">Legal & Documentation</option>
                    <option value="Banking">Banking & Financial</option>
                    <option value="Verification">Property Verification</option>
                    <option value="Other">Other</option>
                  </select>
                  {errors.assistanceCategory && (
                    <p className="text-red-500 text-sm mt-1">{errors.assistanceCategory.message}</p>
                  )}
                </div>

                {/* Property Details Section */}
                <div className="bg-gray-50 border border-gray-200 rounded-lg p-6">
                  <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                    <MapPin className="w-5 h-5 text-blue-600 mr-2" />
                    Property Details *
                  </h3>
                  <p className="text-sm text-gray-600 mb-4">Please provide the property details for assistance:</p>
                  
                  <div className="grid md:grid-cols-3 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Property ID *</label>
                      <input
                        type="text"
                        {...register('propertyId', { required: 'Property ID is required' })}
                        className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                          errors.propertyId || propertyValidationError ? 'border-red-300' : 'border-gray-300'
                        }`}
                        placeholder="e.g., PROP123456"
                      />
                      {errors.propertyId && (
                        <p className="text-red-500 text-sm mt-1">{errors.propertyId.message}</p>
                      )}
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Area/Locality *</label>
                      <input
                        type="text"
                        {...register('propertyArea', { required: 'Property area is required' })}
                        className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                          errors.propertyArea || propertyValidationError ? 'border-red-300' : 'border-gray-300'
                        }`}
                        placeholder="e.g., Bandra West, Koramangala"
                      />
                      {errors.propertyArea && (
                        <p className="text-red-500 text-sm mt-1">{errors.propertyArea.message}</p>
                      )}
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">City *</label>
                      <input
                        type="text"
                        {...register('propertyCity', { required: 'Property city is required' })}
                        className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                          errors.propertyCity || propertyValidationError ? 'border-red-300' : 'border-gray-300'
                        }`}
                        placeholder="e.g., Mumbai, Bangalore"
                      />
                      {errors.propertyCity && (
                        <p className="text-red-500 text-sm mt-1">{errors.propertyCity.message}</p>
                      )}
                    </div>
                  </div>
                  {propertyValidationError && (
                    <div className="mt-3 p-3 bg-red-50 border border-red-200 rounded-lg">
                      <p className="text-red-600 text-sm flex items-center">
                        <AlertCircle className="w-4 h-4 mr-2" />
                        {propertyValidationError}
                      </p>
                    </div>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Priority Level</label>
                  <select 
                    {...register('priorityLevel')}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    <option value="Low">Low - General inquiry</option>
                    <option value="Medium">Medium - Need assistance soon</option>
                    <option value="High">High - Urgent assistance required</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Detailed Description *</label>
                  <textarea
                    rows={5}
                    {...register('description', { required: 'Description is required' })}
                    className={`w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                      errors.description ? 'border-red-300' : 'border-gray-300'
                    }`}
                    placeholder="Please provide detailed information about your assistance requirement..."
                  ></textarea>
                  {errors.description && (
                    <p className="text-red-500 text-sm mt-1">{errors.description.message}</p>
                  )}
                </div>

                <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                  <div className="flex items-start">
                    <Info className="w-5 h-5 text-blue-600 mr-3 mt-0.5" />
                    <div>
                      <h4 className="text-sm font-medium text-blue-900 mb-1">What happens next?</h4>
                      <ul className="text-sm text-blue-800 space-y-1">
                        <li>• You'll receive a confirmation email with your request ID</li>
                        <li>• Our team will review and assign an officer within 24 hours</li>
                        <li>• You'll be contacted for initial consultation within 2-3 days</li>
                        <li>• Regular updates will be provided throughout the process</li>
                      </ul>
                    </div>
                  </div>
                </div>

                <button
                  type="submit"
                  disabled={isSubmitting}
                  className="w-full bg-[#0056D2] text-white py-3 px-6 rounded-lg hover:bg-blue-700 transition-colors font-medium disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {isSubmitting ? 'Submitting...' : 'Submit Assistance Request'}
                </button>
              </form>
            </div>
          </div>
        )}

        {activeTab === 'faq' && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8">
            <h2 className="text-2xl font-bold text-gray-900 mb-8 text-center">Frequently Asked Questions</h2>
            <div className="max-w-3xl mx-auto space-y-6">
              {faqs.map((faq, index) => (
                <div key={index} className="border border-gray-200 rounded-lg p-6">
                  <h3 className="text-lg font-semibold text-gray-900 mb-3 flex items-center">
                    <HelpCircle className="w-5 h-5 text-blue-600 mr-2" />
                    {faq.question}
                  </h3>
                  <p className="text-gray-600 leading-relaxed">{faq.answer}</p>
                </div>
              ))}
            </div>
            
            <div className="mt-8 text-center">
              <div className="bg-gray-50 border border-gray-200 rounded-lg p-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-2">Still have questions?</h3>
                <p className="text-gray-600 mb-4">Our support team is here to help you with any additional queries.</p>
                <button 
                  onClick={() => window.location.href = '/contact'}
                  className="bg-[#0056D2] text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors"
                >
                  Contact Support
                </button>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}