import React, { useState, useCallback } from 'react';
import { Upload, X, FileText, AlertTriangle, CheckCircle } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { riskComplianceService } from '../../../api/riskCompliance';

interface DocumentUploadProps {
  submissionId: string;
  onDocumentUploaded: (documentType: string, documentId: string) => void;
  onNext: () => void;
}

interface UploadedDoc {
  id: string;
  name: string;
  type: string;
  size: number;
  status: 'uploading' | 'uploaded' | 'error';
  documentType: string;
}

const REQUIRED_DOCUMENTS = [
  { type: 'sale_deed', label: 'Sale Deed', required: true },
  { type: 'khata_rtc', label: 'Khata/RTC', required: true },
  { type: 'tax_receipt', label: 'Tax Receipt', required: true },
  { type: 'encumbrance_certificate', label: 'Encumbrance Certificate', required: true },
  { type: 'rera_certificate', label: 'RERA Certificate', required: true }
];

export function DocumentUpload({ submissionId, onDocumentUploaded, onNext }: DocumentUploadProps) {
  const [uploadedDocs, setUploadedDocs] = useState<UploadedDoc[]>([]);
  const [dragActive, setDragActive] = useState(false);
  const [selectedDocType, setSelectedDocType] = useState<string>('');

  const handleDrag = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (e.type === 'dragenter' || e.type === 'dragover') {
      setDragActive(true);
    } else if (e.type === 'dragleave') {
      setDragActive(false);
    }
  }, []);

  const handleDrop = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setDragActive(false);
    
    if (e.dataTransfer.files && e.dataTransfer.files[0]) {
      handleFiles(e.dataTransfer.files);
    }
  }, []);

  const handleFiles = async (files: FileList) => {
    if (!selectedDocType) {
      alert('Please select a document type first');
      return;
    }

    const file = files[0];
    if (!file) return;

    // Validate file
    const maxSize = 10 * 1024 * 1024; // 10MB
    const allowedTypes = ['application/pdf', 'image/jpeg', 'image/jpg', 'image/png'];
    
    if (file.size > maxSize) {
      alert('File size must be less than 10MB');
      return;
    }
    
    if (!allowedTypes.includes(file.type)) {
      alert('Only PDF, JPG, and PNG files are allowed');
      return;
    }

    // Check if document type already uploaded
    if (uploadedDocs.some(doc => doc.documentType === selectedDocType)) {
      alert('This document type has already been uploaded');
      return;
    }

    const tempId = Date.now().toString();
    const newDoc: UploadedDoc = {
      id: tempId,
      name: file.name,
      type: file.type,
      size: file.size,
      status: 'uploading',
      documentType: selectedDocType
    };

    setUploadedDocs(prev => [...prev, newDoc]);

    try {
      const result = await riskComplianceService.uploadDocument(submissionId, selectedDocType, file);
      
      setUploadedDocs(prev => 
        prev.map(doc => 
          doc.id === tempId 
            ? { ...doc, id: result.document_id, status: 'uploaded' as const }
            : doc
        )
      );

      onDocumentUploaded(selectedDocType, result.document_id);
      setSelectedDocType(''); // Reset selection
    } catch (error) {
      console.error('Upload failed:', error);
      setUploadedDocs(prev => 
        prev.map(doc => 
          doc.id === tempId 
            ? { ...doc, status: 'error' as const }
            : doc
        )
      );
    }
  };

  const removeDoc = (id: string) => {
    setUploadedDocs(prev => prev.filter(doc => doc.id !== id));
  };

  const getUploadedDocTypes = () => {
    return uploadedDocs.filter(doc => doc.status === 'uploaded').map(doc => doc.documentType);
  };

  const getRequiredDocsStatus = () => {
    const uploadedTypes = getUploadedDocTypes();
    return REQUIRED_DOCUMENTS.map(doc => ({
      ...doc,
      uploaded: uploadedTypes.includes(doc.type)
    }));
  };

  const canProceed = () => {
    const uploadedTypes = getUploadedDocTypes();
    return REQUIRED_DOCUMENTS.every(doc => uploadedTypes.includes(doc.type));
  };

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h2 className="text-2xl font-bold text-gray-900 mb-4">Document Upload</h2>
        <p className="text-gray-600">
          Upload all required documents for risk assessment. All documents must be in PDF, JPG, or PNG format.
        </p>
      </div>

      {/* Document Type Selection */}
      <div className="mb-6">
        <label className="block text-sm font-medium text-gray-700 mb-2">
          Select Document Type
        </label>
        <select
          value={selectedDocType}
          onChange={(e) => setSelectedDocType(e.target.value)}
          className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
        >
          <option value="">Choose document type...</option>
          {REQUIRED_DOCUMENTS.map(doc => (
            <option 
              key={doc.type} 
              value={doc.type}
              disabled={uploadedDocs.some(uploaded => uploaded.documentType === doc.type && uploaded.status === 'uploaded')}
            >
              {doc.label} {doc.required && '*'}
            </option>
          ))}
        </select>
      </div>

      {/* Upload Area */}
      <div
        className={`border-2 border-dashed rounded-lg p-8 text-center transition-colors ${
          dragActive 
            ? 'border-[#0056D2] bg-blue-50' 
            : 'border-gray-300 hover:border-[#0056D2]'
        } ${!selectedDocType ? 'opacity-50 cursor-not-allowed' : 'cursor-pointer'}`}
        onDragEnter={handleDrag}
        onDragLeave={handleDrag}
        onDragOver={handleDrag}
        onDrop={handleDrop}
        onClick={() => {
          if (selectedDocType) {
            document.getElementById('file-upload')?.click();
          }
        }}
      >
        <Upload className="w-12 h-12 text-gray-400 mx-auto mb-4" />
        <p className="text-lg font-medium text-gray-700 mb-2">
          {selectedDocType ? 'Drop files here or click to browse' : 'Select document type first'}
        </p>
        <p className="text-sm text-gray-500 mb-4">
          Supported formats: PDF, JPG, PNG (Max 10MB)
        </p>
        <input
          type="file"
          id="file-upload"
          className="hidden"
          accept=".pdf,.jpg,.jpeg,.png"
          onChange={(e) => e.target.files && handleFiles(e.target.files)}
          disabled={!selectedDocType}
        />
        {selectedDocType && (
          <Button className="bg-[#0056D2] hover:bg-[#003d9a]">
            Choose File
          </Button>
        )}
      </div>

      {/* Uploaded Documents */}
      {uploadedDocs.length > 0 && (
        <div className="mt-8">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Uploaded Documents</h3>
          <div className="space-y-3">
            {uploadedDocs.map((doc) => (
              <div key={doc.id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                <div className="flex items-center">
                  <FileText className="w-8 h-8 text-gray-500 mr-3" />
                  <div>
                    <p className="font-medium text-gray-900">{doc.name}</p>
                    <p className="text-sm text-gray-500">
                      {REQUIRED_DOCUMENTS.find(d => d.type === doc.documentType)?.label} • {(doc.size / 1024 / 1024).toFixed(2)} MB
                    </p>
                  </div>
                </div>
                <div className="flex items-center space-x-2">
                  {doc.status === 'uploading' && (
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-[#0056D2]"></div>
                  )}
                  {doc.status === 'uploaded' && (
                    <CheckCircle className="w-5 h-5 text-green-500" />
                  )}
                  {doc.status === 'error' && (
                    <AlertTriangle className="w-5 h-5 text-red-500" />
                  )}
                  <button
                    onClick={() => removeDoc(doc.id)}
                    className="text-gray-400 hover:text-red-500"
                  >
                    <X className="w-5 h-5" />
                  </button>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Required Documents Checklist */}
      <div className="mt-8">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Required Documents Checklist</h3>
        <div className="space-y-2">
          {getRequiredDocsStatus().map((doc) => (
            <div key={doc.type} className="flex items-center justify-between p-3 bg-white border rounded-lg">
              <div className="flex items-center">
                {doc.uploaded ? (
                  <CheckCircle className="w-5 h-5 text-green-500 mr-3" />
                ) : (
                  <div className="w-5 h-5 border-2 border-gray-300 rounded-full mr-3"></div>
                )}
                <span className={`font-medium ${doc.uploaded ? 'text-green-700' : 'text-gray-700'}`}>
                  {doc.label}
                </span>
              </div>
              <span className={`text-sm px-2 py-1 rounded ${
                doc.uploaded 
                  ? 'bg-green-100 text-green-800' 
                  : 'bg-gray-100 text-gray-600'
              }`}>
                {doc.uploaded ? 'Uploaded' : 'Required'}
              </span>
            </div>
          ))}
        </div>
      </div>

      {/* Navigation */}
      <div className="flex justify-end mt-8">
        <Button
          onClick={onNext}
          disabled={!canProceed()}
          className="bg-[#0056D2] hover:bg-[#003d9a] disabled:opacity-50 disabled:cursor-not-allowed"
        >
          Continue to RERA Check
        </Button>
      </div>

      {!canProceed() && (
        <div className="mt-4 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
          <div className="flex items-center">
            <AlertTriangle className="w-5 h-5 text-yellow-600 mr-2" />
            <p className="text-sm text-yellow-800">
              Please upload all required documents to continue.
            </p>
          </div>
        </div>
      )}
    </div>
  );
}