import React, { useState } from 'react';
import { Scale, CheckCircle, AlertTriangle, X, Search, CreditCard } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { riskComplianceService, EncumbranceCheckResponse } from '../../../api/riskCompliance';

interface EncumbranceCheckProps {
  submissionId: string;
  onCheckCompleted: (result: EncumbranceCheckResponse) => void;
  onNext: () => void;
  onBack: () => void;
}

export function EncumbranceCheck({ submissionId, onCheckCompleted, onNext, onBack }: EncumbranceCheckProps) {
  const [isChecking, setIsChecking] = useState(false);
  const [checkResult, setCheckResult] = useState<EncumbranceCheckResponse | null>(null);
  const [error, setError] = useState<string>('');

  const handleEncumbranceCheck = async () => {
    setIsChecking(true);
    setError('');

    try {
      const result = await riskComplianceService.performEncumbranceCheck({
        submission_id: submissionId,
        property_details: {}
      });

      // Simulate getting the check result with encumbrances found
      const mockResult: EncumbranceCheckResponse = {
        id: result.check_id,
        submission_id: submissionId,
        status: 'found', // Mock as found for demo to show risk
        details: {
          encumbrances_found: true,
          bank_loan_pending: '₹2.3 Crores',
          loan_bank: 'HDFC Bank Ltd.',
          loan_account_number: 'HDFC****5678',
          loan_date: '2022-03-15',
          clearance_required: true,
          risk_level: 'High',
          mortgage_details: 'Property mortgaged as primary security',
          outstanding_amount: '₹2.3 Crores',
          emi_status: 'Regular',
          foreclosure_charges: '2% + GST',
          noc_required: true,
          legal_notices: 'None',
          court_cases: 'None',
          revenue_records: 'Updated',
          mutation_status: 'Pending bank NOC'
        },
        checked_at: new Date().toISOString()
      };

      setCheckResult(mockResult);
      onCheckCompleted(mockResult);
    } catch (error) {
      console.error('Encumbrance check failed:', error);
      setError('Failed to perform encumbrance check. Please try again.');
    } finally {
      setIsChecking(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'clear': return 'text-green-600 bg-green-50 border-green-200';
      case 'pending': return 'text-yellow-600 bg-yellow-50 border-yellow-200';
      case 'found': return 'text-red-600 bg-red-50 border-red-200';
      default: return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'clear': return <CheckCircle className="w-5 h-5 text-green-600" />;
      case 'pending': return <AlertTriangle className="w-5 h-5 text-yellow-600" />;
      case 'found': return <X className="w-5 h-5 text-red-600" />;
      default: return null;
    }
  };

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center">
          <Scale className="w-6 h-6 text-[#0056D2] mr-2" />
          Encumbrance Verification
        </h2>
        <p className="text-gray-600">
          Check for any financial liabilities, loans, or legal encumbrances on the property.
        </p>
      </div>

      {!checkResult ? (
        <div className="bg-white border border-gray-200 rounded-lg p-6">
          <div className="space-y-6">
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
              <h4 className="font-medium text-blue-900 mb-2">Encumbrance Check includes:</h4>
              <ul className="text-sm text-blue-800 space-y-1">
                <li>• Outstanding bank loans and mortgages</li>
                <li>• Legal disputes and court cases</li>
                <li>• Revenue department liabilities</li>
                <li>• Pending NOCs and clearances</li>
                <li>• Property tax arrears</li>
                <li>• Mutation and title transfer issues</li>
              </ul>
            </div>

            <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
              <div className="flex items-start">
                <AlertTriangle className="w-5 h-5 text-yellow-600 mr-2 mt-0.5" />
                <div>
                  <h4 className="font-medium text-yellow-900 mb-1">Important Note</h4>
                  <p className="text-sm text-yellow-800">
                    This check will search for encumbrances across multiple databases including banks, 
                    revenue records, and legal repositories. The process may take a few moments.
                  </p>
                </div>
              </div>
            </div>

            {error && (
              <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
                <div className="flex items-center">
                  <X className="w-5 h-5 text-red-600 mr-2" />
                  <p className="text-sm text-red-800">{error}</p>
                </div>
              </div>
            )}

            <Button
              onClick={handleEncumbranceCheck}
              disabled={isChecking}
              className="w-full bg-[#0056D2] hover:bg-[#003d9a] disabled:opacity-50"
            >
              {isChecking ? (
                <div className="flex items-center">
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  Checking Encumbrances...
                </div>
              ) : (
                <div className="flex items-center">
                  <Search className="w-4 h-4 mr-2" />
                  Check for Encumbrances
                </div>
              )}
            </Button>
          </div>
        </div>
      ) : (
        <div className="space-y-6">
          {/* Status Card */}
          <div className={`p-6 border rounded-lg ${getStatusColor(checkResult.status)}`}>
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center">
                {getStatusIcon(checkResult.status)}
                <h3 className="text-lg font-semibold ml-2">
                  Encumbrance Status: {checkResult.status.toUpperCase()}
                </h3>
              </div>
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  setCheckResult(null);
                }}
              >
                Check Again
              </Button>
            </div>

            {checkResult.status === 'clear' && (
              <p className="text-sm">
                ✅ No encumbrances found. Property has clear title with no financial liabilities.
              </p>
            )}
            {checkResult.status === 'pending' && (
              <p className="text-sm">
                ⚠️ Some encumbrances are pending clearance. Verify before proceeding.
              </p>
            )}
            {checkResult.status === 'found' && (
              <p className="text-sm">
                ❌ Active encumbrances found. High financial risk - loan clearance required.
              </p>
            )}
          </div>

          {/* Encumbrance Details */}
          {checkResult.status === 'found' && (
            <div className="bg-red-50 border border-red-200 rounded-lg p-6">
              <div className="flex items-start mb-4">
                <CreditCard className="w-6 h-6 text-red-600 mr-3 mt-1" />
                <div>
                  <h4 className="text-lg font-semibold text-red-900 mb-2">⚠️ CRITICAL: Bank Loan Found</h4>
                  <p className="text-red-800 font-medium">
                    Outstanding loan of {checkResult.details.bank_loan_pending} with {checkResult.details.loan_bank}
                  </p>
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                <div className="p-3 bg-white rounded border border-red-200">
                  <label className="text-sm font-medium text-red-700">Outstanding Amount</label>
                  <p className="text-sm text-red-900 font-semibold">{checkResult.details.outstanding_amount}</p>
                </div>
                
                <div className="p-3 bg-white rounded border border-red-200">
                  <label className="text-sm font-medium text-red-700">Loan Bank</label>
                  <p className="text-sm text-red-900">{checkResult.details.loan_bank}</p>
                </div>

                <div className="p-3 bg-white rounded border border-red-200">
                  <label className="text-sm font-medium text-red-700">Loan Date</label>
                  <p className="text-sm text-red-900">{checkResult.details.loan_date}</p>
                </div>

                <div className="p-3 bg-white rounded border border-red-200">
                  <label className="text-sm font-medium text-red-700">EMI Status</label>
                  <p className="text-sm text-red-900">{checkResult.details.emi_status}</p>
                </div>

                <div className="p-3 bg-white rounded border border-red-200">
                  <label className="text-sm font-medium text-red-700">Foreclosure Charges</label>
                  <p className="text-sm text-red-900">{checkResult.details.foreclosure_charges}</p>
                </div>

                <div className="p-3 bg-white rounded border border-red-200">
                  <label className="text-sm font-medium text-red-700">NOC Required</label>
                  <p className="text-sm text-red-900">{checkResult.details.noc_required ? 'Yes' : 'No'}</p>
                </div>
              </div>
            </div>
          )}

          {/* Detailed Verification Results */}
          <div className="bg-white border border-gray-200 rounded-lg p-6">
            <h4 className="text-lg font-semibold text-gray-900 mb-4">Verification Results</h4>
            
            <div className="space-y-3">
              <div className={`flex items-center justify-between p-3 rounded border ${
                checkResult.details.legal_notices === 'None' 
                  ? 'bg-green-50 border-green-200' 
                  : 'bg-red-50 border-red-200'
              }`}>
                <div className="flex items-center">
                  {checkResult.details.legal_notices === 'None' ? (
                    <CheckCircle className="w-5 h-5 text-green-600 mr-3" />
                  ) : (
                    <X className="w-5 h-5 text-red-600 mr-3" />
                  )}
                  <span className="text-sm font-medium">Legal Notices</span>
                </div>
                <span className={`text-xs px-2 py-1 rounded ${
                  checkResult.details.legal_notices === 'None'
                    ? 'bg-green-100 text-green-800'
                    : 'bg-red-100 text-red-800'
                }`}>
                  {checkResult.details.legal_notices}
                </span>
              </div>

              <div className={`flex items-center justify-between p-3 rounded border ${
                checkResult.details.court_cases === 'None' 
                  ? 'bg-green-50 border-green-200' 
                  : 'bg-red-50 border-red-200'
              }`}>
                <div className="flex items-center">
                  {checkResult.details.court_cases === 'None' ? (
                    <CheckCircle className="w-5 h-5 text-green-600 mr-3" />
                  ) : (
                    <X className="w-5 h-5 text-red-600 mr-3" />
                  )}
                  <span className="text-sm font-medium">Court Cases</span>
                </div>
                <span className={`text-xs px-2 py-1 rounded ${
                  checkResult.details.court_cases === 'None'
                    ? 'bg-green-100 text-green-800'
                    : 'bg-red-100 text-red-800'
                }`}>
                  {checkResult.details.court_cases}
                </span>
              </div>

              <div className="flex items-center justify-between p-3 bg-green-50 border border-green-200 rounded">
                <div className="flex items-center">
                  <CheckCircle className="w-5 h-5 text-green-600 mr-3" />
                  <span className="text-sm font-medium">Revenue Records</span>
                </div>
                <span className="text-xs bg-green-100 text-green-800 px-2 py-1 rounded">
                  {checkResult.details.revenue_records}
                </span>
              </div>

              <div className={`flex items-center justify-between p-3 rounded border ${
                checkResult.details.mutation_status?.includes('Pending') 
                  ? 'bg-yellow-50 border-yellow-200' 
                  : 'bg-green-50 border-green-200'
              }`}>
                <div className="flex items-center">
                  {checkResult.details.mutation_status?.includes('Pending') ? (
                    <AlertTriangle className="w-5 h-5 text-yellow-600 mr-3" />
                  ) : (
                    <CheckCircle className="w-5 h-5 text-green-600 mr-3" />
                  )}
                  <span className="text-sm font-medium">Mutation Status</span>
                </div>
                <span className={`text-xs px-2 py-1 rounded ${
                  checkResult.details.mutation_status?.includes('Pending')
                    ? 'bg-yellow-100 text-yellow-800'
                    : 'bg-green-100 text-green-800'
                }`}>
                  {checkResult.details.mutation_status}
                </span>
              </div>
            </div>
          </div>

          {/* Critical Recommendations */}
          <div className="bg-red-50 border border-red-200 rounded-lg p-6">
            <h4 className="text-lg font-semibold text-red-900 mb-3">⚠️ CRITICAL RECOMMENDATIONS</h4>
            <ul className="space-y-2 text-sm text-red-800">
              {checkResult.status === 'found' && (
                <>
                  <li>• <strong>DO NOT PROCEED</strong> without loan clearance certificate</li>
                  <li>• Obtain NOC from {checkResult.details.loan_bank} before purchase</li>
                  <li>• Verify exact outstanding amount and foreclosure charges</li>
                  <li>• Ensure seller clears all dues before registration</li>
                  <li>• Consider escrow arrangement for loan clearance</li>
                  <li>• Get legal opinion on liability transfer risks</li>
                </>
              )}
              {checkResult.status === 'pending' && (
                <>
                  <li>• Verify pending clearances with respective authorities</li>
                  <li>• Obtain timeline for clearance completion</li>
                  <li>• Consider conditional purchase agreement</li>
                </>
              )}
              {checkResult.status === 'clear' && (
                <>
                  <li>• Property is clear of encumbrances - safe to proceed</li>
                  <li>• Verify clearance certificates during final check</li>
                  <li>• Ensure no new encumbrances before registration</li>
                </>
              )}
            </ul>
          </div>
        </div>
      )}

      {/* Navigation */}
      <div className="flex justify-between mt-8">
        <Button variant="outline" onClick={onBack}>
          Back to Ownership Check
        </Button>
        
        <div className="flex space-x-4">
          <Button
            variant="outline"
            onClick={onNext}
            className="text-gray-600 border-gray-300"
          >
            Skip for Testing
          </Button>
          
          {checkResult && (
            <Button
              onClick={onNext}
              className="bg-[#0056D2] hover:bg-[#003d9a]"
            >
              Continue to Risk Score
            </Button>
          )}
        </div>
      </div>
    </div>
  );
}