import React, { useState } from 'react';
import { Scale, AlertTriangle, CheckCircle, Clock } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { riskComplianceService } from '../../../api/riskCompliance';

interface LitigationAnalysisProps {
  submissionId: string;
  onNext: () => void;
  onBack: () => void;
}

interface LitigationResult {
  success: boolean;
  litigation_score: number;
  total_cases: number;
  active_cases: number;
  resolved_cases: number;
  risk_level: string;
  cases: Array<{
    case_type: string;
    status: string;
    severity: string;
    description: string;
    court: string;
    year: string;
  }>;
  recommendations: string[];
}

export function LitigationAnalysis({ submissionId, onNext, onBack }: LitigationAnalysisProps) {
  const [analysis, setAnalysis] = useState<LitigationResult | null>(null);
  const [analyzing, setAnalyzing] = useState(false);
  const [error, setError] = useState('');

  const handleAnalyze = async () => {
    setAnalyzing(true);
    setError('');
    
    try {
      const result = await riskComplianceService.analyzeLitigation(submissionId);
      
      if (result.success) {
        setAnalysis(result.data);
      } else {
        setError('Analysis failed: ' + (result.data?.error || 'Unknown error'));
      }
    } catch (err: any) {
      console.error('Litigation analysis failed:', err);
      setError('Failed to analyze litigation: ' + (err.message || 'Network error'));
    } finally {
      setAnalyzing(false);
    }
  };

  const getRiskColor = (riskLevel: string) => {
    switch (riskLevel) {
      case 'low': return 'text-green-600 bg-green-100';
      case 'medium': return 'text-orange-600 bg-orange-100';
      case 'high': return 'text-red-600 bg-red-100';
      default: return 'text-gray-600 bg-gray-100';
    }
  };

  const getScoreColor = (score: number) => {
    if (score >= 80) return 'text-green-600';
    if (score >= 60) return 'text-orange-600';
    return 'text-red-600';
  };

  const getStatusColor = (status: string) => {
    switch (status.toLowerCase()) {
      case 'resolved': return 'text-green-600 bg-green-100';
      case 'active': return 'text-red-600 bg-red-100';
      case 'pending': return 'text-orange-600 bg-orange-100';
      default: return 'text-gray-600 bg-gray-100';
    }
  };

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center">
          <Scale className="w-6 h-6 text-[#0056D2] mr-2" />
          Litigation Analysis
        </h2>
        <p className="text-gray-600">
          Comprehensive analysis of legal disputes and litigation history associated with the property.
        </p>
      </div>

      {!analysis && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 mb-6">
          <h3 className="text-lg font-semibold text-blue-900 mb-3">
            Ready for Litigation Analysis
          </h3>
          <p className="text-blue-700 mb-4">
            Search for any ongoing or past legal disputes, court cases, and litigation risks associated with this property.
          </p>
          <Button
            onClick={handleAnalyze}
            disabled={analyzing}
            className="bg-blue-600 hover:bg-blue-700"
          >
            {analyzing ? (
              <div className="flex items-center">
                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                Analyzing Litigation...
              </div>
            ) : (
              'Start Litigation Analysis'
            )}
          </Button>
        </div>
      )}

      {error && (
        <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
          <div className="flex items-center">
            <AlertTriangle className="w-5 h-5 text-red-600 mr-2" />
            <p className="text-red-800">{error}</p>
          </div>
        </div>
      )}

      {analysis && (
        <div className="space-y-6">
          {/* Analysis Results */}
          <div className="bg-white border border-gray-200 rounded-lg p-6">
            <h3 className="text-lg font-semibold mb-4">Litigation Assessment</h3>
            
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
              <div className="text-center p-4 bg-gray-50 rounded-lg">
                <div className={`text-2xl font-bold ${getScoreColor(analysis.litigation_score)}`}>
                  {analysis.litigation_score}%
                </div>
                <div className="text-sm text-gray-600">Litigation Score</div>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg">
                <div className="text-2xl font-bold text-blue-600">{analysis.total_cases}</div>
                <div className="text-sm text-gray-600">Total Cases</div>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg">
                <div className="text-2xl font-bold text-red-600">{analysis.active_cases}</div>
                <div className="text-sm text-gray-600">Active Cases</div>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg">
                <div className={`text-2xl font-bold px-3 py-1 rounded ${getRiskColor(analysis.risk_level)}`}>
                  {analysis.risk_level.toUpperCase()}
                </div>
                <div className="text-sm text-gray-600">Risk Level</div>
              </div>
            </div>

            {/* Cases Found */}
            {analysis.total_cases > 0 && (
              <div className="mb-6">
                <h4 className="font-semibold text-gray-900 mb-3">
                  Legal Cases ({analysis.total_cases})
                </h4>
                <div className="space-y-3">
                  {analysis.cases.map((case_, index) => (
                    <div key={index} className="border border-gray-200 rounded-lg p-4">
                      <div className="flex items-start justify-between mb-2">
                        <div className="flex-1">
                          <div className="font-medium text-gray-900">{case_.case_type}</div>
                          <div className="text-gray-600 text-sm">{case_.description}</div>
                          <div className="text-gray-500 text-xs mt-1">
                            {case_.court} • {case_.year}
                          </div>
                        </div>
                        <div className="flex flex-col items-end space-y-1">
                          <span className={`px-2 py-1 rounded text-xs ${getStatusColor(case_.status)}`}>
                            {case_.status.toUpperCase()}
                          </span>
                          <span className={`px-2 py-1 rounded text-xs ${
                            case_.severity === 'high' ? 'bg-red-100 text-red-800' :
                            case_.severity === 'medium' ? 'bg-orange-100 text-orange-800' :
                            'bg-yellow-100 text-yellow-800'
                          }`}>
                            {case_.severity.toUpperCase()}
                          </span>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* No Cases Found */}
            {analysis.total_cases === 0 && (
              <div className="bg-green-50 border border-green-200 rounded-lg p-4 mb-6">
                <div className="flex items-center">
                  <CheckCircle className="w-5 h-5 text-green-600 mr-2" />
                  <div>
                    <div className="font-medium text-green-800">No Litigation Found</div>
                    <div className="text-green-700 text-sm">No legal disputes or court cases found associated with this property.</div>
                  </div>
                </div>
              </div>
            )}

            {/* Recommendations */}
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
              <h4 className="font-semibold text-blue-800 mb-2">Recommendations</h4>
              <ul className="space-y-1">
                {analysis.recommendations.map((rec, index) => (
                  <li key={index} className="text-blue-700 text-sm flex items-start">
                    <CheckCircle className="w-4 h-4 text-blue-600 mr-2 mt-0.5 flex-shrink-0" />
                    {rec}
                  </li>
                ))}
              </ul>
            </div>
          </div>
        </div>
      )}

      {/* Navigation */}
      <div className="flex justify-between mt-8">
        <Button variant="outline" onClick={onBack}>
          Back to Title Chain
        </Button>
        <Button
          onClick={onNext}
          disabled={!analysis}
          className="bg-[#0056D2] hover:bg-[#003d9a] disabled:opacity-50"
        >
          Continue to Encumbrance Check
        </Button>
      </div>
    </div>
  );
}