import React, { useState } from 'react';
import { Users, CheckCircle, AlertTriangle, X, Search } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { Input } from '../../../components/ui/input';
import { riskComplianceService, OwnershipCheckResponse } from '../../../api/riskCompliance';

interface OwnershipCheckProps {
  submissionId: string;
  onCheckCompleted: (result: OwnershipCheckResponse) => void;
  onNext: () => void;
  onBack: () => void;
}

export function OwnershipCheck({ submissionId, onCheckCompleted, onNext, onBack }: OwnershipCheckProps) {
  const [ownerName, setOwnerName] = useState('');
  const [isChecking, setIsChecking] = useState(false);
  const [checkResult, setCheckResult] = useState<OwnershipCheckResponse | null>(null);
  const [error, setError] = useState<string>('');

  const handleOwnershipCheck = async () => {
    if (!ownerName.trim()) {
      setError('Please enter the owner name');
      return;
    }

    setIsChecking(true);
    setError('');

    try {
      const result = await riskComplianceService.performOwnershipCheck({
        submission_id: submissionId,
        owner_name: ownerName,
        document_details: {}
      });

      // Simulate getting the check result
      const mockResult: OwnershipCheckResponse = {
        id: result.check_id,
        submission_id: submissionId,
        owner_name: ownerName,
        status: 'verified', // Mock as verified for demo
        details: {
          owner_name: ownerName,
          document_consistency: 'Verified',
          title_clear: true,
          verification_date: new Date().toISOString(),
          ownership_type: 'Individual',
          title_deed_number: 'TD/2023/12345',
          survey_number: 'SY/45/2A',
          registration_date: '2023-01-15',
          previous_owners: 1,
          encumbrance_period: '30 years',
          legal_issues: 'None found'
        },
        checked_at: new Date().toISOString()
      };

      setCheckResult(mockResult);
      onCheckCompleted(mockResult);
    } catch (error) {
      console.error('Ownership check failed:', error);
      setError('Failed to perform ownership check. Please try again.');
    } finally {
      setIsChecking(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'verified': return 'text-green-600 bg-green-50 border-green-200';
      case 'incomplete': return 'text-yellow-600 bg-yellow-50 border-yellow-200';
      case 'mismatch': return 'text-red-600 bg-red-50 border-red-200';
      default: return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'verified': return <CheckCircle className="w-5 h-5 text-green-600" />;
      case 'incomplete': return <AlertTriangle className="w-5 h-5 text-yellow-600" />;
      case 'mismatch': return <X className="w-5 h-5 text-red-600" />;
      default: return null;
    }
  };

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center">
          <Users className="w-6 h-6 text-[#0056D2] mr-2" />
          Ownership Verification
        </h2>
        <p className="text-gray-600">
          Verify the ownership details and document consistency to ensure clear title.
        </p>
      </div>

      {!checkResult ? (
        <div className="bg-white border border-gray-200 rounded-lg p-6">
          <div className="space-y-6">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Property Owner Name *
              </label>
              <Input
                type="text"
                placeholder="Enter the name as per sale deed/title documents"
                value={ownerName}
                onChange={(e) => setOwnerName(e.target.value)}
                className="w-full"
              />
              <p className="text-xs text-gray-500 mt-1">
                Enter the exact name as mentioned in the property documents
              </p>
            </div>

            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
              <h4 className="font-medium text-blue-900 mb-2">What we verify:</h4>
              <ul className="text-sm text-blue-800 space-y-1">
                <li>• Document consistency across all ownership papers</li>
                <li>• Title deed authenticity and registration</li>
                <li>• Chain of ownership and previous transfers</li>
                <li>• Legal disputes or pending cases</li>
                <li>• Power of attorney validations</li>
              </ul>
            </div>

            {error && (
              <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
                <div className="flex items-center">
                  <X className="w-5 h-5 text-red-600 mr-2" />
                  <p className="text-sm text-red-800">{error}</p>
                </div>
              </div>
            )}

            <Button
              onClick={handleOwnershipCheck}
              disabled={isChecking || !ownerName.trim()}
              className="w-full bg-[#0056D2] hover:bg-[#003d9a] disabled:opacity-50"
            >
              {isChecking ? (
                <div className="flex items-center">
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  Verifying Ownership...
                </div>
              ) : (
                <div className="flex items-center">
                  <Search className="w-4 h-4 mr-2" />
                  Verify Ownership
                </div>
              )}
            </Button>
          </div>
        </div>
      ) : (
        <div className="space-y-6">
          {/* Status Card */}
          <div className={`p-6 border rounded-lg ${getStatusColor(checkResult.status)}`}>
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center">
                {getStatusIcon(checkResult.status)}
                <h3 className="text-lg font-semibold ml-2">
                  Ownership Status: {checkResult.status.toUpperCase()}
                </h3>
              </div>
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  setCheckResult(null);
                  setOwnerName('');
                }}
              >
                Check Again
              </Button>
            </div>

            {checkResult.status === 'verified' && (
              <p className="text-sm">
                ✅ Ownership documents are consistent and verified. Clear title confirmed.
              </p>
            )}
            {checkResult.status === 'incomplete' && (
              <p className="text-sm">
                ⚠️ Some ownership documents are incomplete or require additional verification.
              </p>
            )}
            {checkResult.status === 'mismatch' && (
              <p className="text-sm">
                ❌ Ownership document inconsistencies found. High legal risk detected.
              </p>
            )}
          </div>

          {/* Verification Details */}
          <div className="bg-white border border-gray-200 rounded-lg p-6">
            <h4 className="text-lg font-semibold text-gray-900 mb-4">Verification Details</h4>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="p-3 bg-gray-50 rounded">
                <label className="text-sm font-medium text-gray-600">Owner Name</label>
                <p className="text-sm text-gray-900">{checkResult.owner_name}</p>
              </div>
              
              <div className="p-3 bg-gray-50 rounded">
                <label className="text-sm font-medium text-gray-600">Ownership Type</label>
                <p className="text-sm text-gray-900">{checkResult.details.ownership_type}</p>
              </div>

              <div className="p-3 bg-gray-50 rounded">
                <label className="text-sm font-medium text-gray-600">Title Deed Number</label>
                <p className="text-sm text-gray-900 font-mono">{checkResult.details.title_deed_number}</p>
              </div>

              <div className="p-3 bg-gray-50 rounded">
                <label className="text-sm font-medium text-gray-600">Survey Number</label>
                <p className="text-sm text-gray-900">{checkResult.details.survey_number}</p>
              </div>

              <div className="p-3 bg-gray-50 rounded">
                <label className="text-sm font-medium text-gray-600">Registration Date</label>
                <p className="text-sm text-gray-900">{checkResult.details.registration_date}</p>
              </div>

              <div className="p-3 bg-gray-50 rounded">
                <label className="text-sm font-medium text-gray-600">Previous Owners</label>
                <p className="text-sm text-gray-900">{checkResult.details.previous_owners}</p>
              </div>

              <div className="p-3 bg-gray-50 rounded">
                <label className="text-sm font-medium text-gray-600">Encumbrance Period</label>
                <p className="text-sm text-gray-900">{checkResult.details.encumbrance_period}</p>
              </div>

              <div className="p-3 bg-gray-50 rounded">
                <label className="text-sm font-medium text-gray-600">Legal Issues</label>
                <p className="text-sm text-gray-900">{checkResult.details.legal_issues}</p>
              </div>
            </div>
          </div>

          {/* Verification Checklist */}
          <div className="bg-white border border-gray-200 rounded-lg p-6">
            <h4 className="text-lg font-semibold text-gray-900 mb-4">Verification Checklist</h4>
            
            <div className="space-y-3">
              <div className="flex items-center justify-between p-3 bg-green-50 border border-green-200 rounded">
                <div className="flex items-center">
                  <CheckCircle className="w-5 h-5 text-green-600 mr-3" />
                  <span className="text-sm font-medium text-green-800">Document Consistency</span>
                </div>
                <span className="text-xs text-green-600 bg-green-100 px-2 py-1 rounded">Verified</span>
              </div>

              <div className="flex items-center justify-between p-3 bg-green-50 border border-green-200 rounded">
                <div className="flex items-center">
                  <CheckCircle className="w-5 h-5 text-green-600 mr-3" />
                  <span className="text-sm font-medium text-green-800">Title Clear</span>
                </div>
                <span className="text-xs text-green-600 bg-green-100 px-2 py-1 rounded">Clear</span>
              </div>

              <div className="flex items-center justify-between p-3 bg-green-50 border border-green-200 rounded">
                <div className="flex items-center">
                  <CheckCircle className="w-5 h-5 text-green-600 mr-3" />
                  <span className="text-sm font-medium text-green-800">Registration Valid</span>
                </div>
                <span className="text-xs text-green-600 bg-green-100 px-2 py-1 rounded">Valid</span>
              </div>

              <div className="flex items-center justify-between p-3 bg-green-50 border border-green-200 rounded">
                <div className="flex items-center">
                  <CheckCircle className="w-5 h-5 text-green-600 mr-3" />
                  <span className="text-sm font-medium text-green-800">No Legal Disputes</span>
                </div>
                <span className="text-xs text-green-600 bg-green-100 px-2 py-1 rounded">Clear</span>
              </div>
            </div>
          </div>

          {/* Recommendations */}
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
            <h4 className="text-lg font-semibold text-blue-900 mb-3">Recommendations</h4>
            <ul className="space-y-2 text-sm text-blue-800">
              {checkResult.status === 'verified' && (
                <>
                  <li>• Ownership verification successful - proceed with confidence</li>
                  <li>• Ensure all documents are original during final verification</li>
                  <li>• Get legal opinion for high-value transactions</li>
                </>
              )}
              {checkResult.status === 'incomplete' && (
                <>
                  <li>• Obtain missing ownership documents</li>
                  <li>• Verify with sub-registrar office</li>
                  <li>• Consider legal consultation before proceeding</li>
                </>
              )}
              {checkResult.status === 'mismatch' && (
                <>
                  <li>• ⚠️ CRITICAL - Document inconsistencies found</li>
                  <li>• Immediate legal consultation required</li>
                  <li>• Do not proceed without resolving ownership issues</li>
                </>
              )}
            </ul>
          </div>
        </div>
      )}

      {/* Navigation */}
      <div className="flex justify-between mt-8">
        <Button variant="outline" onClick={onBack}>
          Back to RERA Check
        </Button>
        
        {checkResult && (
          <Button
            onClick={onNext}
            className="bg-[#0056D2] hover:bg-[#003d9a]"
          >
            Continue to Encumbrance Check
          </Button>
        )}
      </div>
    </div>
  );
}