import React, { useState, useEffect } from 'react';
import { Clock, User, CheckCircle, AlertCircle, ArrowLeft } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { riskComplianceService, OwnershipHistoryRecord } from '../../../api/riskCompliance';

interface OwnershipHistoryProps {
  propertyId: string;
  onBack: () => void;
}

interface OwnershipHistoryProps {
  propertyId?: string;
  submissionId?: string;
  onBack: () => void;
}

export function OwnershipHistory({ propertyId, submissionId, onBack }: OwnershipHistoryProps) {
  const [history, setHistory] = useState<OwnershipHistoryRecord[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  useEffect(() => {
    const fetchHistory = async () => {
      try {
        setLoading(true);
        let data: OwnershipHistoryRecord[] = [];
        
        if (submissionId) {
          // Get ownership history based on uploaded documents
          data = await riskComplianceService.getOwnershipHistoryBySubmission(submissionId);
        } else if (propertyId) {
          // Get ownership history by property ID
          data = await riskComplianceService.getOwnershipHistory(propertyId);
        }
        
        // If no data from documents, show dummy data for demo
        if (data.length === 0) {
          data = [
            {
              owner_name: "Rajesh Kumar",
              transfer_date: "2024-01-15",
              transfer_type: "sale",
              previous_owner: "Amit Sharma",
              document_type: "sale_deed",
              is_current_owner: true,
              verified: true
            },
            {
              owner_name: "Amit Sharma",
              transfer_date: "2019-03-20",
              transfer_type: "inheritance",
              previous_owner: "Late Mohan Sharma",
              document_type: "succession_certificate",
              is_current_owner: false,
              verified: true
            },
            {
              owner_name: "Late Mohan Sharma",
              transfer_date: "2010-07-10",
              transfer_type: "sale",
              previous_owner: "ABC Builders",
              document_type: "sale_deed",
              is_current_owner: false,
              verified: true
            }
          ];
        }
        
        setHistory(data);
      } catch (err) {
        console.error('Failed to fetch ownership history:', err);
        setError('Failed to load ownership history');
      } finally {
        setLoading(false);
      }
    };

    fetchHistory();
  }, [propertyId, submissionId]);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="text-center p-8">
        <AlertCircle className="w-12 h-12 text-red-500 mx-auto mb-4" />
        <p className="text-red-600 mb-4">{error}</p>
        <Button onClick={onBack}>Go Back</Button>
      </div>
    );
  }

  return (
    <div className="max-w-4xl mx-auto p-6">
      {/* Header */}
      <div className="mb-8">
        <Button variant="ghost" onClick={onBack} className="mb-4">
          <ArrowLeft className="w-4 h-4 mr-2" />
          Back to Ownership Check
        </Button>
        
        <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center">
          <Clock className="w-6 h-6 text-[#0056D2] mr-2" />
          Property Ownership History
        </h2>
        <p className="text-gray-600">
          Complete timeline of ownership transfers for Property ID: {propertyId}
        </p>
      </div>

      {/* Timeline */}
      <div className="space-y-6">
        {history.map((record, index) => (
          <div key={index} className="relative">
            {/* Timeline line */}
            {index < history.length - 1 && (
              <div className="absolute left-6 top-12 w-0.5 h-16 bg-gray-300"></div>
            )}
            
            {/* Timeline item */}
            <div className="flex items-start space-x-4">
              {/* Timeline dot */}
              <div className={`flex-shrink-0 w-12 h-12 rounded-full flex items-center justify-center ${
                record.is_current_owner 
                  ? 'bg-green-100 border-2 border-green-500' 
                  : 'bg-blue-100 border-2 border-blue-500'
              }`}>
                <User className={`w-6 h-6 ${
                  record.is_current_owner ? 'text-green-600' : 'text-blue-600'
                }`} />
              </div>

              {/* Content */}
              <div className="flex-1 bg-white border border-gray-200 rounded-lg p-6">
                <div className="flex items-start justify-between mb-4">
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900 flex items-center">
                      {record.owner_name}
                      {record.is_current_owner && (
                        <span className="ml-2 px-2 py-1 bg-green-100 text-green-800 text-xs rounded">
                          Current Owner
                        </span>
                      )}
                    </h3>
                    <p className="text-sm text-gray-600">
                      {new Date(record.transfer_date).toLocaleDateString('en-IN', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric'
                      })}
                    </p>
                  </div>
                  
                  {record.verified && (
                    <CheckCircle className="w-5 h-5 text-green-600" />
                  )}
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
                  <div>
                    <span className="font-medium text-gray-600">Transfer Type:</span>
                    <p className="text-gray-900 capitalize">{record.transfer_type}</p>
                  </div>
                  
                  <div>
                    <span className="font-medium text-gray-600">Document Type:</span>
                    <p className="text-gray-900 capitalize">
                      {record.document_type.replace('_', ' ')}
                    </p>
                  </div>
                  
                  {record.previous_owner && (
                    <div>
                      <span className="font-medium text-gray-600">Previous Owner:</span>
                      <p className="text-gray-900">{record.previous_owner}</p>
                    </div>
                  )}
                </div>

                <div className="mt-4 flex items-center justify-between">
                  <div className="flex items-center space-x-4">
                    <span className={`px-2 py-1 rounded text-xs ${
                      record.verified 
                        ? 'bg-green-100 text-green-800' 
                        : 'bg-yellow-100 text-yellow-800'
                    }`}>
                      {record.verified ? 'Verified' : 'Pending Verification'}
                    </span>
                  </div>
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>

      {/* Summary */}
      <div className="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
        <h4 className="text-lg font-semibold text-blue-900 mb-3">
          Ownership Summary
        </h4>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
          <div>
            <span className="font-medium text-blue-700">Total Transfers:</span>
            <p className="text-blue-900">{history.length}</p>
          </div>
          <div>
            <span className="font-medium text-blue-700">Verified Records:</span>
            <p className="text-blue-900">
              {history.filter(h => h.verified).length} of {history.length}
            </p>
          </div>
          <div>
            <span className="font-medium text-blue-700">Property Age:</span>
            <p className="text-blue-900">
              {history.length > 0 
                ? `${new Date().getFullYear() - new Date(history[history.length - 1].transfer_date).getFullYear()} years`
                : 'N/A'
              }
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}