import React, { useState, useEffect } from 'react';
import { X, Clock, User, CheckCircle, AlertCircle } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { riskComplianceService, OwnershipHistoryRecord } from '../../../api/riskCompliance';

interface OwnershipHistoryModalProps {
  isOpen: boolean;
  onClose: () => void;
  submissionId: string;
}

export function OwnershipHistoryModal({ isOpen, onClose, submissionId }: OwnershipHistoryModalProps) {
  const [history, setHistory] = useState<OwnershipHistoryRecord[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  useEffect(() => {
    if (isOpen && submissionId) {
      fetchOwnershipHistory();
    }
  }, [isOpen, submissionId]);

  const fetchOwnershipHistory = async () => {
    try {
      setLoading(true);
      setError('');
      
      // Get ownership history based on uploaded documents
      const data = await riskComplianceService.getOwnershipHistoryBySubmission(submissionId);
      setHistory(data);
    } catch (err) {
      console.error('Failed to fetch ownership history:', err);
      setError('Failed to load ownership history from uploaded documents');
    } finally {
      setLoading(false);
    }
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] flex flex-col">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b">
          <div className="flex items-center">
            <Clock className="w-6 h-6 text-[#0056D2] mr-2" />
            <h2 className="text-xl font-bold text-gray-900">
              Property Ownership History
            </h2>
          </div>
          <button
            onClick={onClose}
            className="text-gray-400 hover:text-gray-600"
          >
            <X className="w-6 h-6" />
          </button>
        </div>

        {/* Content */}
        <div className="flex-1 overflow-hidden p-6">
          {loading ? (
            <div className="flex items-center justify-center h-32">
              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-[#0056D2]"></div>
              <span className="ml-2 text-gray-600">Analyzing uploaded documents...</span>
            </div>
          ) : error ? (
            <div className="text-center p-8">
              <AlertCircle className="w-12 h-12 text-red-500 mx-auto mb-4" />
              <p className="text-red-600 mb-4">{error}</p>
              <Button onClick={fetchOwnershipHistory}>Try Again</Button>
            </div>
          ) : history.length === 0 ? (
            <div className="text-center p-8">
              <Clock className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <p className="text-gray-600 mb-4">
                No ownership history found in uploaded documents.
              </p>
              <p className="text-sm text-gray-500">
                Upload sale deed, encumbrance certificate, or other ownership documents to see the property timeline.
              </p>
            </div>
          ) : (
            <>
              <div className="mb-6">
                <p className="text-gray-600">
                  Ownership timeline extracted from your uploaded documents
                </p>
              </div>

              {/* Timeline */}
              <div className="space-y-6 overflow-y-auto" style={{maxHeight: 'calc(90vh - 300px)'}}>
                {history.map((record, index) => (
                  <div key={index} className="relative">
                    {/* Timeline line */}
                    {index < history.length - 1 && (
                      <div className="absolute left-6 top-12 w-0.5 h-16 bg-gray-300"></div>
                    )}
                    
                    {/* Timeline item */}
                    <div className="flex items-start space-x-4">
                      {/* Timeline dot */}
                      <div className={`flex-shrink-0 w-12 h-12 rounded-full flex items-center justify-center ${
                        record.is_current_owner 
                          ? 'bg-green-100 border-2 border-green-500' 
                          : 'bg-blue-100 border-2 border-blue-500'
                      }`}>
                        <User className={`w-6 h-6 ${
                          record.is_current_owner ? 'text-green-600' : 'text-blue-600'
                        }`} />
                      </div>

                      {/* Content */}
                      <div className="flex-1 bg-white border border-gray-200 rounded-lg p-4 md:p-6">
                        <div className="flex items-start justify-between mb-4">
                          <div>
                            <h3 className="text-lg font-semibold text-gray-900 flex items-center">
                              {record.owner_name}
                              {record.is_current_owner && (
                                <span className="ml-2 px-2 py-1 bg-green-100 text-green-800 text-xs rounded">
                                  Current Owner
                                </span>
                              )}
                            </h3>
                            <p className="text-sm text-gray-600">
                              {new Date(record.transfer_date).toLocaleDateString('en-IN', {
                                year: 'numeric',
                                month: 'long',
                                day: 'numeric'
                              })}
                            </p>
                          </div>
                          
                          {record.verified && (
                            <CheckCircle className="w-5 h-5 text-green-600" />
                          )}
                        </div>

                        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 text-sm">
                          <div>
                            <span className="font-medium text-gray-600">Transfer Type:</span>
                            <p className="text-gray-900 capitalize">{record.transfer_type}</p>
                          </div>
                          
                          <div>
                            <span className="font-medium text-gray-600">Document Type:</span>
                            <p className="text-gray-900 capitalize">
                              {record.document_type.replace('_', ' ')}
                            </p>
                          </div>
                          
                          {record.previous_owner && (
                            <div>
                              <span className="font-medium text-gray-600">Previous Owner:</span>
                              <p className="text-gray-900">{record.previous_owner}</p>
                            </div>
                          )}
                        </div>

                        <div className="mt-4 flex items-center justify-between">
                          <span className={`px-2 py-1 rounded text-xs ${
                            record.verified 
                              ? 'bg-green-100 text-green-800' 
                              : 'bg-yellow-100 text-yellow-800'
                          }`}>
                            {record.verified ? 'Verified from Documents' : 'Pending Verification'}
                          </span>
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>

              {/* Summary */}
              <div className="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
                <h4 className="text-lg font-semibold text-blue-900 mb-3">
                  Document Analysis Summary
                </h4>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 text-sm">
                  <div>
                    <span className="font-medium text-blue-700">Total Transfers:</span>
                    <p className="text-blue-900">{history.length}</p>
                  </div>
                  <div>
                    <span className="font-medium text-blue-700">Verified Records:</span>
                    <p className="text-blue-900">
                      {history.filter(h => h.verified).length} of {history.length}
                    </p>
                  </div>
                  <div>
                    <span className="font-medium text-blue-700">Source:</span>
                    <p className="text-blue-900">Uploaded Documents</p>
                  </div>
                </div>
              </div>
            </>
          )}
        </div>

        {/* Footer */}
        <div className="flex-shrink-0 flex justify-end p-6 border-t bg-gray-50">
          <Button onClick={onClose} className="bg-[#0056D2] hover:bg-[#003d9a]">
            Close
          </Button>
        </div>
      </div>
    </div>
  );
}