import React, { useState } from 'react';
import { Shield, CheckCircle, AlertTriangle, X, Search } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { Input } from '../../../components/ui/input';
import { riskComplianceService, ReraCheckResponse } from '../../../api/riskCompliance';

interface ReraCheckProps {
  submissionId: string;
  onCheckCompleted: (result: ReraCheckResponse) => void;
  onNext: () => void;
  onBack: () => void;
}

export function ReraCheck({ submissionId, onCheckCompleted, onNext, onBack }: ReraCheckProps) {
  const [reraId, setReraId] = useState('');
  const [projectName, setProjectName] = useState('');
  const [isChecking, setIsChecking] = useState(false);
  const [checkResult, setCheckResult] = useState<ReraCheckResponse | null>(null);
  const [error, setError] = useState<string>('');

  const handleReraCheck = async () => {
    if (!reraId && !projectName) {
      setError('Please provide either RERA ID or Project Name');
      return;
    }

    setIsChecking(true);
    setError('');

    try {
      const result = await riskComplianceService.performReraCheck({
        rera_id: reraId,
        project_name: projectName,
        submission_id: submissionId
      });

      // Simulate getting the check result (in real app, you'd fetch it)
      const mockResult: ReraCheckResponse = {
        id: result.check_id,
        submission_id: submissionId,
        rera_id: reraId,
        project_name: projectName,
        status: reraId && reraId.length >= 10 ? 'pass' : reraId ? 'warning' : 'fail',
        details: {
          registration_number: reraId,
          project_name: projectName,
          validity: reraId ? 'Active' : 'Not Found',
          completion_percentage: reraId ? 89 : 0,
          valid_till: reraId ? '2025-12-31' : null,
          developer_name: 'Sample Developer Ltd.',
          project_area: '15.5 acres',
          total_units: 450,
          approved_plans: reraId ? 'Yes' : 'No'
        },
        checked_at: new Date().toISOString()
      };

      setCheckResult(mockResult);
      onCheckCompleted(mockResult);
    } catch (error) {
      console.error('RERA check failed:', error);
      setError('Failed to perform RERA check. Please try again.');
    } finally {
      setIsChecking(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'pass': return 'text-green-600 bg-green-50 border-green-200';
      case 'warning': return 'text-yellow-600 bg-yellow-50 border-yellow-200';
      case 'fail': return 'text-red-600 bg-red-50 border-red-200';
      default: return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'pass': return <CheckCircle className="w-5 h-5 text-green-600" />;
      case 'warning': return <AlertTriangle className="w-5 h-5 text-yellow-600" />;
      case 'fail': return <X className="w-5 h-5 text-red-600" />;
      default: return null;
    }
  };

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center">
          <Shield className="w-6 h-6 text-[#0056D2] mr-2" />
          RERA Compliance Check
        </h2>
        <p className="text-gray-600">
          Verify the RERA registration status of the project to ensure legal compliance.
        </p>
      </div>

      {!checkResult ? (
        <div className="bg-white border border-gray-200 rounded-lg p-6">
          <div className="space-y-6">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                RERA Registration ID
              </label>
              <Input
                type="text"
                placeholder="Enter RERA ID (e.g., PRM/KA/RERA/1251/446/PR/010819/002054)"
                value={reraId}
                onChange={(e) => setReraId(e.target.value)}
                className="w-full"
              />
              <p className="text-xs text-gray-500 mt-1">
                RERA ID is usually found on project brochures and agreements
              </p>
            </div>

            <div className="text-center text-gray-500">
              <span>OR</span>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Project Name
              </label>
              <Input
                type="text"
                placeholder="Enter project name for RERA lookup"
                value={projectName}
                onChange={(e) => setProjectName(e.target.value)}
                className="w-full"
              />
              <p className="text-xs text-gray-500 mt-1">
                We'll search for the project in RERA database
              </p>
            </div>

            {error && (
              <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
                <div className="flex items-center">
                  <X className="w-5 h-5 text-red-600 mr-2" />
                  <p className="text-sm text-red-800">{error}</p>
                </div>
              </div>
            )}

            <Button
              onClick={handleReraCheck}
              disabled={isChecking || (!reraId && !projectName)}
              className="w-full bg-[#0056D2] hover:bg-[#003d9a] disabled:opacity-50"
            >
              {isChecking ? (
                <div className="flex items-center">
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  Checking RERA Status...
                </div>
              ) : (
                <div className="flex items-center">
                  <Search className="w-4 h-4 mr-2" />
                  Check RERA Compliance
                </div>
              )}
            </Button>
          </div>
        </div>
      ) : (
        <div className="space-y-6">
          {/* Status Card */}
          <div className={`p-6 border rounded-lg ${getStatusColor(checkResult.status)}`}>
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center">
                {getStatusIcon(checkResult.status)}
                <h3 className="text-lg font-semibold ml-2">
                  RERA Status: {checkResult.status.toUpperCase()}
                </h3>
              </div>
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  setCheckResult(null);
                  setReraId('');
                  setProjectName('');
                }}
              >
                Check Again
              </Button>
            </div>

            {checkResult.status === 'pass' && (
              <p className="text-sm">
                ✅ Project is registered with RERA and compliant with regulations.
              </p>
            )}
            {checkResult.status === 'warning' && (
              <p className="text-sm">
                ⚠️ RERA registration found but requires verification. Please check details carefully.
              </p>
            )}
            {checkResult.status === 'fail' && (
              <p className="text-sm">
                ❌ No valid RERA registration found. This is a high-risk investment.
              </p>
            )}
          </div>

          {/* Details */}
          <div className="bg-white border border-gray-200 rounded-lg p-6">
            <h4 className="text-lg font-semibold text-gray-900 mb-4">Registration Details</h4>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {checkResult.rera_id && (
                <div className="p-3 bg-gray-50 rounded">
                  <label className="text-sm font-medium text-gray-600">RERA ID</label>
                  <p className="text-sm text-gray-900 font-mono">{checkResult.rera_id}</p>
                </div>
              )}
              
              {checkResult.project_name && (
                <div className="p-3 bg-gray-50 rounded">
                  <label className="text-sm font-medium text-gray-600">Project Name</label>
                  <p className="text-sm text-gray-900">{checkResult.project_name}</p>
                </div>
              )}

              {checkResult.details.developer_name && (
                <div className="p-3 bg-gray-50 rounded">
                  <label className="text-sm font-medium text-gray-600">Developer</label>
                  <p className="text-sm text-gray-900">{checkResult.details.developer_name}</p>
                </div>
              )}

              {checkResult.details.validity && (
                <div className="p-3 bg-gray-50 rounded">
                  <label className="text-sm font-medium text-gray-600">Status</label>
                  <p className="text-sm text-gray-900">{checkResult.details.validity}</p>
                </div>
              )}

              {checkResult.details.completion_percentage && (
                <div className="p-3 bg-gray-50 rounded">
                  <label className="text-sm font-medium text-gray-600">Completion</label>
                  <p className="text-sm text-gray-900">{checkResult.details.completion_percentage}%</p>
                </div>
              )}

              {checkResult.details.valid_till && (
                <div className="p-3 bg-gray-50 rounded">
                  <label className="text-sm font-medium text-gray-600">Valid Till</label>
                  <p className="text-sm text-gray-900">{checkResult.details.valid_till}</p>
                </div>
              )}

              {checkResult.details.project_area && (
                <div className="p-3 bg-gray-50 rounded">
                  <label className="text-sm font-medium text-gray-600">Project Area</label>
                  <p className="text-sm text-gray-900">{checkResult.details.project_area}</p>
                </div>
              )}

              {checkResult.details.total_units && (
                <div className="p-3 bg-gray-50 rounded">
                  <label className="text-sm font-medium text-gray-600">Total Units</label>
                  <p className="text-sm text-gray-900">{checkResult.details.total_units}</p>
                </div>
              )}
            </div>
          </div>

          {/* Recommendations */}
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
            <h4 className="text-lg font-semibold text-blue-900 mb-3">Recommendations</h4>
            <ul className="space-y-2 text-sm text-blue-800">
              {checkResult.status === 'pass' && (
                <>
                  <li>• Project is RERA compliant - good for investment</li>
                  <li>• Verify completion timeline with developer</li>
                  <li>• Check for any pending approvals</li>
                </>
              )}
              {checkResult.status === 'warning' && (
                <>
                  <li>• Verify RERA details with official RERA website</li>
                  <li>• Check for any compliance issues or penalties</li>
                  <li>• Consult legal expert before proceeding</li>
                </>
              )}
              {checkResult.status === 'fail' && (
                <>
                  <li>• ⚠️ AVOID - No valid RERA registration found</li>
                  <li>• High legal and financial risk</li>
                  <li>• Consider other RERA-compliant projects</li>
                </>
              )}
            </ul>
          </div>
        </div>
      )}

      {/* Navigation */}
      <div className="flex justify-between mt-8">
        <Button variant="outline" onClick={onBack}>
          Back to Documents
        </Button>
        
        {checkResult && (
          <Button
            onClick={onNext}
            className="bg-[#0056D2] hover:bg-[#003d9a]"
          >
            Continue to Ownership Check
          </Button>
        )}
      </div>
    </div>
  );
}