import React, { useState } from 'react';
import { Shield, ArrowLeft } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { Input } from '../../../components/ui/input';
import { useNavigate } from 'react-router-dom';
import { Stepper } from './Stepper';
import { DocumentUpload } from './DocumentUpload';
import { ReraCheck } from './ReraCheck';
import { OwnershipCheck } from './OwnershipCheck';
import { EncumbranceCheck } from './EncumbranceCheck';
import { RiskScore } from './RiskScore';
import { RiskReport } from './RiskReport';
import { VerificationScore } from './VerificationScore';
import { TitleChainAnalysis } from './TitleChainAnalysis';
import { LitigationAnalysis } from './LitigationAnalysis';
import { riskComplianceService, RiskSubmissionResponse, ReraCheckResponse, OwnershipCheckResponse, EncumbranceCheckResponse, RiskScoreResponse } from '../../../api/riskCompliance';

// Wrapper component to fetch actual submission data
function RiskReportWrapper({ submissionId, onBack, onComplete }: { submissionId: string, onBack: () => void, onComplete: () => void }) {
  const [submission, setSubmission] = useState<RiskSubmissionResponse | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  React.useEffect(() => {
    const fetchSubmission = async () => {
      try {
        setLoading(true);
        const data = await riskComplianceService.getSubmission(submissionId);
        setSubmission(data);
      } catch (err) {
        console.error('Error fetching submission:', err);
        setError('Failed to load submission data');
      } finally {
        setLoading(false);
      }
    };

    if (submissionId) {
      fetchSubmission();
    }
  }, [submissionId]);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (error || !submission) {
    return (
      <div className="text-center p-8">
        <p className="text-red-600">{error || 'Failed to load submission'}</p>
        <Button onClick={onBack} className="mt-4">Go Back</Button>
      </div>
    );
  }

  return (
    <RiskReport
      submission={submission}
      onBack={onBack}
      onComplete={onComplete}
    />
  );
}

const STEPS = [
  { id: 1, title: 'Property Info', description: 'Basic details' },
  { id: 2, title: 'Documents', description: 'Upload files' },
  { id: 3, title: 'RERA Check', description: 'Compliance' },
  { id: 4, title: 'Ownership', description: 'Verification' },
  { id: 5, title: 'Title Chain', description: 'Chain Analysis' },
  { id: 6, title: 'Litigation', description: 'Legal Disputes' },
  { id: 7, title: 'Encumbrance', description: 'Liabilities' },
  { id: 8, title: 'Risk Score', description: 'Assessment' },
  { id: 9, title: 'Verification Score', description: 'Overall Score' },
  { id: 10, title: 'Report', description: 'Final report' }
];

export function RiskCompliancePage() {
  const [currentStep, setCurrentStep] = useState(1);
  const [completedSteps, setCompletedSteps] = useState<number[]>([]);
  const [submissionId, setSubmissionId] = useState<string>('');
  const [submission, setSubmission] = useState<RiskSubmissionResponse | null>(null);
  const navigate = useNavigate();
  
  // Form data
  const [propertyForm, setPropertyForm] = useState({
    property_address: '',
    builder_name: '',
    project_name: '',
    property_type: '',
    budget_range: ''
  });

  // Check results
  const [reraResult, setReraResult] = useState<ReraCheckResponse | null>(null);
  const [ownershipResult, setOwnershipResult] = useState<OwnershipCheckResponse | null>(null);
  const [encumbranceResult, setEncumbranceResult] = useState<EncumbranceCheckResponse | null>(null);
  const [scoreResult, setScoreResult] = useState<RiskScoreResponse | null>(null);

  const [isSubmitting, setIsSubmitting] = useState(false);
  const [error, setError] = useState<string>('');

  const handlePropertySubmit = async () => {
    if (!propertyForm.property_address || !propertyForm.property_type) {
      setError('Please fill in all required fields');
      return;
    }

    setIsSubmitting(true);
    setError('');

    try {
      const result = await riskComplianceService.createSubmission(propertyForm);
      setSubmissionId(result.submission_id);
      setCompletedSteps([1]);
      setCurrentStep(2);
    } catch (error) {
      console.error('Submission failed:', error);
      setError('Failed to create submission. Please try again.');
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleStepComplete = (stepId: number) => {
    if (!completedSteps.includes(stepId)) {
      setCompletedSteps([...completedSteps, stepId]);
    }
  };

  const handleDocumentUploaded = (documentType: string, documentId: string) => {
    console.log('Document uploaded:', documentType, documentId);
  };

  const handleReraCheckCompleted = (result: ReraCheckResponse) => {
    setReraResult(result);
    handleStepComplete(3);
  };

  const handleOwnershipCheckCompleted = (result: OwnershipCheckResponse) => {
    setOwnershipResult(result);
    handleStepComplete(4);
  };

  const handleEncumbranceCheckCompleted = (result: EncumbranceCheckResponse) => {
    setEncumbranceResult(result);
    handleStepComplete(5);
  };

  const handleScoreCalculated = (result: RiskScoreResponse) => {
    setScoreResult(result);
    handleStepComplete(6);
  };

  const handleComplete = () => {
    // Navigate to home page
    navigate('/');
  };

  const goToStep = (step: number) => {
    // Allow navigation to any step after property info is completed
    if (completedSteps.includes(1) || step === 1) {
      setCurrentStep(step);
    } else if (step > 1 && !submissionId) {
      // If trying to access steps 2+ without submissionId, show a message
      setError('Please complete Property Info first to access other features');
      setTimeout(() => setError(''), 3000);
    }
  };

  const renderCurrentStep = () => {
    switch (currentStep) {
      case 1:
        return (
          <div className="max-w-2xl mx-auto p-6">
            <div className="mb-8 text-center">
              <Shield className="w-16 h-16 text-[#0056D2] mx-auto mb-4" />
              <h2 className="text-2xl font-bold text-gray-900 mb-4">Risk & Compliance Assessment</h2>
              <p className="text-gray-600">
                Get a comprehensive risk analysis of your property investment with legal compliance verification.
              </p>
            </div>

            <div className="bg-white border border-gray-200 rounded-lg p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-6">Property Information</h3>
              
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Property Address *
                  </label>
                  <Input
                    type="text"
                    placeholder="Enter complete property address"
                    value={propertyForm.property_address}
                    onChange={(e) => setPropertyForm({...propertyForm, property_address: e.target.value})}
                    className="w-full"
                  />
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Builder/Developer Name
                    </label>
                    <Input
                      type="text"
                      placeholder="Enter builder name"
                      value={propertyForm.builder_name}
                      onChange={(e) => setPropertyForm({...propertyForm, builder_name: e.target.value})}
                      className="w-full"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Project Name
                    </label>
                    <Input
                      type="text"
                      placeholder="Enter project name"
                      value={propertyForm.project_name}
                      onChange={(e) => setPropertyForm({...propertyForm, project_name: e.target.value})}
                      className="w-full"
                    />
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Property Type *
                    </label>
                    <select
                      className="w-full h-10 rounded-md border border-input bg-background px-3 py-2 text-sm"
                      value={propertyForm.property_type}
                      onChange={(e) => setPropertyForm({...propertyForm, property_type: e.target.value})}
                    >
                      <option value="">Select Type</option>
                      <option value="apartment">Apartment</option>
                      <option value="villa">Villa</option>
                      <option value="house">House</option>
                      <option value="commercial">Commercial</option>
                    </select>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Budget Range
                    </label>
                    <select
                      className="w-full h-10 rounded-md border border-input bg-background px-3 py-2 text-sm"
                      value={propertyForm.budget_range}
                      onChange={(e) => setPropertyForm({...propertyForm, budget_range: e.target.value})}
                    >
                      <option value="">Select Budget</option>
                      <option value="50L-1Cr">₹50L - ₹1Cr</option>
                      <option value="1Cr-2Cr">₹1Cr - ₹2Cr</option>
                      <option value="2Cr-5Cr">₹2Cr - ₹5Cr</option>
                      <option value="5Cr+">₹5Cr+</option>
                    </select>
                  </div>
                </div>

                {error && (
                  <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
                    <p className="text-sm text-red-800">{error}</p>
                  </div>
                )}

                <Button
                  onClick={handlePropertySubmit}
                  disabled={isSubmitting}
                  className="w-full bg-[#0056D2] hover:bg-[#003d9a] disabled:opacity-50"
                >
                  {isSubmitting ? (
                    <div className="flex items-center">
                      <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                      Starting Assessment...
                    </div>
                  ) : (
                    'Start Risk Assessment'
                  )}
                </Button>
              </div>
            </div>
          </div>
        );

      case 2:
        return (
          <DocumentUpload
            submissionId={submissionId}
            onDocumentUploaded={handleDocumentUploaded}
            onNext={() => {
              handleStepComplete(2);
              setCurrentStep(3);
            }}
          />
        );

      case 3:
        return (
          <ReraCheck
            submissionId={submissionId}
            onCheckCompleted={handleReraCheckCompleted}
            onNext={() => setCurrentStep(4)}
            onBack={() => setCurrentStep(2)}
          />
        );

      case 4:
        return (
          <OwnershipCheck
            submissionId={submissionId}
            onCheckCompleted={handleOwnershipCheckCompleted}
            onNext={() => setCurrentStep(5)}
            onBack={() => setCurrentStep(3)}
          />
        );

      case 5:
        return (
          <TitleChainAnalysis
            submissionId={submissionId}
            onNext={() => setCurrentStep(6)}
            onBack={() => setCurrentStep(4)}
          />
        );

      case 6:
        return (
          <LitigationAnalysis
            submissionId={submissionId}
            onNext={() => setCurrentStep(7)}
            onBack={() => setCurrentStep(5)}
          />
        );

      case 7:
        return (
          <EncumbranceCheck
            submissionId={submissionId}
            onCheckCompleted={handleEncumbranceCheckCompleted}
            onNext={() => setCurrentStep(8)}
            onBack={() => setCurrentStep(6)}
          />
        );

      case 8:
        return (
          <RiskScore
            submissionId={submissionId}
            onScoreCalculated={handleScoreCalculated}
            onNext={() => {
              handleStepComplete(8);
              setCurrentStep(9);
            }}
            onBack={() => setCurrentStep(7)}
          />
        );

      case 9:
        return (
          <VerificationScore
            submissionId={submissionId}
            onBack={() => {
              handleStepComplete(9);
              setCurrentStep(10);
            }}
          />
        );

      case 10:
        return (
          <RiskReportWrapper
            submissionId={submissionId}
            onBack={() => setCurrentStep(9)}
            onComplete={handleComplete}
          />
        );

      default:
        return null;
    }
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            <div className="flex items-center">
              <Button
                variant="ghost"
                onClick={() => window.history.back()}
                className="mr-4"
              >
                <ArrowLeft className="w-4 h-4 mr-2" />
                Back
              </Button>
              <h1 className="text-xl font-semibold text-gray-900">Risk & Compliance Assessment</h1>
            </div>
          </div>
        </div>
      </div>

      {/* Stepper */}
      {currentStep > 1 && (
        <div className="bg-white border-b border-gray-200">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <Stepper
              steps={STEPS}
              currentStep={currentStep}
              completedSteps={completedSteps}
              onStepClick={goToStep}
            />
          </div>
        </div>
      )}

      {/* Main Content */}
      <div className="py-8">
        {renderCurrentStep()}
      </div>
    </div>
  );
}