import React, { useState } from 'react';
import { FileText, Download, Share2, AlertTriangle, CheckCircle, Clock, Mail, Phone } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { riskComplianceService, RiskSubmissionResponse } from '../../../api/riskCompliance';
import jsPDF from 'jspdf';
import { useNavigate } from 'react-router-dom';

interface RiskReportProps {
  submission: RiskSubmissionResponse;
  onBack: () => void;
  onComplete: () => void;
}

export function RiskReport({ submission, onBack, onComplete }: RiskReportProps) {
  const [isGeneratingReport, setIsGeneratingReport] = useState(false);
  const [reportUrl, setReportUrl] = useState<string | null>(null);
  const [error, setError] = useState<string>('');
  const navigate = useNavigate();

  const handleGenerateReport = async () => {
    setIsGeneratingReport(true);
    setError('');

    try {
      // Generate PDF using jsPDF
      const pdf = new jsPDF('p', 'mm', 'a4');
      const pageWidth = pdf.internal.pageSize.getWidth();
      const pageHeight = pdf.internal.pageSize.getHeight();
      let yPosition = 20;

      // Header
      pdf.setFontSize(24);
      pdf.setTextColor(0, 86, 210); // NAL Blue
      pdf.text('NAL REAL ESTATE', pageWidth / 2, yPosition, { align: 'center' });
      yPosition += 10;
      
      pdf.setFontSize(18);
      pdf.setTextColor(0, 0, 0);
      pdf.text('Risk & Compliance Assessment Report', pageWidth / 2, yPosition, { align: 'center' });
      yPosition += 15;
      
      pdf.setFontSize(10);
      pdf.setTextColor(100, 100, 100);
      pdf.text(`Property ID: ${submission.id}`, pageWidth / 2, yPosition, { align: 'center' });
      yPosition += 5;
      pdf.text(`Generated on: ${new Date().toLocaleDateString()}`, pageWidth / 2, yPosition, { align: 'center' });
      yPosition += 20;

      // Property Information Section
      pdf.setFontSize(14);
      pdf.setTextColor(0, 0, 0);
      pdf.text('Property Information', 20, yPosition);
      yPosition += 10;
      
      pdf.setFontSize(10);
      pdf.text(`Address: ${submission.property_address}`, 20, yPosition);
      yPosition += 6;
      
      if (submission.builder_name) {
        pdf.text(`Builder: ${submission.builder_name}`, 20, yPosition);
        yPosition += 6;
      }
      
      if (submission.project_name) {
        pdf.text(`Project: ${submission.project_name}`, 20, yPosition);
        yPosition += 6;
      }
      
      pdf.text(`Property Type: ${submission.property_type}`, 20, yPosition);
      yPosition += 6;
      
      if (submission.budget_range) {
        pdf.text(`Budget Range: ${submission.budget_range}`, 20, yPosition);
        yPosition += 15;
      }

      // Risk Score Section
      if (submission.risk_score) {
        pdf.setFontSize(14);
        pdf.text('Risk Assessment Summary', 20, yPosition);
        yPosition += 10;
        
        const riskColor = submission.risk_score.risk_level === 'low' ? [34, 197, 94] : 
                         submission.risk_score.risk_level === 'high' ? [239, 68, 68] : [245, 158, 11];
        
        pdf.setTextColor(riskColor[0], riskColor[1], riskColor[2]);
        pdf.setFontSize(24);
        pdf.text(`${submission.risk_score.overall_score}/100`, pageWidth / 2, yPosition, { align: 'center' });
        yPosition += 10;
        
        pdf.setFontSize(16);
        pdf.text(`${submission.risk_score.risk_level.toUpperCase()} RISK`, pageWidth / 2, yPosition, { align: 'center' });
        yPosition += 15;
      }

      // Verification Results
      pdf.setFontSize(14);
      pdf.setTextColor(0, 0, 0);
      pdf.text('Verification Results', 20, yPosition);
      yPosition += 10;
      
      pdf.setFontSize(10);
      
      // RERA Check
      const reraStatus = submission.rera_check?.status || 'Not Checked';
      pdf.text(`RERA Compliance: ${reraStatus.toUpperCase()}`, 20, yPosition);
      yPosition += 6;
      
      // Ownership Check
      const ownershipStatus = submission.ownership_check?.status || 'Not Checked';
      pdf.text(`Ownership Verification: ${ownershipStatus.toUpperCase()}`, 20, yPosition);
      yPosition += 6;
      
      // Encumbrance Check
      const encumbranceStatus = submission.encumbrance_check?.status || 'Not Checked';
      pdf.text(`Encumbrance Check: ${encumbranceStatus.toUpperCase()}`, 20, yPosition);
      yPosition += 6;
      
      // Documents
      const docCount = submission.documents?.length || 0;
      pdf.text(`Documents: ${docCount}/5 uploaded`, 20, yPosition);
      yPosition += 15;

      // Recommendations
      pdf.setFontSize(14);
      pdf.text('Recommendations', 20, yPosition);
      yPosition += 10;
      
      pdf.setFontSize(10);
      const recommendations = submission.risk_score?.risk_level === 'low' ? [
        '✓ Property shows excellent compliance - Safe to proceed',
        '✓ All major risk factors have been addressed',
        '✓ Consider finalizing the purchase agreement'
      ] : submission.risk_score?.risk_level === 'high' ? [
        '⚠ High risk factors identified - Proceed with caution',
        '⚠ Consult legal expert before making commitments',
        '⚠ Consider alternative properties'
      ] : [
        '⚠ Some risk factors need attention',
        '⚠ Complete all pending verifications',
        '⚠ Consult with our experts for guidance'
      ];
      
      recommendations.forEach(rec => {
        pdf.text(rec, 20, yPosition);
        yPosition += 6;
      });
      
      yPosition += 10;
      pdf.text('📋 Keep all verification documents for future reference', 20, yPosition);
      yPosition += 6;
      pdf.text('🔄 Re-verify compliance status before final purchase', 20, yPosition);

      // Footer
      pdf.setFontSize(8);
      pdf.setTextColor(100, 100, 100);
      const footerY = pageHeight - 30;
      pdf.text('NAL Real Estate Platform', pageWidth / 2, footerY, { align: 'center' });
      pdf.text('This report is generated based on available data and should be used for reference only.', pageWidth / 2, footerY + 5, { align: 'center' });
      pdf.text('For expert consultation, contact: support@nal.com | +91-80-1234-5678', pageWidth / 2, footerY + 10, { align: 'center' });
      pdf.text(`© ${new Date().getFullYear()} NAL Real Estate. All rights reserved. | Report ID: ${submission.id}`, pageWidth / 2, footerY + 15, { align: 'center' });

      // Save PDF
      const fileName = `NAL-Risk-Compliance-Report-${submission.id}.pdf`;
      pdf.save(fileName);
      
      setReportUrl(`downloaded://${fileName}`);
      
      // Show success message
      alert('NAL Risk Compliance Report downloaded successfully as PDF!');
    } catch (error) {
      console.error('Report generation failed:', error);
      setError('Failed to generate report. Please try again.');
    } finally {
      setIsGeneratingReport(false);
    }
  };

  const handleDownloadReport = () => {
    if (reportUrl) {
      window.open(reportUrl, '_blank');
    }
  };

  const handleShareReport = async () => {
    if (navigator.share && reportUrl) {
      try {
        await navigator.share({
          title: 'Risk & Compliance Report',
          text: 'Property Risk Assessment Report',
          url: reportUrl
        });
      } catch (error) {
        console.log('Share failed:', error);
      }
    } else {
      // Fallback: copy to clipboard
      if (reportUrl) {
        navigator.clipboard.writeText(reportUrl);
        alert('Report link copied to clipboard!');
      }
    }
  };

  const getRiskColor = (level?: string) => {
    switch (level) {
      case 'low': return 'text-green-600 bg-green-50 border-green-200';
      case 'medium': return 'text-yellow-600 bg-yellow-50 border-yellow-200';
      case 'high': return 'text-orange-600 bg-orange-50 border-orange-200';
      case 'critical': return 'text-red-600 bg-red-50 border-red-200';
      default: return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center">
          <FileText className="w-6 h-6 text-[#0056D2] mr-2" />
          Risk Assessment Report
        </h2>
        <p className="text-gray-600">
          Your comprehensive property risk analysis is complete. Review the summary and download the detailed report.
        </p>
      </div>

      {/* Report Summary */}
      <div className="bg-white border border-gray-200 rounded-lg p-6 mb-8">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Report Summary</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <h4 className="font-medium text-gray-700 mb-3">Property Information</h4>
            <div className="space-y-2 text-sm">
              <div><span className="text-gray-600">Address:</span> {submission.property_address}</div>
              {submission.builder_name && (
                <div><span className="text-gray-600">Builder:</span> {submission.builder_name}</div>
              )}
              {submission.project_name && (
                <div><span className="text-gray-600">Project:</span> {submission.project_name}</div>
              )}
              <div><span className="text-gray-600">Type:</span> {submission.property_type}</div>
              {submission.budget_range && (
                <div><span className="text-gray-600">Budget:</span> {submission.budget_range}</div>
              )}
            </div>
          </div>

          <div>
            <h4 className="font-medium text-gray-700 mb-3">Assessment Status</h4>
            <div className="space-y-2 text-sm">
              <div><span className="text-gray-600">Documents:</span> {(() => {
                console.log('Documents array:', submission.documents);
                const count = submission.documents?.length || 0;
                return `${count}/5 uploaded`;
              })()}</div>
              <div><span className="text-gray-600">RERA Status:</span> 
                <span className={`ml-1 px-2 py-1 rounded text-xs ${
                  submission.rera_check?.status === 'pass' ? 'bg-green-100 text-green-800' :
                  submission.rera_check?.status === 'warning' ? 'bg-yellow-100 text-yellow-800' :
                  'bg-red-100 text-red-800'
                }`}>
                  {submission.rera_check?.status?.toUpperCase() || 'Not Checked'}
                </span>
              </div>
              <div><span className="text-gray-600">Ownership:</span>
                <span className={`ml-1 px-2 py-1 rounded text-xs ${
                  submission.ownership_check?.status === 'verified' ? 'bg-green-100 text-green-800' :
                  submission.ownership_check?.status === 'incomplete' ? 'bg-yellow-100 text-yellow-800' :
                  'bg-red-100 text-red-800'
                }`}>
                  {submission.ownership_check?.status?.toUpperCase() || 'Not Checked'}
                </span>
              </div>
              <div><span className="text-gray-600">Encumbrances:</span>
                <span className={`ml-1 px-2 py-1 rounded text-xs ${
                  submission.encumbrance_check?.status === 'clear' ? 'bg-green-100 text-green-800' :
                  submission.encumbrance_check?.status === 'pending' ? 'bg-yellow-100 text-yellow-800' :
                  'bg-red-100 text-red-800'
                }`}>
                  {submission.encumbrance_check?.status?.toUpperCase() || 'Not Checked'}
                </span>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Risk Score Card */}
      {submission.risk_score && (
        <div className={`p-6 border-2 rounded-lg mb-8 ${getRiskColor(submission.risk_score.risk_level)}`}>
          <div className="flex items-center justify-between">
            <div>
              <h3 className="text-xl font-bold mb-2">Final Risk Assessment</h3>
              <div className="flex items-center space-x-4">
                <div className="text-3xl font-bold">
                  {submission.risk_score.overall_score}/100
                </div>
                <div className="text-lg font-semibold uppercase">
                  {submission.risk_score.risk_level} Risk
                </div>
              </div>
            </div>
            <div className="text-right">
              {submission.risk_score.risk_level === 'critical' ? (
                <div className="flex items-center text-red-600">
                  <AlertTriangle className="w-6 h-6 mr-2" />
                  <span className="font-bold">AVOID PROPERTY</span>
                </div>
              ) : submission.risk_score.risk_level === 'low' ? (
                <div className="flex items-center text-green-600">
                  <CheckCircle className="w-6 h-6 mr-2" />
                  <span className="font-bold">SAFE TO PROCEED</span>
                </div>
              ) : (
                <div className="flex items-center text-yellow-600">
                  <Clock className="w-6 h-6 mr-2" />
                  <span className="font-bold">PROCEED WITH CAUTION</span>
                </div>
              )}
            </div>
          </div>
        </div>
      )}

      {/* Key Findings */}
      <div className="bg-white border border-gray-200 rounded-lg p-6 mb-8">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Key Findings</h3>
        
        <div className="space-y-4">
          {/* RERA Finding */}
          {submission.rera_check && (
            <div className="flex items-start p-3 bg-gray-50 rounded">
              <div className={`w-3 h-3 rounded-full mt-1.5 mr-3 ${
                submission.rera_check.status === 'pass' ? 'bg-green-500' :
                submission.rera_check.status === 'warning' ? 'bg-yellow-500' : 'bg-red-500'
              }`}></div>
              <div>
                <span className="font-medium">RERA Compliance:</span>
                <span className="ml-2">{submission.rera_check.status === 'pass' ? 'Project is RERA registered and compliant' :
                  submission.rera_check.status === 'warning' ? 'RERA registration needs verification' :
                  'No valid RERA registration found'}</span>
              </div>
            </div>
          )}

          {/* Encumbrance Finding */}
          {submission.encumbrance_check && (
            <div className="flex items-start p-3 bg-gray-50 rounded">
              <div className={`w-3 h-3 rounded-full mt-1.5 mr-3 ${
                submission.encumbrance_check.status === 'clear' ? 'bg-green-500' :
                submission.encumbrance_check.status === 'pending' ? 'bg-yellow-500' : 'bg-red-500'
              }`}></div>
              <div>
                <span className="font-medium">Encumbrances:</span>
                <span className="ml-2">{submission.encumbrance_check.status === 'clear' ? 'No encumbrances found - clear title' :
                  submission.encumbrance_check.status === 'pending' ? 'Some encumbrances pending clearance' :
                  'Active bank loan of ₹2.3 Crores found - high risk'}</span>
              </div>
            </div>
          )}

          {/* Document Finding */}
          <div className="flex items-start p-3 bg-gray-50 rounded">
            <div className={`w-3 h-3 rounded-full mt-1.5 mr-3 ${
              (submission.documents?.length || 0) >= 5 ? 'bg-green-500' : 'bg-yellow-500'
            }`}></div>
            <div>
              <span className="font-medium">Documents:</span>
              <span className="ml-2">{submission.documents?.length || 0}/5 required documents uploaded</span>
            </div>
          </div>
        </div>
      </div>

      {/* Report Generation */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 mb-8">
        <h3 className="text-lg font-semibold text-blue-900 mb-4">Detailed PDF Report</h3>
        
        {!reportUrl ? (
          <div className="text-center">
            <FileText className="w-12 h-12 text-blue-600 mx-auto mb-4" />
            <p className="text-blue-800 mb-4">
              Generate a comprehensive PDF report with all verification details, risk analysis, and recommendations.
            </p>
            
            {error && (
              <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded text-red-800 text-sm">
                {error}
              </div>
            )}
            
            <Button
              onClick={handleGenerateReport}
              disabled={isGeneratingReport}
              className="bg-blue-600 hover:bg-blue-700"
            >
              {isGeneratingReport ? (
                <div className="flex items-center">
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  Generating Report...
                </div>
              ) : (
                <div className="flex items-center">
                  <FileText className="w-4 h-4 mr-2" />
                  Generate PDF Report
                </div>
              )}
            </Button>
          </div>
        ) : (
          <div className="text-center">
            <CheckCircle className="w-12 h-12 text-green-600 mx-auto mb-4" />
            <p className="text-green-800 mb-4 font-medium">
              Your detailed risk assessment report is ready!
            </p>
            
            <div className="flex justify-center space-x-4">
              <Button onClick={handleDownloadReport} className="bg-green-600 hover:bg-green-700">
                <Download className="w-4 h-4 mr-2" />
                Download Report
              </Button>
              
              <Button onClick={handleShareReport} variant="outline">
                <Share2 className="w-4 h-4 mr-2" />
                Share Report
              </Button>
            </div>
          </div>
        )}
      </div>

      {/* Expert Consultation */}
      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-6 mb-8">
        <h3 className="text-lg font-semibold text-yellow-900 mb-4">Need Expert Consultation?</h3>
        
        <p className="text-yellow-800 mb-4">
          Our legal and real estate experts are available to discuss your risk assessment results 
          and provide personalized guidance for your property investment decision.
        </p>
        
        <div className="flex flex-col sm:flex-row gap-4">
          <Button className="bg-yellow-600 hover:bg-yellow-700">
            <Phone className="w-4 h-4 mr-2" />
            Schedule Consultation
          </Button>
          
          <Button variant="outline" className="border-yellow-300 text-yellow-700 hover:bg-yellow-50">
            <Mail className="w-4 h-4 mr-2" />
            Email Expert
          </Button>
        </div>
      </div>

      {/* Navigation */}
      <div className="flex justify-between">
        <Button variant="outline" onClick={onBack}>
          Back to Risk Score
        </Button>
        
        <Button
          onClick={async () => {
            // Generate and download PDF automatically
            if (!reportUrl) {
              await handleGenerateReport();
            } else {
              // If report already generated, download it
              handleDownloadReport();
            }
            // Navigate to home page after a short delay
            setTimeout(() => {
              navigate('/');
            }, 1500);
          }}
          className="bg-[#0056D2] hover:bg-[#003d9a]"
        >
          Complete Assessment & Generate Report
        </Button>
      </div>
    </div>
  );
}