import React, { useState, useEffect } from 'react';
import { Target, AlertTriangle, TrendingDown, Shield, FileText, Users, Scale, Award } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { riskComplianceService, RiskScoreResponse } from '../../../api/riskCompliance';

interface RiskScoreProps {
  submissionId: string;
  onScoreCalculated: (result: RiskScoreResponse) => void;
  onNext: () => void;
  onBack: () => void;
}

export function RiskScore({ submissionId, onScoreCalculated, onNext, onBack }: RiskScoreProps) {
  const [isCalculating, setIsCalculating] = useState(false);
  const [scoreResult, setScoreResult] = useState<RiskScoreResponse | null>(null);
  const [error, setError] = useState<string>('');

  useEffect(() => {
    // Auto-calculate score when component mounts
    handleCalculateScore();
  }, []);

  const handleCalculateScore = async () => {
    setIsCalculating(true);
    setError('');

    try {
      const result = await riskComplianceService.calculateRiskScore(submissionId);

      // Simulate getting the score result
      const mockResult: RiskScoreResponse = {
        id: result.score_id,
        submission_id: submissionId,
        overall_score: 32, // Low score indicating high risk
        risk_level: 'critical',
        document_score: 60, // Some documents uploaded
        rera_score: 90, // RERA is compliant
        ownership_score: 85, // Ownership verified
        encumbrance_score: 10, // Major encumbrances found
        breakdown: {
          document_completeness: '5/5 documents uploaded',
          rera_compliance: 'Active registration',
          ownership_verification: 'Verified',
          encumbrance_status: 'Encumbrances found'
        },
        recommendations: [
          'CRITICAL: Clear all encumbrances before purchase',
          'Obtain loan clearance certificate from HDFC Bank',
          'Verify exact outstanding amount and foreclosure charges',
          'Consider escrow arrangement for loan clearance',
          'Get legal opinion on liability transfer risks',
          'Do not proceed without bank NOC'
        ],
        calculated_at: new Date().toISOString()
      };

      setScoreResult(mockResult);
      onScoreCalculated(mockResult);
    } catch (error) {
      console.error('Score calculation failed:', error);
      setError('Failed to calculate risk score. Please try again.');
    } finally {
      setIsCalculating(false);
    }
  };

  const getRiskColor = (level: string) => {
    switch (level) {
      case 'low': return 'text-green-600 bg-green-50 border-green-200';
      case 'medium': return 'text-yellow-600 bg-yellow-50 border-yellow-200';
      case 'high': return 'text-orange-600 bg-orange-50 border-orange-200';
      case 'critical': return 'text-red-600 bg-red-50 border-red-200';
      default: return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  const getScoreColor = (score: number) => {
    if (score >= 80) return 'text-green-600';
    if (score >= 60) return 'text-yellow-600';
    if (score >= 40) return 'text-orange-600';
    return 'text-red-600';
  };

  const getProgressColor = (score: number) => {
    if (score >= 80) return 'bg-green-500';
    if (score >= 60) return 'bg-yellow-500';
    if (score >= 40) return 'bg-orange-500';
    return 'bg-red-500';
  };

  if (isCalculating) {
    return (
      <div className="max-w-4xl mx-auto p-6">
        <div className="text-center py-12">
          <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-[#0056D2] mx-auto mb-4"></div>
          <h2 className="text-xl font-semibold text-gray-900 mb-2">Calculating Risk Score</h2>
          <p className="text-gray-600">Analyzing all verification results...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="max-w-4xl mx-auto p-6">
        <div className="bg-red-50 border border-red-200 rounded-lg p-6 text-center">
          <AlertTriangle className="w-12 h-12 text-red-600 mx-auto mb-4" />
          <h2 className="text-xl font-semibold text-red-900 mb-2">Calculation Failed</h2>
          <p className="text-red-700 mb-4">{error}</p>
          <Button onClick={handleCalculateScore} className="bg-red-600 hover:bg-red-700">
            Try Again
          </Button>
        </div>
      </div>
    );
  }

  if (!scoreResult) {
    return null;
  }

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center">
          <Target className="w-6 h-6 text-[#0056D2] mr-2" />
          Risk Assessment Score
        </h2>
        <p className="text-gray-600">
          Comprehensive risk analysis based on all verification checks.
        </p>
      </div>

      {/* Overall Risk Score */}
      <div className={`p-8 border-2 rounded-lg mb-8 ${getRiskColor(scoreResult.risk_level)}`}>
        <div className="text-center">
          <div className="flex items-center justify-center mb-4">
            <TrendingDown className="w-8 h-8 mr-3" />
            <h3 className="text-2xl font-bold">RISK ASSESSMENT</h3>
          </div>
          
          <div className="mb-6">
            <div className={`text-6xl font-bold mb-2 ${getScoreColor(scoreResult.overall_score)}`}>
              {scoreResult.overall_score}/100
            </div>
            <div className="text-xl font-semibold uppercase tracking-wide">
              {scoreResult.risk_level} RISK
            </div>
          </div>

          {scoreResult.risk_level === 'critical' && (
            <div className="bg-white p-4 rounded-lg border-2 border-red-300">
              <div className="flex items-center justify-center mb-2">
                <AlertTriangle className="w-6 h-6 text-red-600 mr-2" />
                <span className="text-lg font-bold text-red-900">⚠️ AVOID THIS PROPERTY</span>
              </div>
              <p className="text-sm text-red-800">
                This property has critical legal and financial risks. We strongly recommend avoiding this investment.
              </p>
            </div>
          )}
        </div>
      </div>

      {/* Score Breakdown */}
      <div className="bg-white border border-gray-200 rounded-lg p-6 mb-8">
        <h4 className="text-lg font-semibold text-gray-900 mb-6">Score Breakdown</h4>
        
        <div className="space-y-6">
          {/* Document Score */}
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <FileText className="w-5 h-5 text-gray-600 mr-3" />
              <div>
                <span className="font-medium text-gray-900">Document Completeness</span>
                <p className="text-sm text-gray-600">{scoreResult.breakdown.document_completeness}</p>
              </div>
            </div>
            <div className="flex items-center">
              <div className="w-32 h-2 bg-gray-200 rounded-full mr-3">
                <div 
                  className={`h-2 rounded-full ${getProgressColor(scoreResult.document_score)}`}
                  style={{ width: `${scoreResult.document_score}%` }}
                ></div>
              </div>
              <span className={`font-semibold ${getScoreColor(scoreResult.document_score)}`}>
                {scoreResult.document_score}/100
              </span>
            </div>
          </div>

          {/* RERA Score */}
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <Shield className="w-5 h-5 text-gray-600 mr-3" />
              <div>
                <span className="font-medium text-gray-900">RERA Compliance</span>
                <p className="text-sm text-gray-600">{scoreResult.breakdown.rera_compliance}</p>
              </div>
            </div>
            <div className="flex items-center">
              <div className="w-32 h-2 bg-gray-200 rounded-full mr-3">
                <div 
                  className={`h-2 rounded-full ${getProgressColor(scoreResult.rera_score)}`}
                  style={{ width: `${scoreResult.rera_score}%` }}
                ></div>
              </div>
              <span className={`font-semibold ${getScoreColor(scoreResult.rera_score)}`}>
                {scoreResult.rera_score}/100
              </span>
            </div>
          </div>

          {/* Ownership Score */}
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <Users className="w-5 h-5 text-gray-600 mr-3" />
              <div>
                <span className="font-medium text-gray-900">Ownership Verification</span>
                <p className="text-sm text-gray-600">{scoreResult.breakdown.ownership_verification}</p>
              </div>
            </div>
            <div className="flex items-center">
              <div className="w-32 h-2 bg-gray-200 rounded-full mr-3">
                <div 
                  className={`h-2 rounded-full ${getProgressColor(scoreResult.ownership_score)}`}
                  style={{ width: `${scoreResult.ownership_score}%` }}
                ></div>
              </div>
              <span className={`font-semibold ${getScoreColor(scoreResult.ownership_score)}`}>
                {scoreResult.ownership_score}/100
              </span>
            </div>
          </div>

          {/* Encumbrance Score */}
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <Scale className="w-5 h-5 text-gray-600 mr-3" />
              <div>
                <span className="font-medium text-gray-900">Encumbrance Check</span>
                <p className="text-sm text-gray-600">{scoreResult.breakdown.encumbrance_status}</p>
              </div>
            </div>
            <div className="flex items-center">
              <div className="w-32 h-2 bg-gray-200 rounded-full mr-3">
                <div 
                  className={`h-2 rounded-full ${getProgressColor(scoreResult.encumbrance_score)}`}
                  style={{ width: `${scoreResult.encumbrance_score}%` }}
                ></div>
              </div>
              <span className={`font-semibold ${getScoreColor(scoreResult.encumbrance_score)}`}>
                {scoreResult.encumbrance_score}/100
              </span>
            </div>
          </div>
        </div>
      </div>

      {/* Risk Factors */}
      <div className="bg-red-50 border border-red-200 rounded-lg p-6 mb-8">
        <h4 className="text-lg font-semibold text-red-900 mb-4 flex items-center">
          <AlertTriangle className="w-5 h-5 mr-2" />
          Critical Risk Factors
        </h4>
        
        <div className="space-y-3">
          <div className="flex items-start p-3 bg-white rounded border border-red-200">
            <div className="w-2 h-2 bg-red-500 rounded-full mt-2 mr-3 flex-shrink-0"></div>
            <div>
              <span className="font-medium text-red-900">Bank Loan Encumbrance</span>
              <p className="text-sm text-red-700">₹2.3 Crores outstanding loan with HDFC Bank</p>
            </div>
          </div>
          
          <div className="flex items-start p-3 bg-white rounded border border-red-200">
            <div className="w-2 h-2 bg-red-500 rounded-full mt-2 mr-3 flex-shrink-0"></div>
            <div>
              <span className="font-medium text-red-900">NOC Required</span>
              <p className="text-sm text-red-700">Bank clearance certificate mandatory before purchase</p>
            </div>
          </div>
          
          <div className="flex items-start p-3 bg-white rounded border border-red-200">
            <div className="w-2 h-2 bg-red-500 rounded-full mt-2 mr-3 flex-shrink-0"></div>
            <div>
              <span className="font-medium text-red-900">Financial Liability Risk</span>
              <p className="text-sm text-red-700">Potential liability transfer to buyer if not cleared properly</p>
            </div>
          </div>
        </div>
      </div>

      {/* Recommendations */}
      <div className="bg-white border border-gray-200 rounded-lg p-6 mb-8">
        <h4 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
          <Award className="w-5 h-5 text-[#0056D2] mr-2" />
          Expert Recommendations
        </h4>
        
        <div className="space-y-3">
          {scoreResult.recommendations.map((recommendation, index) => (
            <div key={index} className="flex items-start">
              <div className="w-6 h-6 bg-[#0056D2] text-white rounded-full flex items-center justify-center text-xs font-bold mr-3 mt-0.5 flex-shrink-0">
                {index + 1}
              </div>
              <p className="text-sm text-gray-700">{recommendation}</p>
            </div>
          ))}
        </div>
      </div>

      {/* Final Recommendation */}
      <div className="bg-red-50 border-2 border-red-300 rounded-lg p-6 mb-8">
        <div className="text-center">
          <AlertTriangle className="w-12 h-12 text-red-600 mx-auto mb-4" />
          <h3 className="text-xl font-bold text-red-900 mb-2">INVESTMENT RECOMMENDATION</h3>
          <div className="text-lg font-semibold text-red-800 mb-3">
            ❌ AVOID THIS PROPERTY
          </div>
          <p className="text-red-700 max-w-2xl mx-auto">
            Based on our comprehensive risk analysis, this property carries critical financial risks 
            due to outstanding bank encumbrances. We strongly recommend avoiding this investment 
            or ensuring complete loan clearance before proceeding.
          </p>
        </div>
      </div>

      {/* Navigation */}
      <div className="flex justify-between">
        <Button variant="outline" onClick={onBack}>
          Back to Encumbrance Check
        </Button>
        
        <Button
          onClick={onNext}
          className="bg-[#0056D2] hover:bg-[#003d9a]"
        >
          Generate Final Report
        </Button>
      </div>
    </div>
  );
}