import React, { useState, useEffect } from 'react';
import { Link2, AlertTriangle, CheckCircle, Clock, ArrowRight } from 'lucide-react';
import { Button } from '../../../components/ui/button';
import { riskComplianceService } from '../../../api/riskCompliance';

interface TitleChainAnalysisProps {
  submissionId: string;
  onNext: () => void;
  onBack: () => void;
}

interface AnalysisResult {
  success: boolean;
  integrity_score: number;
  total_transfers: number;
  verified_transfers: number;
  gaps_found: number;
  risk_level: string;
  gaps: Array<{
    type: string;
    description: string;
    severity: string;
  }>;
  recommendations: string[];
}

export function TitleChainAnalysis({ submissionId, onNext, onBack }: TitleChainAnalysisProps) {
  const [analysis, setAnalysis] = useState<AnalysisResult | null>(null);
  const [loading, setLoading] = useState(false);
  const [analyzing, setAnalyzing] = useState(false);
  const [error, setError] = useState('');

  const handleAnalyze = async () => {
    setAnalyzing(true);
    setError('');
    
    try {
      const result = await riskComplianceService.analyzeTitleChain(submissionId);
      
      if (result.success) {
        setAnalysis(result.data);
      } else {
        setError('Analysis failed: ' + (result.data?.error || 'Unknown error'));
      }
    } catch (err: any) {
      console.error('Title chain analysis failed:', err);
      setError('Failed to analyze title chain: ' + (err.message || 'Network error'));
    } finally {
      setAnalyzing(false);
    }
  };

  const getRiskColor = (riskLevel: string) => {
    switch (riskLevel) {
      case 'low': return 'text-green-600 bg-green-100';
      case 'medium': return 'text-orange-600 bg-orange-100';
      case 'high': return 'text-red-600 bg-red-100';
      default: return 'text-gray-600 bg-gray-100';
    }
  };

  const getScoreColor = (score: number) => {
    if (score >= 80) return 'text-green-600';
    if (score >= 60) return 'text-orange-600';
    return 'text-red-600';
  };

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h2 className="text-2xl font-bold text-gray-900 mb-4 flex items-center">
          <Link2 className="w-6 h-6 text-[#0056D2] mr-2" />
          Title Chain Analysis
        </h2>
        <p className="text-gray-600">
          Comprehensive analysis of property ownership chain integrity and legal continuity.
        </p>
      </div>

      {!analysis && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 mb-6">
          <h3 className="text-lg font-semibold text-blue-900 mb-3">
            Ready for Title Chain Analysis
          </h3>
          <p className="text-blue-700 mb-4">
            Analyze the complete ownership chain to identify gaps, verify legal continuity, and assess title integrity.
          </p>
          <Button
            onClick={handleAnalyze}
            disabled={analyzing}
            className="bg-blue-600 hover:bg-blue-700"
          >
            {analyzing ? (
              <div className="flex items-center">
                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                Analyzing Chain...
              </div>
            ) : (
              'Start Title Chain Analysis'
            )}
          </Button>
        </div>
      )}

      {error && (
        <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
          <div className="flex items-center">
            <AlertTriangle className="w-5 h-5 text-red-600 mr-2" />
            <p className="text-red-800">{error}</p>
          </div>
        </div>
      )}

      {analysis && (
        <div className="space-y-6">
          {/* Analysis Results */}
          <div className="bg-white border border-gray-200 rounded-lg p-6">
            <h3 className="text-lg font-semibold mb-4">Chain Integrity Assessment</h3>
            
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
              <div className="text-center p-4 bg-gray-50 rounded-lg">
                <div className={`text-2xl font-bold ${getScoreColor(analysis.integrity_score)}`}>
                  {analysis.integrity_score}%
                </div>
                <div className="text-sm text-gray-600">Integrity Score</div>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg">
                <div className="text-2xl font-bold text-blue-600">{analysis.total_transfers}</div>
                <div className="text-sm text-gray-600">Total Transfers</div>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg">
                <div className="text-2xl font-bold text-green-600">{analysis.verified_transfers}</div>
                <div className="text-sm text-gray-600">Verified</div>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg">
                <div className={`text-2xl font-bold px-3 py-1 rounded ${getRiskColor(analysis.risk_level)}`}>
                  {analysis.risk_level.toUpperCase()}
                </div>
                <div className="text-sm text-gray-600">Risk Level</div>
              </div>
            </div>

            {/* Chain Visualization */}
            <div className="mb-6">
              <h4 className="font-semibold text-gray-900 mb-3">Ownership Chain</h4>
              <div className="bg-gray-50 p-4 rounded-lg">
                <div className="flex items-center justify-between text-sm">
                  <span className="font-medium">ABC Builders (2008)</span>
                  <ArrowRight className="w-4 h-4 text-gray-400" />
                  <span className="font-medium">Mohan Sharma (2019)</span>
                  <ArrowRight className="w-4 h-4 text-gray-400" />
                  <span className="font-medium">Amit Sharma (2022)</span>
                  <ArrowRight className="w-4 h-4 text-gray-400" />
                  <span className="font-medium text-green-600">Rajesh Kumar (2024) ✓</span>
                </div>
              </div>
            </div>

            {/* Gaps Found */}
            {analysis.gaps_found > 0 && (
              <div className="mb-6">
                <h4 className="font-semibold text-gray-900 mb-3">
                  Gaps Detected ({analysis.gaps_found})
                </h4>
                <div className="space-y-2">
                  {analysis.gaps.map((gap, index) => (
                    <div key={index} className="flex items-start p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                      <AlertTriangle className="w-5 h-5 text-yellow-600 mr-2 mt-0.5" />
                      <div>
                        <div className="font-medium text-yellow-800">{gap.type.replace('_', ' ').toUpperCase()}</div>
                        <div className="text-yellow-700 text-sm">{gap.description}</div>
                        <div className={`text-xs px-2 py-1 rounded mt-1 inline-block ${
                          gap.severity === 'high' ? 'bg-red-100 text-red-800' :
                          gap.severity === 'medium' ? 'bg-orange-100 text-orange-800' :
                          'bg-yellow-100 text-yellow-800'
                        }`}>
                          {gap.severity.toUpperCase()} SEVERITY
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Recommendations */}
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
              <h4 className="font-semibold text-blue-800 mb-2">Recommendations</h4>
              <ul className="space-y-1">
                {analysis.recommendations.map((rec, index) => (
                  <li key={index} className="text-blue-700 text-sm flex items-start">
                    <CheckCircle className="w-4 h-4 text-blue-600 mr-2 mt-0.5 flex-shrink-0" />
                    {rec}
                  </li>
                ))}
              </ul>
            </div>
          </div>
        </div>
      )}

      {/* Navigation */}
      <div className="flex justify-between mt-8">
        <Button variant="outline" onClick={onBack}>
          Back to Ownership Check
        </Button>
        <Button
          onClick={onNext}
          disabled={!analysis}
          className="bg-[#0056D2] hover:bg-[#003d9a] disabled:opacity-50"
        >
          Continue to Encumbrance Check
        </Button>
      </div>
    </div>
  );
}