import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { FileText, Clock, CheckCircle2, AlertTriangle, XCircle, Eye, Download, Plus, Filter } from 'lucide-react';

interface Report {
  id: string;
  reportNumber: string;
  propertyAddress: string;
  propertyType: string;
  status: 'initiated' | 'in_progress' | 'completed' | 'failed';
  overallScore: number | null;
  riskLevel: 'low' | 'medium' | 'high' | 'critical' | null;
  createdAt: string;
  completedAt: string | null;
  progress: number;
}

const MyLegalHealthReports: React.FC = () => {
  const navigate = useNavigate();
  const [filterStatus, setFilterStatus] = useState<string>('all');

  // Mock data - will be replaced with API call
  const [reports] = useState<Report[]>([
    {
      id: '1',
      reportNumber: 'LHR-2024-001',
      propertyAddress: 'MG Road, Bangalore, Karnataka',
      propertyType: 'Villa',
      status: 'completed',
      overallScore: 85,
      riskLevel: 'low',
      createdAt: '2024-01-15',
      completedAt: '2024-01-25',
      progress: 100
    },
    {
      id: '2',
      reportNumber: 'LHR-2024-002',
      propertyAddress: 'HSR Layout, Bangalore, Karnataka',
      propertyType: 'Apartment',
      status: 'in_progress',
      overallScore: null,
      riskLevel: null,
      createdAt: '2024-02-01',
      completedAt: null,
      progress: 60
    },
    {
      id: '3',
      reportNumber: 'LHR-2024-003',
      propertyAddress: 'Whitefield, Bangalore, Karnataka',
      propertyType: 'Plot',
      status: 'initiated',
      overallScore: null,
      riskLevel: null,
      createdAt: '2024-02-06',
      completedAt: null,
      progress: 10
    }
  ]);

  const getStatusConfig = (status: string) => {
    const configs = {
      initiated: {
        label: 'Initiated',
        icon: <Clock className="w-4 h-4" />,
        color: 'bg-gray-100 text-gray-700 border-gray-300'
      },
      in_progress: {
        label: 'In Progress',
        icon: <Clock className="w-4 h-4" />,
        color: 'bg-blue-100 text-blue-700 border-blue-300'
      },
      completed: {
        label: 'Completed',
        icon: <CheckCircle2 className="w-4 h-4" />,
        color: 'bg-green-100 text-green-700 border-green-300'
      },
      failed: {
        label: 'Failed',
        icon: <XCircle className="w-4 h-4" />,
        color: 'bg-red-100 text-red-700 border-red-300'
      }
    };
    return configs[status as keyof typeof configs] || configs.initiated;
  };

  const getRiskConfig = (risk: string | null) => {
    if (!risk) return null;
    const configs = {
      low: { label: 'Low Risk', color: 'bg-green-100 text-green-700' },
      medium: { label: 'Medium Risk', color: 'bg-yellow-100 text-yellow-700' },
      high: { label: 'High Risk', color: 'bg-orange-100 text-orange-700' },
      critical: { label: 'Critical Risk', color: 'bg-red-100 text-red-700' }
    };
    return configs[risk as keyof typeof configs];
  };

  const filteredReports = filterStatus === 'all' 
    ? reports 
    : reports.filter(r => r.status === filterStatus);

  const stats = {
    total: reports.length,
    completed: reports.filter(r => r.status === 'completed').length,
    inProgress: reports.filter(r => r.status === 'in_progress').length,
    initiated: reports.filter(r => r.status === 'initiated').length
  };

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="container mx-auto px-4 max-w-7xl">
        <div className="flex items-center justify-between mb-8">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">My Legal Health Reports</h1>
            <p className="text-gray-600 mt-2">Track and manage your property verification reports</p>
          </div>
          <button
            onClick={() => navigate('/buyer/legal-health/create-report')}
            className="px-6 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 transition-colors flex items-center gap-2"
          >
            <Plus className="w-5 h-5" />
            Order New Report
          </button>
        </div>

        {/* Stats Cards */}
        <div className="grid md:grid-cols-4 gap-6 mb-8">
          <div className="bg-white rounded-lg shadow-md p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Total Reports</p>
                <p className="text-3xl font-bold text-gray-900 mt-1">{stats.total}</p>
              </div>
              <FileText className="w-10 h-10 text-blue-600" />
            </div>
          </div>
          <div className="bg-white rounded-lg shadow-md p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Completed</p>
                <p className="text-3xl font-bold text-green-600 mt-1">{stats.completed}</p>
              </div>
              <CheckCircle2 className="w-10 h-10 text-green-600" />
            </div>
          </div>
          <div className="bg-white rounded-lg shadow-md p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">In Progress</p>
                <p className="text-3xl font-bold text-blue-600 mt-1">{stats.inProgress}</p>
              </div>
              <Clock className="w-10 h-10 text-blue-600" />
            </div>
          </div>
          <div className="bg-white rounded-lg shadow-md p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Initiated</p>
                <p className="text-3xl font-bold text-gray-600 mt-1">{stats.initiated}</p>
              </div>
              <Clock className="w-10 h-10 text-gray-600" />
            </div>
          </div>
        </div>

        {/* Filter */}
        <div className="bg-white rounded-lg shadow-md p-4 mb-6">
          <div className="flex items-center gap-4">
            <Filter className="w-5 h-5 text-gray-600" />
            <span className="font-medium text-gray-700">Filter:</span>
            {['all', 'initiated', 'in_progress', 'completed'].map((status) => (
              <button
                key={status}
                onClick={() => setFilterStatus(status)}
                className={`px-4 py-2 rounded-lg font-medium transition-colors ${
                  filterStatus === status
                    ? 'bg-blue-600 text-white'
                    : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                }`}
              >
                {status === 'all' ? 'All' : status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
              </button>
            ))}
          </div>
        </div>

        {/* Reports List */}
        <div className="space-y-4">
          {filteredReports.length === 0 ? (
            <div className="bg-white rounded-lg shadow-md p-12 text-center">
              <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
              <h3 className="text-xl font-semibold text-gray-900 mb-2">No reports found</h3>
              <p className="text-gray-600 mb-6">Start by ordering your first legal health report</p>
              <button
                onClick={() => navigate('/buyer/legal-health/create-report')}
                className="px-6 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 transition-colors inline-flex items-center gap-2"
              >
                <Plus className="w-5 h-5" />
                Order Report
              </button>
            </div>
          ) : (
            filteredReports.map((report) => {
              const statusConfig = getStatusConfig(report.status);
              const riskConfig = getRiskConfig(report.riskLevel);

              return (
                <div key={report.id} className="bg-white rounded-lg shadow-md p-6 hover:shadow-lg transition-shadow">
                  <div className="flex items-start justify-between mb-4">
                    <div className="flex-1">
                      <div className="flex items-center gap-3 mb-2">
                        <h3 className="text-xl font-bold text-gray-900">{report.reportNumber}</h3>
                        <span className={`px-3 py-1 rounded-full text-sm font-medium border flex items-center gap-1 ${statusConfig.color}`}>
                          {statusConfig.icon}
                          {statusConfig.label}
                        </span>
                        {riskConfig && (
                          <span className={`px-3 py-1 rounded-full text-sm font-medium ${riskConfig.color}`}>
                            {riskConfig.label}
                          </span>
                        )}
                      </div>
                      <p className="text-gray-600 mb-1">{report.propertyAddress}</p>
                      <p className="text-sm text-gray-500">Property Type: {report.propertyType}</p>
                    </div>
                    {report.overallScore !== null && (
                      <div className="text-right">
                        <p className="text-sm text-gray-600">Overall Score</p>
                        <p className="text-4xl font-bold text-blue-600">{report.overallScore}/100</p>
                      </div>
                    )}
                  </div>

                  {report.status === 'in_progress' && (
                    <div className="mb-4">
                      <div className="flex items-center justify-between text-sm text-gray-600 mb-2">
                        <span>Verification Progress</span>
                        <span className="font-semibold">{report.progress}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div
                          className="bg-blue-600 h-2 rounded-full transition-all"
                          style={{ width: `${report.progress}%` }}
                        />
                      </div>
                    </div>
                  )}

                  <div className="flex items-center justify-between pt-4 border-t">
                    <div className="text-sm text-gray-600">
                      <span>Created: {new Date(report.createdAt).toLocaleDateString()}</span>
                      {report.completedAt && (
                        <span className="ml-4">Completed: {new Date(report.completedAt).toLocaleDateString()}</span>
                      )}
                    </div>
                    <div className="flex items-center gap-3">
                      <button
                        onClick={() => navigate(`/buyer/legal-health/report/${report.id}`)}
                        className="px-4 py-2 bg-blue-600 text-white rounded-lg font-medium hover:bg-blue-700 transition-colors flex items-center gap-2"
                      >
                        <Eye className="w-4 h-4" />
                        View Details
                      </button>
                      {report.status === 'completed' && (
                        <button className="px-4 py-2 bg-green-600 text-white rounded-lg font-medium hover:bg-green-700 transition-colors flex items-center gap-2">
                          <Download className="w-4 h-4" />
                          Download PDF
                        </button>
                      )}
                    </div>
                  </div>
                </div>
              );
            })
          )}
        </div>
      </div>
    </div>
  );
};

export default MyLegalHealthReports;
