import React, { useState } from 'react';
import { useNavigate, useParams } from 'react-router-dom';
import { ArrowLeft, FileCheck, Scale, Building2, AlertTriangle, CheckCircle2, Clock, Download, Upload, XCircle } from 'lucide-react';

interface VerificationModule {
  id: string;
  moduleType: string;
  title: string;
  status: 'pending' | 'in_progress' | 'completed' | 'failed';
  score: number | null;
  findings: string[];
  issues: string[];
  recommendations: string[];
}

const ReportDetails: React.FC = () => {
  const navigate = useNavigate();
  const { id } = useParams();

  // Mock data
  const [report] = useState({
    id: id,
    reportNumber: 'LHR-2024-002',
    propertyAddress: 'HSR Layout, Bangalore, Karnataka',
    propertyType: 'Apartment',
    status: 'in_progress',
    overallScore: null,
    riskLevel: null,
    createdAt: '2024-02-01',
    completedAt: null,
    progress: 60,
    recommendation: null
  });

  const [modules] = useState<VerificationModule[]>([
    {
      id: '1',
      moduleType: 'ownership',
      title: 'Ownership Verification',
      status: 'completed',
      score: 95,
      findings: ['Owner identity verified', 'Title deed authentic', 'No third-party claims'],
      issues: [],
      recommendations: ['All ownership documents are in order']
    },
    {
      id: '2',
      moduleType: 'documents',
      title: 'Document Validation',
      status: 'completed',
      score: 90,
      findings: ['Sale deed verified', 'Tax receipts up to date', 'Khata certificate valid'],
      issues: ['Encumbrance certificate pending from seller'],
      recommendations: ['Request encumbrance certificate from seller']
    },
    {
      id: '3',
      moduleType: 'litigation',
      title: 'Litigation Check',
      status: 'in_progress',
      score: null,
      findings: ['High Court search completed', 'No Supreme Court cases found'],
      issues: [],
      recommendations: []
    },
    {
      id: '4',
      moduleType: 'encumbrance',
      title: 'Financial Check',
      status: 'pending',
      score: null,
      findings: [],
      issues: [],
      recommendations: []
    },
    {
      id: '5',
      moduleType: 'zoning',
      title: 'Zoning Compliance',
      status: 'pending',
      score: null,
      findings: [],
      issues: [],
      recommendations: []
    },
    {
      id: '6',
      moduleType: 'project_approval',
      title: 'Project Approvals',
      status: 'pending',
      score: null,
      findings: [],
      issues: [],
      recommendations: []
    }
  ]);

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle2 className="w-6 h-6 text-green-600" />;
      case 'in_progress':
        return <Clock className="w-6 h-6 text-blue-600" />;
      case 'failed':
        return <XCircle className="w-6 h-6 text-red-600" />;
      default:
        return <Clock className="w-6 h-6 text-gray-400" />;
    }
  };

  const getModuleIcon = (type: string) => {
    const icons: Record<string, JSX.Element> = {
      ownership: <FileCheck className="w-6 h-6" />,
      documents: <FileCheck className="w-6 h-6" />,
      litigation: <Scale className="w-6 h-6" />,
      encumbrance: <AlertTriangle className="w-6 h-6" />,
      zoning: <Building2 className="w-6 h-6" />,
      project_approval: <CheckCircle2 className="w-6 h-6" />
    };
    return icons[type] || <FileCheck className="w-6 h-6" />;
  };

  const completedModules = modules.filter(m => m.status === 'completed').length;
  const progressPercentage = (completedModules / modules.length) * 100;

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="container mx-auto px-4 max-w-6xl">
        <button
          onClick={() => navigate('/buyer/legal-health/my-reports')}
          className="flex items-center gap-2 text-gray-600 hover:text-gray-900 mb-6"
        >
          <ArrowLeft className="w-5 h-5" />
          Back to My Reports
        </button>

        {/* Report Header */}
        <div className="bg-white rounded-lg shadow-md p-8 mb-6">
          <div className="flex items-start justify-between mb-6">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 mb-2">{report.reportNumber}</h1>
              <p className="text-lg text-gray-600 mb-1">{report.propertyAddress}</p>
              <p className="text-sm text-gray-500">Property Type: {report.propertyType}</p>
            </div>
            <div className="text-right">
              <span className="inline-block px-4 py-2 bg-blue-100 text-blue-700 rounded-full font-semibold mb-2">
                In Progress
              </span>
              <p className="text-sm text-gray-600">Created: {new Date(report.createdAt).toLocaleDateString()}</p>
            </div>
          </div>

          {/* Overall Progress */}
          <div className="bg-gray-50 rounded-lg p-6">
            <div className="flex items-center justify-between mb-3">
              <h3 className="font-semibold text-gray-900">Overall Verification Progress</h3>
              <span className="text-2xl font-bold text-blue-600">{completedModules}/6 Steps</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-3 mb-2">
              <div
                className="bg-blue-600 h-3 rounded-full transition-all"
                style={{ width: `${progressPercentage}%` }}
              />
            </div>
            <p className="text-sm text-gray-600">Estimated completion: 5 more days</p>
          </div>
        </div>

        {/* Action Buttons */}
        <div className="flex items-center gap-4 mb-6">
          <button
            onClick={() => navigate(`/buyer/legal-health/upload-documents/${id}`)}
            className="px-6 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 transition-colors flex items-center gap-2"
          >
            <Upload className="w-5 h-5" />
            Upload Documents
          </button>
          {report.status === 'completed' && (
            <>
              <button className="px-6 py-3 bg-green-600 text-white rounded-lg font-semibold hover:bg-green-700 transition-colors flex items-center gap-2">
                <Download className="w-5 h-5" />
                Download Report PDF
              </button>
              <button className="px-6 py-3 bg-purple-600 text-white rounded-lg font-semibold hover:bg-purple-700 transition-colors flex items-center gap-2">
                <Download className="w-5 h-5" />
                Download Certificate
              </button>
            </>
          )}
        </div>

        {/* Verification Modules */}
        <div className="space-y-4">
          <h2 className="text-2xl font-bold text-gray-900 mb-4">Verification Modules</h2>
          {modules.map((module, index) => (
            <div key={module.id} className="bg-white rounded-lg shadow-md overflow-hidden">
              <div className="p-6">
                <div className="flex items-start justify-between mb-4">
                  <div className="flex items-start gap-4 flex-1">
                    <div className="flex-shrink-0 w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center text-blue-600">
                      {getModuleIcon(module.moduleType)}
                    </div>
                    <div className="flex-1">
                      <div className="flex items-center gap-3 mb-2">
                        <h3 className="text-xl font-bold text-gray-900">
                          Step {index + 1}: {module.title}
                        </h3>
                        {getStatusIcon(module.status)}
                      </div>
                      <p className="text-sm text-gray-600 capitalize">{module.status.replace('_', ' ')}</p>
                    </div>
                  </div>
                  {module.score !== null && (
                    <div className="text-right">
                      <p className="text-sm text-gray-600">Score</p>
                      <p className="text-3xl font-bold text-blue-600">{module.score}/100</p>
                    </div>
                  )}
                </div>

                {module.status !== 'pending' && (
                  <div className="space-y-4 mt-6">
                    {module.findings.length > 0 && (
                      <div>
                        <h4 className="font-semibold text-gray-900 mb-2 flex items-center gap-2">
                          <CheckCircle2 className="w-5 h-5 text-green-600" />
                          Findings
                        </h4>
                        <ul className="space-y-1 ml-7">
                          {module.findings.map((finding, idx) => (
                            <li key={idx} className="text-gray-700">• {finding}</li>
                          ))}
                        </ul>
                      </div>
                    )}

                    {module.issues.length > 0 && (
                      <div>
                        <h4 className="font-semibold text-gray-900 mb-2 flex items-center gap-2">
                          <AlertTriangle className="w-5 h-5 text-orange-600" />
                          Issues Found
                        </h4>
                        <ul className="space-y-1 ml-7">
                          {module.issues.map((issue, idx) => (
                            <li key={idx} className="text-orange-700">• {issue}</li>
                          ))}
                        </ul>
                      </div>
                    )}

                    {module.recommendations.length > 0 && (
                      <div>
                        <h4 className="font-semibold text-gray-900 mb-2 flex items-center gap-2">
                          <FileCheck className="w-5 h-5 text-blue-600" />
                          Recommendations
                        </h4>
                        <ul className="space-y-1 ml-7">
                          {module.recommendations.map((rec, idx) => (
                            <li key={idx} className="text-blue-700">• {rec}</li>
                          ))}
                        </ul>
                      </div>
                    )}
                  </div>
                )}
              </div>
            </div>
          ))}
        </div>

        {/* Timeline */}
        <div className="bg-white rounded-lg shadow-md p-6 mt-6">
          <h2 className="text-xl font-bold text-gray-900 mb-4">Activity Timeline</h2>
          <div className="space-y-4">
            {[
              { date: '2024-02-06', event: 'Document Validation completed', type: 'success' },
              { date: '2024-02-04', event: 'Ownership Verification completed', type: 'success' },
              { date: '2024-02-02', event: 'Documents uploaded by buyer', type: 'info' },
              { date: '2024-02-01', event: 'Report created', type: 'info' }
            ].map((item, idx) => (
              <div key={idx} className="flex items-start gap-4">
                <div className={`w-3 h-3 rounded-full mt-1.5 ${
                  item.type === 'success' ? 'bg-green-600' : 'bg-blue-600'
                }`} />
                <div className="flex-1">
                  <p className="font-medium text-gray-900">{item.event}</p>
                  <p className="text-sm text-gray-600">{new Date(item.date).toLocaleDateString()}</p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
};

export default ReportDetails;
