import React, { useState } from 'react';
import { useNavigate, useParams } from 'react-router-dom';
import { ArrowLeft, Upload, FileText, CheckCircle2, X, AlertCircle } from 'lucide-react';

interface Document {
  id: string;
  name: string;
  type: string;
  size: number;
  uploadedAt: string;
  status: 'uploaded' | 'verified' | 'rejected';
}

const UploadDocuments: React.FC = () => {
  const navigate = useNavigate();
  const { id } = useParams();
  const [dragActive, setDragActive] = useState(false);
  const [documents, setDocuments] = useState<Document[]>([]);

  const requiredDocuments = [
    { type: 'sale_deed', label: 'Sale Deed', required: true },
    { type: 'title_deed', label: 'Title Deed', required: true },
    { type: 'tax_receipts', label: 'Property Tax Receipts', required: true },
    { type: 'khata_certificate', label: 'Khata Certificate', required: true },
    { type: 'encumbrance_certificate', label: 'Encumbrance Certificate (30 years)', required: true },
    { type: 'noc', label: 'NOC from Society/Builder', required: false },
    { type: 'rera_certificate', label: 'RERA Certificate', required: false },
    { type: 'occupancy_certificate', label: 'Occupancy Certificate', required: false }
  ];

  const handleDrag = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (e.type === 'dragenter' || e.type === 'dragover') {
      setDragActive(true);
    } else if (e.type === 'dragleave') {
      setDragActive(false);
    }
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setDragActive(false);
    
    if (e.dataTransfer.files && e.dataTransfer.files[0]) {
      handleFiles(e.dataTransfer.files);
    }
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    e.preventDefault();
    if (e.target.files && e.target.files[0]) {
      handleFiles(e.target.files);
    }
  };

  const handleFiles = (files: FileList) => {
    const newDocs: Document[] = Array.from(files).map(file => ({
      id: Math.random().toString(36).substr(2, 9),
      name: file.name,
      type: file.type,
      size: file.size,
      uploadedAt: new Date().toISOString(),
      status: 'uploaded'
    }));
    setDocuments([...documents, ...newDocs]);
  };

  const removeDocument = (docId: string) => {
    setDocuments(documents.filter(doc => doc.id !== docId));
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
  };

  const getStatusBadge = (status: string) => {
    const configs = {
      uploaded: { label: 'Uploaded', color: 'bg-blue-100 text-blue-700' },
      verified: { label: 'Verified', color: 'bg-green-100 text-green-700' },
      rejected: { label: 'Rejected', color: 'bg-red-100 text-red-700' }
    };
    return configs[status as keyof typeof configs] || configs.uploaded;
  };

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="container mx-auto px-4 max-w-5xl">
        <button
          onClick={() => navigate(`/buyer/legal-health/report/${id}`)}
          className="flex items-center gap-2 text-gray-600 hover:text-gray-900 mb-6"
        >
          <ArrowLeft className="w-5 h-5" />
          Back to Report Details
        </button>

        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900 mb-2">Upload Property Documents</h1>
          <p className="text-gray-600">Upload all required documents for legal verification</p>
        </div>

        {/* Required Documents Checklist */}
        <div className="bg-white rounded-lg shadow-md p-6 mb-6">
          <h2 className="text-xl font-bold text-gray-900 mb-4">Required Documents</h2>
          <div className="space-y-3">
            {requiredDocuments.map((doc) => {
              const uploaded = documents.some(d => d.name.toLowerCase().includes(doc.type));
              return (
                <div key={doc.type} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                  <div className="flex items-center gap-3">
                    {uploaded ? (
                      <CheckCircle2 className="w-5 h-5 text-green-600" />
                    ) : (
                      <div className="w-5 h-5 rounded-full border-2 border-gray-300" />
                    )}
                    <div>
                      <p className="font-medium text-gray-900">{doc.label}</p>
                      {doc.required && (
                        <p className="text-xs text-red-600">Required</p>
                      )}
                    </div>
                  </div>
                  {uploaded && (
                    <span className="text-sm text-green-600 font-medium">Uploaded</span>
                  )}
                </div>
              );
            })}
          </div>
        </div>

        {/* Upload Area */}
        <div className="bg-white rounded-lg shadow-md p-6 mb-6">
          <h2 className="text-xl font-bold text-gray-900 mb-4">Upload Files</h2>
          
          <div
            className={`border-2 border-dashed rounded-lg p-12 text-center transition-colors ${
              dragActive ? 'border-blue-600 bg-blue-50' : 'border-gray-300 hover:border-gray-400'
            }`}
            onDragEnter={handleDrag}
            onDragLeave={handleDrag}
            onDragOver={handleDrag}
            onDrop={handleDrop}
          >
            <Upload className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <p className="text-lg font-medium text-gray-900 mb-2">
              Drag and drop files here
            </p>
            <p className="text-gray-600 mb-4">or</p>
            <label className="inline-block px-6 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 cursor-pointer transition-colors">
              Browse Files
              <input
                type="file"
                multiple
                onChange={handleChange}
                className="hidden"
                accept=".pdf,.jpg,.jpeg,.png"
              />
            </label>
            <p className="text-sm text-gray-500 mt-4">
              Supported formats: PDF, JPG, PNG (Max 10MB per file)
            </p>
          </div>

          <div className="mt-4 bg-blue-50 border-l-4 border-blue-500 p-4 rounded">
            <div className="flex items-start gap-3">
              <AlertCircle className="w-5 h-5 text-blue-600 flex-shrink-0 mt-0.5" />
              <div className="text-sm text-blue-800">
                <p className="font-semibold mb-1">Important Tips:</p>
                <ul className="space-y-1">
                  <li>• Ensure all documents are clear and readable</li>
                  <li>• Upload original documents or certified copies</li>
                  <li>• File names should indicate document type</li>
                  <li>• All pages of multi-page documents must be included</li>
                </ul>
              </div>
            </div>
          </div>
        </div>

        {/* Uploaded Documents List */}
        {documents.length > 0 && (
          <div className="bg-white rounded-lg shadow-md p-6">
            <h2 className="text-xl font-bold text-gray-900 mb-4">Uploaded Documents ({documents.length})</h2>
            <div className="space-y-3">
              {documents.map((doc) => {
                const statusBadge = getStatusBadge(doc.status);
                return (
                  <div key={doc.id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                    <div className="flex items-center gap-4 flex-1">
                      <FileText className="w-8 h-8 text-blue-600" />
                      <div className="flex-1">
                        <p className="font-medium text-gray-900">{doc.name}</p>
                        <p className="text-sm text-gray-600">
                          {formatFileSize(doc.size)} • Uploaded {new Date(doc.uploadedAt).toLocaleString()}
                        </p>
                      </div>
                      <span className={`px-3 py-1 rounded-full text-sm font-medium ${statusBadge.color}`}>
                        {statusBadge.label}
                      </span>
                    </div>
                    <button
                      onClick={() => removeDocument(doc.id)}
                      className="ml-4 p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
                    >
                      <X className="w-5 h-5" />
                    </button>
                  </div>
                );
              })}
            </div>

            <div className="mt-6 pt-6 border-t flex items-center justify-between">
              <p className="text-sm text-gray-600">
                {documents.length} document(s) uploaded
              </p>
              <button
                onClick={() => navigate(`/buyer/legal-health/report/${id}`)}
                className="px-6 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 transition-colors"
              >
                Done
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default UploadDocuments;
