import React, { useState } from 'react';
import { Search, Filter, Bookmark } from 'lucide-react';
import { BuyerInfraMap } from '../../components/future-insights/BuyerInfraMap';
import { ForecastChart } from '../../components/future-insights/ForecastChart';
import { HotspotPanel } from '../../components/future-insights/HotspotPanel';

export function BuyerInsightsPage() {
  const [filters, setFilters] = useState({
    city: 'Mumbai',
    locality: '',
    propertyType: 'apartment'
  });

  const handleFilterChange = (key: string, value: string) => {
    setFilters(prev => ({ ...prev, [key]: value }));
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-50">
      <div className="bg-white shadow-sm border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="py-8">
            <div className="text-center">
              <h1 className="text-4xl font-bold bg-gradient-to-r from-blue-600 to-indigo-600 bg-clip-text text-transparent">
                Future Development Insights
              </h1>
              <p className="mt-4 text-lg text-gray-600 max-w-3xl mx-auto">
                AI-powered analysis of upcoming infrastructure projects and price forecasts to make informed investment decisions
              </p>
              <div className="mt-6 flex justify-center space-x-4">
                <div className="flex items-center text-sm text-gray-500">
                  <div className="w-2 h-2 bg-green-500 rounded-full mr-2"></div>
                  ML-Powered Predictions
                </div>
                <div className="flex items-center text-sm text-gray-500">
                  <div className="w-2 h-2 bg-blue-500 rounded-full mr-2"></div>
                  Real-time Analysis
                </div>
                <div className="flex items-center text-sm text-gray-500">
                  <div className="w-2 h-2 bg-purple-500 rounded-full mr-2"></div>
                  Investment Insights
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        {/* Filters */}
        <div className="bg-white shadow-lg border border-gray-200 rounded-2xl p-8 mb-12">
          <div className="flex items-center mb-6">
            <Filter className="w-6 h-6 text-blue-600 mr-3" />
            <h2 className="text-xl font-bold text-gray-900">Search Filters</h2>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <div className="space-y-2">
              <label className="block text-sm font-semibold text-gray-700 mb-2">City</label>
              <select
                value={filters.city}
                onChange={(e) => handleFilterChange('city', e.target.value)}
                className="w-full border-2 border-gray-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 font-medium bg-gray-50 hover:bg-white"
              >
                <option value=""> All Cities</option>
                <option value="Mumbai"> Mumbai</option>
                <option value="Bangalore"> Bangalore</option>
                <option value="Delhi"> Delhi</option>
                <option value="Pune"> Pune</option>
                <option value="Chennai"> Chennai</option>
              </select>
            </div>
            
            <div className="space-y-2">
              <label className="block text-sm font-semibold text-gray-700 mb-2">Locality</label>
              <input
                type="text"
                value={filters.locality}
                onChange={(e) => handleFilterChange('locality', e.target.value)}
                placeholder="🔍 Enter locality name"
                className="w-full border-2 border-gray-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 font-medium bg-gray-50 hover:bg-white"
              />
            </div>
            
            <div className="space-y-2">
              <label className="block text-sm font-semibold text-gray-700 mb-2">Property Type</label>
              <select
                value={filters.propertyType}
                onChange={(e) => handleFilterChange('propertyType', e.target.value)}
                className="w-full border-2 border-gray-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 font-medium bg-gray-50 hover:bg-white"
              >
                <option value="apartment"> Apartment</option>
                <option value="villa"> Villa</option>
                <option value="plot"> Plot</option>
                <option value="commercial"> Commercial</option>
              </select>
            </div>
          </div>
        </div>

        {/* Main Content - Professional Sections */}
        <div className="space-y-12">
          {/* Section 1 - Infrastructure Projects */}
          <div className="bg-white shadow-xl rounded-3xl overflow-hidden border border-gray-100">
            <div className="bg-gradient-to-r from-blue-600 to-indigo-600 px-8 py-6">
              <h2 className="text-2xl font-bold text-black flex items-center">
                <div className="w-8 h-8 bg-white/20 rounded-lg flex items-center justify-center mr-4">
                  🏗️
                </div>
                Infrastructure Development Projects
              </h2>
              <p className="text-blue-100 mt-2">AI-analyzed upcoming infrastructure projects and their impact on property values</p>
            </div>
            <div className="p-0">
              <BuyerInfraMap 
                city={filters.city || undefined}
                locality={filters.locality || undefined}
              />
            </div>
          </div>

          {/* Section 2 - Price Forecasts */}
          <div className="bg-white shadow-xl rounded-3xl overflow-hidden border border-gray-100">
            <div className="bg-gradient-to-r from-green-600 to-emerald-600 px-8 py-6">
              <h2 className="text-2xl font-bold text-black flex items-center">
                <div className="w-8 h-8 bg-white/20 rounded-lg flex items-center justify-center mr-4">
                  📈
                </div>
                ML Price Forecasts & Market Trends
              </h2>
              <p className="text-green-100 mt-2">Machine learning powered price predictions and growth analysis</p>
            </div>
            <div className="p-0">
              <ForecastChart
                city={filters.city || undefined}
                locality={filters.locality || undefined}
                propertyType={filters.propertyType}
              />
            </div>
          </div>

          {/* Section 3 - Development Hotspots */}
          <div className="bg-white shadow-xl rounded-3xl overflow-hidden border border-gray-100">
            <div className="bg-gradient-to-r from-purple-600 to-pink-600 px-8 py-6">
              <h2 className="text-2xl font-bold text-black flex items-center">
                <div className="w-8 h-8 bg-white/20 rounded-lg flex items-center justify-center mr-4">
                  🔥
                </div>
                AI Development Hotspots
              </h2>
              <p className="text-purple-100 mt-2">Intelligent scoring of high-growth potential areas for investment</p>
            </div>
            <div className="p-0">
              <HotspotPanel
                city={filters.city || undefined}
                userRole="buyer"
              />
            </div>
          </div>
          
          {/* Quick Actions */}
          <div className="bg-white shadow-xl rounded-3xl border border-gray-100 overflow-hidden">
            <div className="bg-gradient-to-r from-gray-800 to-gray-900 px-8 py-6">
              <h3 className="text-2xl font-bold text-black flex items-center">
                <div className="w-8 h-8 bg-white/20 rounded-lg flex items-center justify-center mr-4">
                  ⚡
                </div>
                Quick Actions
              </h3>
              <p className="text-black-300 mt-2">Save, export, and compare your market insights</p>
            </div>
            
            <div className="p-8">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <button className="group text-left p-6 border-2 border-gray-200 rounded-2xl hover:border-blue-400 hover:shadow-lg transition-all duration-300 bg-gradient-to-br from-blue-50 to-indigo-50 hover:from-blue-100 hover:to-indigo-100">
                  <div className="flex items-center mb-3">
                    <div className="w-10 h-10 bg-blue-500 rounded-xl flex items-center justify-center text-white font-bold mr-3 group-hover:scale-110 transition-transform">
                      🔖
                    </div>
                    <div className="font-bold text-gray-900">Save Current Search</div>
                  </div>
                  <div className="text-sm text-gray-600">Get notified of new developments and market changes</div>
                </button>
                
                <button className="group text-left p-6 border-2 border-gray-200 rounded-2xl hover:border-green-400 hover:shadow-lg transition-all duration-300 bg-gradient-to-br from-green-50 to-emerald-50 hover:from-green-100 hover:to-emerald-100">
                  <div className="flex items-center mb-3">
                    <div className="w-10 h-10 bg-green-500 rounded-xl flex items-center justify-center text-white font-bold mr-3 group-hover:scale-110 transition-transform">
                      📄
                    </div>
                    <div className="font-bold text-gray-900">Export Report</div>
                  </div>
                  <div className="text-sm text-gray-600">Download comprehensive AI analysis and insights</div>
                </button>
                
                <button className="group text-left p-6 border-2 border-gray-200 rounded-2xl hover:border-purple-400 hover:shadow-lg transition-all duration-300 bg-gradient-to-br from-purple-50 to-pink-50 hover:from-purple-100 hover:to-pink-100">
                  <div className="flex items-center mb-3">
                    <div className="w-10 h-10 bg-purple-500 rounded-xl flex items-center justify-center text-white font-bold mr-3 group-hover:scale-110 transition-transform">
                      🔄
                    </div>
                    <div className="font-bold text-gray-900">Compare Localities</div>
                  </div>
                  <div className="text-sm text-gray-600">Side-by-side ML comparison of different areas</div>
                </button>
              </div>
            </div>
          </div>
        </div>
        
        {/* Footer */}
        <div className="mt-16 text-center py-8">
          <div className="text-sm text-gray-500">
            Powered by <span className="font-semibold text-blue-600">AI & Machine Learning</span> • 
            Real-time data analysis • Investment insights
          </div>
        </div>
      </div>
    </div>
  );
}