import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { ArrowLeft, Calendar, Save } from 'lucide-react';
import { PropertySelector } from '../../components/itinerary/PropertySelector';
import { DraggableList } from '../../components/itinerary/DraggableList';
import { TimelineView } from '../../components/itinerary/TimelineView';
import { ItineraryProperty } from '../../types/itinerary';
import { toast } from 'sonner';

export const CreateItinerary = () => {
  const navigate = useNavigate();
  const [step, setStep] = useState(1);
  const [title, setTitle] = useState('');
  const [date, setDate] = useState('');
  const [selectedPropertyIds, setSelectedPropertyIds] = useState<number[]>([]);
  const [properties, setProperties] = useState<ItineraryProperty[]>([]);

  const mockProperties = [
    { id: 1, title: 'Luxury 3BHK Apartment', address: 'Bandra West', city: 'Mumbai', photo: '/public/3D.png' },
    { id: 2, title: 'Modern Villa', address: 'Juhu', city: 'Mumbai', photo: '/public/3D.png' },
    { id: 3, title: 'Penthouse Suite', address: 'Worli', city: 'Mumbai', photo: '/public/3D.png' },
    { id: 4, title: '2BHK Apartment', address: 'Koregaon Park', city: 'Pune', photo: '/public/3D.png' },
    { id: 5, title: 'Commercial Space', address: 'Hinjewadi', city: 'Pune', photo: '/public/3D.png' }
  ];

  const handleToggleProperty = (propertyId: number) => {
    if (selectedPropertyIds.includes(propertyId)) {
      setSelectedPropertyIds(selectedPropertyIds.filter(id => id !== propertyId));
    } else {
      if (selectedPropertyIds.length >= 6) {
        toast.error('Maximum 6 properties per itinerary');
        return;
      }
      setSelectedPropertyIds([...selectedPropertyIds, propertyId]);
    }
  };

  const handleNext = () => {
    if (step === 1 && (!title || !date)) {
      toast.error('Please enter title and date');
      return;
    }
    if (step === 2 && selectedPropertyIds.length === 0) {
      toast.error('Please select at least one property');
      return;
    }
    if (step === 2) {
      const newProperties: ItineraryProperty[] = selectedPropertyIds.map((id, index) => {
        const prop = mockProperties.find(p => p.id === id)!;
        return {
          id: `prop-${id}`,
          propertyId: id,
          title: prop.title,
          address: prop.address,
          city: prop.city,
          photo: prop.photo,
          visitDate: date,
          visitTime: `${9 + index}:00 AM`,
          duration: 60,
          order: index,
          status: 'pending' as const
        };
      });
      setProperties(newProperties);
    }
    setStep(step + 1);
  };

  const handleSave = () => {
    toast.success('Itinerary created successfully!');
    navigate('/my-itineraries');
  };

  const totalDuration = properties.reduce((sum, p) => sum + p.duration, 0) + (properties.length - 1) * 15;

  return (
    <div className="min-h-screen bg-gray-50 pt-20">
      <div className="max-w-4xl mx-auto px-4 py-8">
        <button onClick={() => step === 1 ? navigate('/my-itineraries') : setStep(step - 1)} className="flex items-center gap-2 text-gray-600 hover:text-gray-900 mb-6">
          <ArrowLeft className="w-5 h-5" />
          Back
        </button>

        <div className="bg-white rounded-lg border border-gray-200 p-6 mb-6">
          <div className="flex items-center justify-between mb-6">
            {[1, 2, 3, 4].map((s) => (
              <div key={s} className="flex items-center">
                <div className={`w-10 h-10 rounded-full flex items-center justify-center font-semibold ${step >= s ? 'bg-blue-600 text-white' : 'bg-gray-200 text-gray-600'}`}>{s}</div>
                {s < 4 && <div className={`w-16 h-1 ${step > s ? 'bg-blue-600' : 'bg-gray-200'}`} />}
              </div>
            ))}
          </div>
          <div className="text-center">
            <h2 className="text-xl font-semibold text-gray-900">
              {step === 1 && 'Basic Details'}
              {step === 2 && 'Select Properties'}
              {step === 3 && 'Arrange Order'}
              {step === 4 && 'Review & Save'}
            </h2>
          </div>
        </div>

        <div className="bg-white rounded-lg border border-gray-200 p-6">
          {step === 1 && (
            <div className="space-y-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Itinerary Title</label>
                <input type="text" value={title} onChange={(e) => setTitle(e.target.value)} placeholder="e.g., Weekend Property Tour" className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Visit Date</label>
                <input type="date" value={date} onChange={(e) => setDate(e.target.value)} min={new Date().toISOString().split('T')[0]} className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" />
              </div>
            </div>
          )}

          {step === 2 && (
            <div>
              <div className="mb-4 p-4 bg-blue-50 border border-blue-200 rounded-lg">
                <p className="text-sm text-blue-800">Selected: {selectedPropertyIds.length}/6 properties</p>
              </div>
              <PropertySelector selectedProperties={selectedPropertyIds} onToggle={handleToggleProperty} />
            </div>
          )}

          {step === 3 && (
            <div>
              <div className="mb-4 p-4 bg-amber-50 border border-amber-200 rounded-lg">
                <p className="text-sm text-amber-800">Drag to reorder properties. Adjust visit times as needed.</p>
              </div>
              <DraggableList properties={properties} onReorder={setProperties} onRemove={(id) => setProperties(properties.filter(p => p.id !== id))} onUpdateTime={(id, time) => setProperties(properties.map(p => p.id === id ? { ...p, visitTime: time } : p))} />
            </div>
          )}

          {step === 4 && (
            <div>
              <div className="mb-6 p-4 bg-green-50 border border-green-200 rounded-lg">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="font-semibold text-green-900">{title}</p>
                    <p className="text-sm text-green-700">{new Date(date).toLocaleDateString('en-US', { weekday: 'long', month: 'long', day: 'numeric' })}</p>
                  </div>
                  <div className="text-right">
                    <p className="text-sm text-green-700">Total Duration</p>
                    <p className="font-semibold text-green-900">{Math.floor(totalDuration / 60)}h {totalDuration % 60}m</p>
                  </div>
                </div>
              </div>
              <TimelineView properties={properties} />
            </div>
          )}

          <div className="flex gap-4 mt-6">
            {step < 4 ? (
              <button onClick={handleNext} className="flex-1 bg-blue-600 text-white py-3 rounded-lg hover:bg-blue-700 transition-colors font-semibold">
                Continue
              </button>
            ) : (
              <button onClick={handleSave} className="flex-1 bg-green-600 text-white py-3 rounded-lg hover:bg-green-700 transition-colors font-semibold flex items-center justify-center gap-2">
                <Save className="w-5 h-5" />
                Save Itinerary
              </button>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};
