import { useParams, useNavigate } from 'react-router-dom';
import { ArrowLeft, Calendar, Clock, MapPin, Play, Edit, Share2 } from 'lucide-react';
import { TimelineView } from '../../components/itinerary/TimelineView';
import { StatusBadge } from '../../components/itinerary/StatusBadge';
import { Itinerary } from '../../types/itinerary';

export const ItineraryDetail = () => {
  const { id } = useParams();
  const navigate = useNavigate();

  // Mock data - replace with API call
  const itinerary: Itinerary = {
    id: id!,
    userId: 'user1',
    title: 'Weekend Property Tour - Mumbai',
    date: '2024-02-15',
    status: 'scheduled',
    totalDuration: 240,
    totalDistance: 15.5,
    createdAt: '2024-01-20T10:00:00Z',
    updatedAt: '2024-01-20T10:00:00Z',
    properties: [
      {
        id: 'p1',
        propertyId: 1,
        title: 'Luxury 3BHK Apartment',
        address: 'Bandra West',
        city: 'Mumbai',
        photo: '/public/3D.png',
        visitDate: '2024-02-15',
        visitTime: '10:00 AM',
        duration: 60,
        order: 0,
        status: 'confirmed',
        latitude: 19.0596,
        longitude: 72.8295
      },
      {
        id: 'p2',
        propertyId: 2,
        title: 'Modern Villa',
        address: 'Juhu',
        city: 'Mumbai',
        photo: '/public/3D.png',
        visitDate: '2024-02-15',
        visitTime: '12:00 PM',
        duration: 60,
        order: 1,
        status: 'confirmed',
        latitude: 19.1075,
        longitude: 72.8263
      },
      {
        id: 'p3',
        propertyId: 3,
        title: 'Penthouse Suite',
        address: 'Worli',
        city: 'Mumbai',
        photo: '/public/3D.png',
        visitDate: '2024-02-15',
        visitTime: '2:00 PM',
        duration: 60,
        order: 2,
        status: 'confirmed',
        latitude: 19.0176,
        longitude: 72.8156
      }
    ],
    travelSegments: []
  };

  return (
    <div className="min-h-screen bg-gray-50 pt-20">
      <div className="max-w-4xl mx-auto px-4 py-8">
        <button onClick={() => navigate('/my-itineraries')} className="flex items-center gap-2 text-gray-600 hover:text-gray-900 mb-6">
          <ArrowLeft className="w-5 h-5" />
          Back to Itineraries
        </button>

        <div className="bg-white rounded-lg border border-gray-200 p-6 mb-6">
          <div className="flex items-start justify-between mb-4">
            <div className="flex-1">
              <h1 className="text-2xl font-bold text-gray-900 mb-2">{itinerary.title}</h1>
              <div className="flex items-center gap-4 text-gray-600">
                <div className="flex items-center gap-2">
                  <Calendar className="w-5 h-5" />
                  <span>{new Date(itinerary.date).toLocaleDateString('en-US', { weekday: 'long', month: 'long', day: 'numeric' })}</span>
                </div>
                <StatusBadge status={itinerary.status} />
              </div>
            </div>
            <div className="flex gap-2">
              <button onClick={() => navigate(`/create-itinerary?edit=${id}`)} className="p-2 border border-gray-300 rounded-lg hover:bg-gray-50">
                <Edit className="w-5 h-5 text-gray-600" />
              </button>
              <button className="p-2 border border-gray-300 rounded-lg hover:bg-gray-50">
                <Share2 className="w-5 h-5 text-gray-600" />
              </button>
            </div>
          </div>

          <div className="grid grid-cols-3 gap-4 p-4 bg-gray-50 rounded-lg">
            <div>
              <p className="text-sm text-gray-600">Properties</p>
              <p className="text-xl font-semibold text-gray-900">{itinerary.properties.length}</p>
            </div>
            <div>
              <p className="text-sm text-gray-600">Total Duration</p>
              <p className="text-xl font-semibold text-gray-900">{Math.floor(itinerary.totalDuration / 60)}h {itinerary.totalDuration % 60}m</p>
            </div>
            <div>
              <p className="text-sm text-gray-600">Distance</p>
              <p className="text-xl font-semibold text-gray-900">{itinerary.totalDistance} km</p>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-lg border border-gray-200 p-6 mb-6">
          <h2 className="text-lg font-semibold text-gray-900 mb-4">Visit Timeline</h2>
          <TimelineView properties={itinerary.properties} />
        </div>

        {itinerary.status === 'scheduled' && (
          <button onClick={() => navigate(`/itinerary/${id}/tour`)} className="w-full bg-blue-600 text-white py-4 rounded-lg hover:bg-blue-700 transition-colors font-semibold flex items-center justify-center gap-2">
            <Play className="w-5 h-5" />
            Start Live Tour
          </button>
        )}
      </div>
    </div>
  );
};
