import { useState } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { ArrowLeft, CheckCircle, Circle, Navigation, Clock, Award } from 'lucide-react';
import { TourCheckIn } from '../../components/itinerary/TourCheckIn';
import { toast } from 'sonner';

export const LiveTour = () => {
  const { id } = useParams();
  const navigate = useNavigate();
  const [currentIndex, setCurrentIndex] = useState(0);
  const [completedProperties, setCompletedProperties] = useState<string[]>([]);
  const [showCheckIn, setShowCheckIn] = useState(false);

  const properties = [
    { id: 'p1', title: 'Luxury 3BHK Apartment', address: 'Bandra West', city: 'Mumbai', photo: '/public/3D.png', time: '10:00 AM', duration: 60 },
    { id: 'p2', title: 'Modern Villa', address: 'Juhu', city: 'Mumbai', photo: '/public/3D.png', time: '12:00 PM', duration: 60 },
    { id: 'p3', title: 'Penthouse Suite', address: 'Worli', city: 'Mumbai', photo: '/public/3D.png', time: '2:00 PM', duration: 60 }
  ];

  const currentProperty = properties[currentIndex];
  const isCompleted = completedProperties.includes(currentProperty.id);
  const allCompleted = completedProperties.length === properties.length;

  const handleCheckIn = (method: 'qr' | 'gps' | 'manual') => {
    setCompletedProperties([...completedProperties, currentProperty.id]);
    setShowCheckIn(false);
    toast.success(`Checked in to ${currentProperty.title}!`);
  };

  const handleNext = () => {
    if (currentIndex < properties.length - 1) {
      setCurrentIndex(currentIndex + 1);
    } else {
      navigate(`/itinerary/${id}/complete`);
    }
  };

  if (allCompleted) {
    return (
      <div className="min-h-screen bg-gray-50 pt-20">
        <div className="max-w-2xl mx-auto px-4 py-8 text-center">
          <div className="bg-white rounded-lg border border-gray-200 p-8">
            <div className="w-20 h-20 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <Award className="w-10 h-10 text-green-600" />
            </div>
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Tour Completed!</h1>
            <p className="text-gray-600 mb-6">You've successfully visited all {properties.length} properties</p>
            <div className="grid grid-cols-3 gap-4 mb-6">
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-600">Properties</p>
                <p className="text-2xl font-bold text-gray-900">{properties.length}</p>
              </div>
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-600">Duration</p>
                <p className="text-2xl font-bold text-gray-900">4h 15m</p>
              </div>
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-600">Distance</p>
                <p className="text-2xl font-bold text-gray-900">15.5 km</p>
              </div>
            </div>
            <button onClick={() => navigate('/my-itineraries')} className="w-full bg-blue-600 text-white py-3 rounded-lg hover:bg-blue-700 transition-colors font-semibold">
              Back to Itineraries
            </button>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 pt-20">
      <div className="max-w-2xl mx-auto px-4 py-8">
        <button onClick={() => navigate(`/itinerary/${id}`)} className="flex items-center gap-2 text-gray-600 hover:text-gray-900 mb-6">
          <ArrowLeft className="w-5 h-5" />
          Exit Tour
        </button>

        <div className="bg-white rounded-lg border border-gray-200 p-4 mb-6">
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-lg font-semibold text-gray-900">Tour Progress</h2>
            <span className="text-sm text-gray-600">{completedProperties.length}/{properties.length} completed</span>
          </div>
          <div className="flex gap-2">
            {properties.map((prop, idx) => (
              <div key={prop.id} className="flex-1 h-2 rounded-full bg-gray-200 overflow-hidden">
                <div className={`h-full transition-all ${completedProperties.includes(prop.id) ? 'bg-green-600' : idx === currentIndex ? 'bg-blue-600 w-1/2' : 'bg-transparent'}`} />
              </div>
            ))}
          </div>
        </div>

        <div className="bg-white rounded-lg border border-gray-200 overflow-hidden mb-6">
          <img src={currentProperty.photo} alt={currentProperty.title} className="w-full h-64 object-cover" />
          <div className="p-6">
            <div className="flex items-start justify-between mb-4">
              <div className="flex-1">
                <div className="flex items-center gap-2 mb-2">
                  <span className="inline-flex items-center justify-center w-8 h-8 rounded-full bg-blue-600 text-white text-sm font-bold">{currentIndex + 1}</span>
                  <h3 className="text-xl font-bold text-gray-900">{currentProperty.title}</h3>
                </div>
                <p className="text-gray-600">{currentProperty.address}, {currentProperty.city}</p>
              </div>
              {isCompleted && <CheckCircle className="w-8 h-8 text-green-600" />}
            </div>

            <div className="flex items-center gap-4 text-sm text-gray-600 mb-6">
              <div className="flex items-center gap-1">
                <Clock className="w-4 h-4" />
                <span>{currentProperty.time}</span>
              </div>
              <span>•</span>
              <span>{currentProperty.duration} min visit</span>
            </div>

            {!isCompleted && !showCheckIn && (
              <div className="space-y-3">
                <button onClick={() => window.open(`https://www.google.com/maps/dir/?api=1&destination=${currentProperty.address},${currentProperty.city}`, '_blank')} className="w-full bg-blue-600 text-white py-3 rounded-lg hover:bg-blue-700 transition-colors font-semibold flex items-center justify-center gap-2">
                  <Navigation className="w-5 h-5" />
                  Navigate to Property
                </button>
                <button onClick={() => setShowCheckIn(true)} className="w-full border-2 border-blue-600 text-blue-600 py-3 rounded-lg hover:bg-blue-50 transition-colors font-semibold">
                  Check In
                </button>
              </div>
            )}

            {showCheckIn && !isCompleted && (
              <TourCheckIn propertyTitle={currentProperty.title} onCheckIn={handleCheckIn} />
            )}

            {isCompleted && (
              <button onClick={handleNext} className="w-full bg-green-600 text-white py-3 rounded-lg hover:bg-green-700 transition-colors font-semibold">
                {currentIndex < properties.length - 1 ? 'Next Property' : 'Complete Tour'}
              </button>
            )}
          </div>
        </div>

        <div className="bg-white rounded-lg border border-gray-200 p-4">
          <h4 className="font-semibold text-gray-900 mb-3">Upcoming Properties</h4>
          <div className="space-y-2">
            {properties.slice(currentIndex + 1).map((prop, idx) => (
              <div key={prop.id} className="flex items-center gap-3 p-3 bg-gray-50 rounded-lg">
                <Circle className="w-5 h-5 text-gray-400" />
                <div className="flex-1">
                  <p className="font-medium text-gray-900">{prop.title}</p>
                  <p className="text-sm text-gray-600">{prop.time}</p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
};
