import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { Plus, Calendar, Filter } from 'lucide-react';
import { ItineraryCard } from '../../components/itinerary/ItineraryCard';
import { Itinerary } from '../../types/itinerary';
import { toast } from 'sonner';

export const MyItineraries = () => {
  const navigate = useNavigate();
  const [filter, setFilter] = useState<'all' | 'upcoming' | 'past' | 'completed'>('all');

  // Mock data - replace with API call
  const [itineraries] = useState<Itinerary[]>([
    {
      id: '1',
      userId: 'user1',
      title: 'Weekend Property Tour - Mumbai',
      date: '2024-02-15',
      status: 'scheduled',
      totalDuration: 240,
      totalDistance: 15.5,
      createdAt: '2024-01-20T10:00:00Z',
      updatedAt: '2024-01-20T10:00:00Z',
      properties: [
        {
          id: 'p1',
          propertyId: 1,
          title: 'Luxury 3BHK Apartment',
          address: 'Bandra West',
          city: 'Mumbai',
          photo: '/public/3D.png',
          visitDate: '2024-02-15',
          visitTime: '10:00 AM',
          duration: 60,
          order: 0,
          status: 'confirmed',
          latitude: 19.0596,
          longitude: 72.8295
        },
        {
          id: 'p2',
          propertyId: 2,
          title: 'Modern Villa',
          address: 'Juhu',
          city: 'Mumbai',
          photo: '/public/3D.png',
          visitDate: '2024-02-15',
          visitTime: '12:00 PM',
          duration: 60,
          order: 1,
          status: 'confirmed',
          latitude: 19.1075,
          longitude: 72.8263
        },
        {
          id: 'p3',
          propertyId: 3,
          title: 'Penthouse Suite',
          address: 'Worli',
          city: 'Mumbai',
          photo: '/public/3D.png',
          visitDate: '2024-02-15',
          visitTime: '2:00 PM',
          duration: 60,
          order: 2,
          status: 'confirmed',
          latitude: 19.0176,
          longitude: 72.8156
        }
      ],
      travelSegments: []
    },
    {
      id: '2',
      userId: 'user1',
      title: 'Investment Properties - Pune',
      date: '2024-01-10',
      status: 'completed',
      totalDuration: 180,
      totalDistance: 12.3,
      createdAt: '2024-01-05T10:00:00Z',
      updatedAt: '2024-01-10T18:00:00Z',
      properties: [
        {
          id: 'p4',
          propertyId: 4,
          title: '2BHK Apartment',
          address: 'Koregaon Park',
          city: 'Pune',
          photo: '/public/3D.png',
          visitDate: '2024-01-10',
          visitTime: '11:00 AM',
          duration: 60,
          order: 0,
          status: 'completed'
        },
        {
          id: 'p5',
          propertyId: 5,
          title: 'Commercial Space',
          address: 'Hinjewadi',
          city: 'Pune',
          photo: '/public/3D.png',
          visitDate: '2024-01-10',
          visitTime: '1:00 PM',
          duration: 60,
          order: 1,
          status: 'completed'
        }
      ],
      travelSegments: []
    }
  ]);

  const filteredItineraries = itineraries.filter(it => {
    if (filter === 'all') return true;
    if (filter === 'upcoming') return it.status === 'scheduled' || it.status === 'draft';
    if (filter === 'past') return new Date(it.date) < new Date();
    if (filter === 'completed') return it.status === 'completed';
    return true;
  });

  const handleView = (id: string) => navigate(`/itinerary/${id}`);
  const handleEdit = (id: string) => navigate(`/create-itinerary?edit=${id}`);
  const handleDuplicate = (id: string) => {
    toast.success('Itinerary duplicated successfully');
  };
  const handleDelete = (id: string) => {
    toast.success('Itinerary deleted successfully');
  };
  const handleStartTour = (id: string) => navigate(`/itinerary/${id}/tour`);

  return (
    <div className="min-h-screen bg-gray-50 pt-20">
      <div className="max-w-7xl mx-auto px-4 py-8">
        <div className="flex items-center justify-between mb-8">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">My Itineraries</h1>
            <p className="text-gray-600 mt-1">Plan and manage your property visits</p>
          </div>
          <button onClick={() => navigate('/create-itinerary')} className="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors flex items-center gap-2">
            <Plus className="w-5 h-5" />
            Create Itinerary
          </button>
        </div>

        <div className="bg-white rounded-lg border border-gray-200 p-4 mb-6">
          <div className="flex items-center gap-4">
            <Filter className="w-5 h-5 text-gray-600" />
            <div className="flex gap-2">
              {(['all', 'upcoming', 'past', 'completed'] as const).map((f) => (
                <button key={f} onClick={() => setFilter(f)} className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${filter === f ? 'bg-blue-600 text-white' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'}`}>
                  {f.charAt(0).toUpperCase() + f.slice(1)}
                </button>
              ))}
            </div>
          </div>
        </div>

        {filteredItineraries.length === 0 ? (
          <div className="bg-white rounded-lg border border-gray-200 p-12 text-center">
            <Calendar className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-xl font-semibold text-gray-900 mb-2">No itineraries found</h3>
            <p className="text-gray-600 mb-6">Create your first itinerary to start planning property visits</p>
            <button onClick={() => navigate('/create-itinerary')} className="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors inline-flex items-center gap-2">
              <Plus className="w-5 h-5" />
              Create Itinerary
            </button>
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {filteredItineraries.map((itinerary) => (
              <ItineraryCard key={itinerary.id} itinerary={itinerary} onView={handleView} onEdit={handleEdit} onDuplicate={handleDuplicate} onDelete={handleDelete} onStartTour={handleStartTour} />
            ))}
          </div>
        )}
      </div>
    </div>
  );
};
