import React, { useState } from 'react';
import { PricePredictionForm } from '../../components/price-prediction/PricePredictionForm';
import { PSFForecasting } from '../../components/price-prediction/PSFForecasting';
import { PricePredictionResult } from '../../components/price-prediction/PricePredictionResult';
import { PredictionResponse } from '../../api/price-prediction/pricePrediction';

interface PricePredictionPageProps {
    onError?: (error: string) => void;
}

export const PricePredictionPage: React.FC<PricePredictionPageProps> = ({ onError }) => {
    const [activeTab, setActiveTab] = useState<'predict' | 'forecast'>('predict');
    const [predictionResult, setPredictionResult] = useState<PredictionResponse | null>(null);

    const handlePredictionComplete = (result: PredictionResponse) => {
        setPredictionResult(result);
    };

    const handleNewPrediction = () => {
        setPredictionResult(null);
    };

    return (
        <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-purple-50">
            <div className="container mx-auto px-4 py-8">
                {/* Header */}
                <div className="text-center mb-8">
                    <h1 className="text-4xl font-bold text-gray-900 mb-4">
                        AI-Powered Price Prediction
                    </h1>
                    <p className="text-lg text-gray-600 max-w-2xl mx-auto">
                        Get accurate property valuations using advanced machine learning models and market analysis
                    </p>
                </div>

                {/* Tab Navigation */}
                <div className="flex justify-center mb-8">
                    <div className="bg-white p-1 rounded-xl shadow-lg border border-gray-200">
                        <button
                            onClick={() => setActiveTab('predict')}
                            className={`px-6 py-3 rounded-lg font-semibold transition-all duration-300 ${
                                activeTab === 'predict'
                                    ? 'bg-[#0056D2] text-white shadow-lg'
                                    : 'text-gray-600 hover:text-[#0056D2] hover:bg-blue-50'
                            }`}
                        >
                            Price Prediction
                        </button>
                        <button
                            onClick={() => setActiveTab('forecast')}
                            className={`px-6 py-3 rounded-lg font-semibold transition-all duration-300 ${
                                activeTab === 'forecast'
                                    ? 'bg-[#0056D2] text-white shadow-lg'
                                    : 'text-gray-600 hover:text-[#0056D2] hover:bg-blue-50'
                            }`}
                        >
                            PSF Forecasting
                        </button>
                    </div>
                </div>

                {/* Content */}
                <div className="max-w-6xl mx-auto">
                    {activeTab === 'predict' && (
                        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                            {/* Form */}
                            <div>
                                <PricePredictionForm
                                    onPredictionComplete={handlePredictionComplete}
                                    onError={onError}
                                />
                                
                                {predictionResult && (
                                    <div className="mt-6">
                                        <button
                                            onClick={handleNewPrediction}
                                            className="w-full bg-gray-100 hover:bg-gray-200 text-gray-700 font-semibold py-3 px-6 rounded-xl transition-all duration-300 border border-gray-300"
                                        >
                                            New Prediction
                                        </button>
                                    </div>
                                )}
                            </div>

                            {/* Results */}
                            <div>
                                {predictionResult ? (
                                    <PricePredictionResult result={predictionResult} />
                                ) : (
                                    <div className="bg-white p-12 rounded-2xl shadow-lg border border-gray-100 text-center">
                                        <svg className="w-20 h-20 mx-auto text-gray-300 mb-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                                        </svg>
                                        <h3 className="text-xl font-semibold text-gray-700 mb-2">
                                            Ready for Prediction
                                        </h3>
                                        <p className="text-gray-500">
                                            Fill out the form to get your AI-powered price prediction
                                        </p>
                                    </div>
                                )}
                            </div>
                        </div>
                    )}

                    {activeTab === 'forecast' && (
                        <div className="max-w-4xl mx-auto">
                            <PSFForecasting onError={onError} />
                        </div>
                    )}
                </div>

                {/* Features Section */}
                <div className="mt-16 max-w-6xl mx-auto">
                    <h2 className="text-2xl font-bold text-center text-gray-900 mb-8">
                        Why Choose Our AI Prediction?
                    </h2>
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div className="bg-white p-6 rounded-xl shadow-lg border border-gray-100 text-center">
                            <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                <svg className="w-6 h-6 text-[#0056D2]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                                </svg>
                            </div>
                            <h3 className="text-lg font-semibold text-gray-900 mb-2">AI-Powered Analysis</h3>
                            <p className="text-sm text-gray-600">
                                Advanced machine learning models trained on extensive real estate data
                            </p>
                        </div>

                        <div className="bg-white p-6 rounded-xl shadow-lg border border-gray-100 text-center">
                            <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                <svg className="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" />
                                </svg>
                            </div>
                            <h3 className="text-lg font-semibold text-gray-900 mb-2">Future Forecasting</h3>
                            <p className="text-sm text-gray-600">
                                Predict property values for years 2025-2030 with market trend analysis
                            </p>
                        </div>

                        <div className="bg-white p-6 rounded-xl shadow-lg border border-gray-100 text-center">
                            <div className="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                <svg className="w-6 h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                                </svg>
                            </div>
                            <h3 className="text-lg font-semibold text-gray-900 mb-2">High Accuracy</h3>
                            <p className="text-sm text-gray-600">
                                Confidence scoring and detailed breakdown of price factors
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
};

export const PricePrediction = PricePredictionPage;
export default PricePredictionPage;