import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { ContractList } from '@/components/contracts/ContractList';
import { ContractForm } from '@/components/contracts/ContractForm';
import { StampPaperViewer } from '@/components/contracts/StampPaperViewer';
import { 
  FileText, 
  Plus, 
  RefreshCw, 
  CheckCircle, 
  Clock, 
  AlertTriangle,
  Download,
  Eye,
  Users,
  Building,
  Calendar
} from 'lucide-react';
import { toast } from '@/utils/toast';

interface ContractStats {
  total: number;
  active: number;
  pending: number;
  completed: number;
  signed: number;
}

interface Contract {
  id: string;
  title: string;
  type: string;
  status: 'draft' | 'review' | 'approval' | 'active' | 'completed' | 'pending_signatures' | 'signed';
  property?: string;
  property_location?: string;
  property_type?: string;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  sale_price?: number;
  monthly_rent?: number;
  property_email?: string;
  property_owner?: string;
  parties: string[];
  party_emails?: string[];
  amount?: number;
  start_date?: string;
  end_date?: string;
  created_at: string;
  next_milestone?: string;
  progress: number;
  buyer_id?: string;
  seller_id?: string;
}

export const ContractManagement: React.FC = () => {
  const [contracts, setContracts] = useState<Contract[]>([]);
  const [stats, setStats] = useState<ContractStats>({
    total: 0,
    active: 0,
    pending: 0,
    completed: 0,
    signed: 0
  });
  const [loading, setLoading] = useState(true);
  const [showCreateForm, setShowCreateForm] = useState(false);
  const [showStampViewer, setShowStampViewer] = useState(false);
  const [selectedContractId, setSelectedContractId] = useState<string | null>(null);

  useEffect(() => {
    loadContracts();
  }, []);

  const loadContracts = async () => {
    try {
      setLoading(true);
      
      // Get user ID from localStorage - check multiple sources
      let userId = null;
      try {
        // Try seller-specific storage first
        userId = localStorage.getItem('sellerId') || localStorage.getItem('user_id');
        
        // Try to get from sellerToken
        if (!userId) {
          const sellerToken = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
          if (sellerToken && sellerToken.includes('.')) {
            try {
              const payload = JSON.parse(atob(sellerToken.split('.')[1]));
              userId = payload.user_id || payload.sub || payload.id;
            } catch (e) {
              console.warn('Failed to decode token:', e);
            }
          }
        }
        
        // Try user object
        if (!userId) {
          const user = JSON.parse(localStorage.getItem('user') || '{}');
          userId = user.user_id || user.id;
        }
        
        // Try sellerData object
        if (!userId) {
          const sellerData = JSON.parse(localStorage.getItem('sellerData') || '{}');
          userId = sellerData.user_id || sellerData.id;
        }
        
        if (!userId) {
          console.error('No user ID found in localStorage');
          toast.error('Please log in to view contracts');
          return;
        }
        
        console.log('Loading contracts for user ID:', userId);
      } catch (e) {
        console.error('Error getting user ID:', e);
        toast.error('Failed to get user information');
        return;
      }

      const response = await fetch(`http://localhost:8090/api/contracts/user/${userId}`, {
        headers: {
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        const data = await response.json();
        const contractsList = data.contracts || [];
        setContracts(contractsList);
        
        // Calculate stats
        const newStats = {
          total: contractsList.length,
          active: contractsList.filter((c: Contract) => c.status === 'active').length,
          pending: contractsList.filter((c: Contract) => c.status === 'pending_signatures').length,
          completed: contractsList.filter((c: Contract) => c.status === 'completed').length,
          signed: contractsList.filter((c: Contract) => c.status === 'signed').length
        };
        setStats(newStats);
      } else {
        throw new Error('Failed to load contracts');
      }
    } catch (error) {
      console.error('Error loading contracts:', error);
      toast.error('Failed to load contracts');
    } finally {
      setLoading(false);
    }
  };

  const handleViewStampPaper = (contractId: string) => {
    setSelectedContractId(contractId);
    setShowStampViewer(true);
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'signed':
      case 'completed':
        return <CheckCircle className="w-4 h-4 text-green-600" />;
      case 'pending_signatures':
        return <Clock className="w-4 h-4 text-yellow-600" />;
      case 'active':
        return <CheckCircle className="w-4 h-4 text-blue-600" />;
      default:
        return <AlertTriangle className="w-4 h-4 text-gray-600" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'signed':
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'pending_signatures':
        return 'bg-yellow-100 text-yellow-800';
      case 'active':
        return 'bg-blue-100 text-blue-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 p-6">
        <div className="max-w-7xl mx-auto">
          <div className="text-center py-12">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
            <p className="text-gray-600">Loading contracts...</p>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 p-6">
      <div className="max-w-7xl mx-auto space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900 flex items-center gap-3">
              <FileText className="w-8 h-8 text-blue-600" />
              Contract Management
            </h1>
            <p className="text-gray-600 mt-1">Manage your property contracts and agreements</p>
          </div>
          <div className="flex gap-3">
            <Button 
              variant="outline" 
              onClick={loadContracts}
              className="flex items-center gap-2"
            >
              <RefreshCw className="w-4 h-4" />
              Refresh
            </Button>
            <Button 
              onClick={() => setShowCreateForm(true)}
              className="flex items-center gap-2 bg-blue-500 hover:bg-blue-600 text-white"
            >
              <Plus className="w-4 h-4" />
              New Contract
            </Button>
          </div>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Total Contracts</p>
                  <p className="text-2xl font-bold text-gray-900">{stats.total}</p>
                </div>
                <FileText className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Active</p>
                  <p className="text-2xl font-bold text-blue-600">{stats.active}</p>
                </div>
                <CheckCircle className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Pending</p>
                  <p className="text-2xl font-bold text-yellow-600">{stats.pending}</p>
                </div>
                <Clock className="w-8 h-8 text-yellow-600" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Completed</p>
                  <p className="text-2xl font-bold text-green-600">{stats.completed}</p>
                </div>
                <CheckCircle className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Signed</p>
                  <p className="text-2xl font-bold text-emerald-600">{stats.signed}</p>
                </div>
                <Users className="w-8 h-8 text-emerald-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* My Contracts Section */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Building className="w-5 h-5 text-blue-600" />
              My Contracts
            </CardTitle>
          </CardHeader>
          <CardContent>
            {contracts.length > 0 ? (
              <ContractList 
                contracts={contracts} 
                onRefresh={loadContracts}
              />
            ) : (
              <div className="text-center py-12">
                <FileText className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No contracts found</h3>
                <p className="text-gray-600 mb-4">You haven't created any contracts yet.</p>
                <Button onClick={() => setShowCreateForm(true)}>
                  <Plus className="w-4 h-4 mr-2" />
                  Create Your First Contract
                </Button>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Signed Contracts with Stamp Papers */}
        {contracts.filter(c => c.status === 'signed').length > 0 && (
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Download className="w-5 h-5 text-green-600" />
                Signed Contracts & Stamp Papers
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {contracts
                  .filter(contract => contract.status === 'signed')
                  .map(contract => (
                    <Card key={contract.id} className="border-green-200 bg-green-50">
                      <CardContent className="p-4">
                        <div className="flex items-start justify-between mb-3">
                          <div>
                            <h4 className="font-semibold text-gray-900">{contract.title}</h4>
                            <p className="text-sm text-gray-600">Contract ID: {contract.id}</p>
                          </div>
                          <Badge className="bg-green-100 text-green-800">
                            <CheckCircle className="w-3 h-3 mr-1" />
                            Signed
                          </Badge>
                        </div>
                        
                        <div className="space-y-2 mb-4">
                          <div className="flex items-center gap-2 text-sm">
                            <Calendar className="w-4 h-4 text-gray-500" />
                            <span>Created: {new Date(contract.created_at).toLocaleDateString()}</span>
                          </div>
                          <div className="flex items-center gap-2 text-sm">
                            <Users className="w-4 h-4 text-gray-500" />
                            <span>Parties: {contract.parties.length}</span>
                          </div>
                        </div>
                        
                        <div className="flex gap-2">
                          <Button 
                            size="sm" 
                            variant="outline"
                            onClick={() => handleViewStampPaper(contract.id)}
                            className="flex-1"
                          >
                            <Eye className="w-4 h-4 mr-1" />
                            View Stamp Paper
                          </Button>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
              </div>
            </CardContent>
          </Card>
        )}

        {/* Legal Notice */}
        <Card className="bg-blue-50 border-blue-200">
          <CardContent className="p-4">
            <div className="flex items-start gap-3">
              <AlertTriangle className="w-5 h-5 text-blue-600 mt-0.5" />
              <div>
                <h4 className="font-medium text-blue-900 mb-1">Legal Notice</h4>
                <p className="text-sm text-blue-800">
                  All digitally signed contracts generate authentic e-stamp papers with legal validity 
                  under the Indian Stamp Act. These documents can be downloaded and used for legal 
                  proceedings. Ensure all parties have signed before proceeding with property transactions.
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Modals */}
      {showCreateForm && (
        <ContractForm 
          isOpen={showCreateForm}
          onClose={() => setShowCreateForm(false)}
          onSuccess={() => {
            setShowCreateForm(false);
            loadContracts();
          }}
        />
      )}

      {showStampViewer && selectedContractId && (
        <StampPaperViewer 
          isOpen={showStampViewer}
          contractId={selectedContractId}
          onClose={() => {
            setShowStampViewer(false);
            setSelectedContractId(null);
          }}
        />
      )}
    </div>
  );
};

export default ContractManagement;