import React, { useState, useRef, useEffect } from 'react';
import { Upload, Eye, Download, X, FileText, Plus, Trash2, Shield, Lock, Cloud } from 'lucide-react';
import {
  uploadDigitalDocument,
  getUserDigitalDocuments,
  downloadDigitalDocument,
  deleteDigitalDocument,
  DigitalDocument
} from '../../api/digitalDocuments';
import { formatDate } from '../../utils/regionalFormatters';

interface Document {
  id: string;
  name: string;
  description?: string;
  file?: File | null;
  uploadDate: string;
  fileUrl?: string;
  fileName: string;
  fileType: string;
  fileAvailable?: boolean;
  fileData?: string; // Base64 encoded file data
  user_id?: string;
  uploaded_at?: string;
  updated_at?: string;
  file_size?: number;
  is_password_protected?: boolean;
  password_hint?: string;
}

interface DigitalVaultProps {
  showHero?: boolean;
}

export function DigitalVault({ showHero = true }: DigitalVaultProps) {
  const [documents, setDocuments] = useState<Document[]>([]);
  const [showUploadModal, setShowUploadModal] = useState(false);
  const [documentName, setDocumentName] = useState('');
  const [documentDescription, setDocumentDescription] = useState('');
  const [documentPassword, setDocumentPassword] = useState('');
  const [documentPasswordHint, setDocumentPasswordHint] = useState('');
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [showSuccessMessage, setShowSuccessMessage] = useState(false);
  const [lastUploadedId, setLastUploadedId] = useState<string | null>(null);
  const [showPasswordModal, setShowPasswordModal] = useState(false);
  const [passwordModalDocument, setPasswordModalDocument] = useState<Document | null>(null);
  const [passwordModalAction, setPasswordModalAction] = useState<'view' | 'download'>('view');
  const [enteredPassword, setEnteredPassword] = useState('');
  const fileInputRef = useRef<HTMLInputElement>(null);

  // Load documents from API on component mount
  useEffect(() => {
    const loadDocuments = async () => {
      try {
        const apiDocuments = await getUserDigitalDocuments();
        // Convert API response to component format
        const formattedDocuments = apiDocuments.map((doc: DigitalDocument) => ({
          id: doc.id,
          name: doc.name,
          description: doc.description,
          file: null, // File object not available from API
          uploadDate: formatDate(doc.uploaded_at),
          fileUrl: undefined, // Will be created when viewing/downloading
          fileName: doc.file_name,
          fileType: doc.file_type,
          fileAvailable: true, // Available from API
          user_id: doc.user_id,
          uploaded_at: doc.uploaded_at,
          updated_at: doc.updated_at,
          file_size: doc.file_size,
          is_password_protected: doc.is_password_protected,
          password_hint: doc.password_hint
        }));
        setDocuments(formattedDocuments);
      } catch (error) {
        console.error('Error loading documents from API:', error);
      }
    };

    loadDocuments();
  }, []);

  // Prevent body scroll when modal is open
  useEffect(() => {
    if (showUploadModal) {
      document.body.style.overflow = 'hidden';
    } else {
      document.body.style.overflow = 'unset';
    }
    return () => {
      document.body.style.overflow = 'unset';
    };
  }, [showUploadModal]);


  const handleFileSelect = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      setSelectedFile(file);
    }
  };

  const handleUpload = async () => {
    if (!selectedFile || !documentName.trim()) {
      alert('Please provide a document name and select a file');
      return;
    }

    // Validate password if provided
    if (documentPassword && documentPassword.length < 6) {
      alert('Password must be at least 6 characters long');
      return;
    }

    try {
      const result = await uploadDigitalDocument(
        documentName.trim(),
        documentDescription.trim(),
        selectedFile,
        documentPassword || undefined,
        documentPasswordHint || undefined
      );

      if (result.success && result.document) {
        // Convert API response to component format
        const newDocument: Document = {
          id: result.document.id,
          name: result.document.name,
          description: result.document.description,
          file: selectedFile,
            uploadDate: formatDate(result.document.uploaded_at),
          fileUrl: URL.createObjectURL(selectedFile),
          fileName: result.document.file_name,
          fileType: result.document.file_type,
          fileAvailable: true,
          user_id: result.document.user_id,
          uploaded_at: result.document.uploaded_at,
          updated_at: result.document.updated_at,
          file_size: result.document.file_size,
          is_password_protected: result.document.is_password_protected,
          password_hint: result.document.password_hint
        };

        setDocuments(prev => [...prev, newDocument]);
        setLastUploadedId(result.document.id);
        setShowSuccessMessage(true);

        // Reset form
        setDocumentName('');
        setDocumentDescription('');
        setDocumentPassword('');
        setDocumentPasswordHint('');
        setSelectedFile(null);
        setShowUploadModal(false);

        if (fileInputRef.current) {
          fileInputRef.current.value = '';
        }

        // Hide success message after 3 seconds
        setTimeout(() => {
          setShowSuccessMessage(false);
          setLastUploadedId(null);
        }, 3000);
      } else {
        alert(`Upload failed: ${result.error || 'Unknown error'}`);
      }
    } catch (error) {
      console.error('Upload error:', error);
      alert('Failed to upload document. Please try again.');
    }
  };

  const handleView = async (document: Document) => {
    // Check if document is password protected
    if (document.is_password_protected) {
      setPasswordModalDocument(document);
      setPasswordModalAction('view');
      setShowPasswordModal(true);
      return;
    }

    await performView(document);
  };

  const performView = async (document: Document, password?: string) => {
    try {
      // Fetch the PDF document
      const token = localStorage.getItem('access_token');
      const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8090';
      
      let response;
      
      if (password) {
        // Use POST with form data for password-protected documents
        const formData = new FormData();
        formData.append('password', password);
        
        response = await fetch(`${API_BASE_URL}/api/digital-documents/${document.id}/download`, {
          method: 'POST',
          headers: {
            ...(token && { 'Authorization': `Bearer ${token}` }),
          },
          body: formData,
        });
      } else {
        // Use GET for non-password-protected documents
        response = await fetch(`${API_BASE_URL}/api/digital-documents/${document.id}/download`, {
          method: 'GET',
          headers: {
            ...(token && { 'Authorization': `Bearer ${token}` }),
          },
        });
      }

      if (!response.ok) {
        throw new Error(`Failed to fetch document: ${response.status}`);
      }

      // Create blob from response
      const blob = await response.blob();
      const viewUrl = URL.createObjectURL(blob);

      // Open in new window with specific features to force inline display
      const viewWindow = window.open('', '_blank', 'width=900,height=700,scrollbars=yes,resizable=yes,status=yes');

      if (viewWindow) {
        // Use object element for better document display
        viewWindow.document.write(`
          <!DOCTYPE html>
          <html>
            <head>
              <title>Document Viewer - ${document.name}</title>
              <style>
                body {
                  margin: 0;
                  padding: 10px;
                  background: #ffffff;
                  font-family: Arial, sans-serif;
                }
                .header {
                  background: #f8f9fa;
                  padding: 10px;
                  border-bottom: 1px solid #dee2e6;
                  font-size: 14px;
                  color: #495057;
                }
                object {
                  width: 100%;
                  height: calc(100vh - 60px);
                  border: none;
                }
                .loading {
                  display: flex;
                  justify-content: center;
                  align-items: center;
                  height: 200px;
                  font-size: 16px;
                  color: #6c757d;
                }
              </style>
            </head>
            <body>
              <div class="header">
                <strong>Viewing:</strong> ${document.name}
              </div>
              <div class="loading">Loading document...</div>
              <object data="${viewUrl}" type="application/pdf">
                <p>Your browser does not support viewing this document.
                   <a href="${viewUrl}" download="${document.fileName || document.name}.pdf">Click here to download</a>
                </p>
              </object>
            </body>
          </html>
        `);
        viewWindow.document.close();

        // Clean up the URL after a longer delay to ensure document loads
        setTimeout(() => {
          URL.revokeObjectURL(viewUrl);
        }, 10000);
      } else {
        alert('Please allow popups to view documents, or use the download button instead.');
      }
    } catch (error) {
      console.error('View failed:', error);
      alert('Unable to view document. Please try again.');
    }
  };

  const handleDownload = async (doc: Document) => {
    console.log('Download button clicked for document:', doc.id, doc.name);
    
    // Check if document is password protected
    if (doc.is_password_protected) {
      setPasswordModalDocument(doc);
      setPasswordModalAction('download');
      setShowPasswordModal(true);
      return;
    }

    await performDownload(doc);
  };

  const performDownload = async (doc: Document, password?: string) => {
    try {
      // Download document as PDF directly from API
      await downloadDigitalDocument(doc.id, password);
      console.log('Download completed successfully');
    } catch (error) {
      console.error('Download failed:', error);
      alert('Download failed. Please try again.');
    }
  };

  const handleDelete = async (documentId: string) => {
    if (window.confirm('Are you sure you want to delete this document? This action cannot be undone.')) {
      try {
        const result = await deleteDigitalDocument(documentId);
        if (result.success) {
          // Remove from local state
          setDocuments(prev => prev.filter(doc => doc.id !== documentId));
          // Reload documents from API to ensure sync with backend
          const apiDocuments = await getUserDigitalDocuments();
          const formattedDocuments = apiDocuments.map((doc: DigitalDocument) => ({
            id: doc.id,
            name: doc.name,
            description: doc.description,
            file: null,
            uploadDate: formatDate(doc.uploaded_at),
            fileUrl: undefined,
            fileName: doc.file_name,
            fileType: doc.file_type,
            fileAvailable: true,
            user_id: doc.user_id,
            uploaded_at: doc.uploaded_at,
            updated_at: doc.updated_at,
            file_size: doc.file_size,
            is_password_protected: doc.is_password_protected,
            password_hint: doc.password_hint
          }));
          setDocuments(formattedDocuments);
        } else {
          alert('Failed to delete document. Please try again.');
        }
      } catch (error) {
        console.error('Delete failed:', error);
        alert('Failed to delete document. Please try again.');
      }
    }
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  const handlePasswordSubmit = async () => {
    if (!enteredPassword.trim()) {
      alert('Please enter a password');
      return;
    }

    if (!passwordModalDocument) return;

    try {
      if (passwordModalAction === 'view') {
        await performView(passwordModalDocument, enteredPassword);
      } else {
        await performDownload(passwordModalDocument, enteredPassword);
      }
      
      // Close modal and reset
      setShowPasswordModal(false);
      setPasswordModalDocument(null);
      setEnteredPassword('');
    } catch (error) {
      console.error('Password verification failed:', error);
      alert('Invalid password or access denied');
    }
  };

  return (
    <>
      {showHero && <style>{`
        .vault-hero {
          background: linear-gradient(135deg, #1e40af 0%, #3b82f6 50%, #60a5fa 100%) !important;
          padding: 6rem 0 !important;
          color: white !important;
          text-align: center !important;
        }
        .vault-hero h1 {
          font-size: 3.5rem !important;
          font-weight: 700 !important;
          margin-bottom: 1.5rem !important;
          color: white !important;
          letter-spacing: -0.025em !important;
        }
        .vault-hero p {
          font-size: 1.25rem !important;
          margin-bottom: 2.5rem !important;
          color: #dbeafe !important;
          max-width: 50rem !important;
          margin-left: auto !important;
          margin-right: auto !important;
          line-height: 1.7 !important;
        }
      `}</style>}
      <div className="min-h-screen bg-gray-50">
        {/* Hero Section */}
        {showHero && <div className="vault-hero">
          <div className="max-w-5xl mx-auto px-4">
            <div className="inline-flex items-center gap-2 bg-blue-500/20 text-blue-100 px-4 py-2 rounded-full text-sm font-medium mb-6">
              <Shield className="w-4 h-4" />
              Secure Document Storage
            </div>
            <h1>Digital Vault</h1>
            <p>
              Store, manage, and access your important documents securely in the cloud. 
              Bank-grade encryption ensures your files are protected while remaining easily accessible whenever you need them.
            </p>
          </div>
        </div>}

        <div className="max-w-7xl mx-auto px-6 py-8">
      <div className="space-y-6">
      {/* Header */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <div className="flex items-center justify-between">
          <div>
            <h2 className="text-2xl font-bold text-gray-900">Digital Vault</h2>
            <p className="text-gray-600">Secure document storage and management</p>
          </div>
          <button
            onClick={() => setShowUploadModal(true)}
            className="flex items-center space-x-2 px-6 py-3 text-white rounded-lg transition-colors shadow-lg font-medium"
            style={{backgroundColor: 'rgb(0, 86, 210)'}}
            onMouseEnter={(e) => (e.currentTarget as HTMLElement).style.backgroundColor = 'rgb(0, 70, 180)'}
            onMouseLeave={(e) => (e.currentTarget as HTMLElement).style.backgroundColor = 'rgb(0, 86, 210)'}
          >
            <Plus className="w-5 h-5" />
            <span>Upload Document</span>
          </button>
        </div>

        {/* Success Message */}
        {showSuccessMessage && (
          <div className="mt-4 p-4 bg-green-50 border border-green-200 rounded-lg">
            <div className="flex items-center">
              <div className="flex-shrink-0">
                <svg className="h-5 w-5 text-green-400" viewBox="0 0 20 20" fill="currentColor">
                  <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                </svg>
              </div>
              <div className="ml-3">
                <p className="text-sm font-medium text-green-800">
                  Document uploaded successfully! The entered field data has been stored and is now displayed below.
                </p>
              </div>
            </div>
          </div>
        )}
      </div>

      {/* Documents List */}
      <div className="bg-white rounded-xl shadow-sm overflow-hidden">
        {documents.length === 0 ? (
          <div className="p-12 text-center">
            <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No documents uploaded yet</h3>
            <p className="text-gray-600 mb-6">Click the "Upload Document" button to add your first document</p>
            <button
              onClick={() => setShowUploadModal(true)}
              className="flex items-center space-x-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors mx-auto"
            >
              <Upload className="w-4 h-4" />
              <span>Upload First Document</span>
            </button>
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Document Name
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Description
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    File Size
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Upload Date
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Actions
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {documents.map((document) => (
                  <tr
                    key={document.id}
                    className={`hover:bg-gray-50 ${document.id === lastUploadedId ? 'bg-green-50 border-l-4 border-green-500' : ''}`}
                  >
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex items-center">
                        <FileText className="w-8 h-8 text-blue-500 mr-3" />
                        <div>
                          <div className="text-sm font-semibold text-gray-900">{document.name}</div>
                          <div className="text-sm text-gray-500">
                            {document.file ? document.file.name : (document.fileName || 'File not available')}
                          </div>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm text-gray-900 max-w-xs truncate">
                        {document.description || 'No description'}
                        {document.is_password_protected && (
                          <div className="flex items-center mt-1 text-xs text-amber-600">
                            <Lock className="w-3 h-3 mr-1" />
                            Password Protected
                          </div>
                        )}
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="text-sm text-gray-900">
                        {document.file_size ? formatFileSize(document.file_size) : 'N/A'}
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="text-sm text-gray-900">{document.uploadDate}</div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                      <div className="flex items-center space-x-2">
                        <button
                          onClick={() => handleView(document)}
                          className="text-blue-600 hover:text-blue-700 transition-colors"
                          title="View Document"
                        >
                          <Eye className="w-4 h-4" />
                        </button>
                        <button
                          onClick={() => handleDownload(document)}
                          className="text-green-600 hover:text-green-700 transition-colors"
                          title="Download Document"
                        >
                          <Download className="w-4 h-4" />
                        </button>
                        <button
                          onClick={() => handleDelete(document.id)}
                          className="text-red-600 hover:text-red-700 transition-colors"
                          title="Delete Document"
                        >
                          <Trash2 className="w-4 h-4" />
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>

      {/* Upload Modal */}
      {showUploadModal && (
        <div className="fixed inset-0 flex items-center justify-center z-50 p-4 overflow-hidden">
          {/* Backdrop with blur */}
          <div className="absolute inset-0 backdrop-blur-sm" onClick={() => setShowUploadModal(false)}></div>
          
          {/* Modal content - scrollable */}
          <div className="relative bg-white rounded-xl max-w-md w-full max-h-[80vh] flex flex-col shadow-lg" onClick={(e) => e.stopPropagation()}>
            <div className="flex-shrink-0 p-6 border-b">
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-semibold text-gray-900">Upload Document</h3>
                <button
                  onClick={() => setShowUploadModal(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <X size={24} />
                </button>
              </div>
            </div>
            
            <div className="flex-1 overflow-y-auto" style={{maxHeight: 'calc(80vh - 100px)'}}>
              <div className="p-6">
                <form onSubmit={(e) => { e.preventDefault(); handleUpload(); }} className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Document Name *
                </label>
                <input
                  type="text"
                  value={documentName}
                  onChange={(e) => setDocumentName(e.target.value)}
                  placeholder="Enter document name"
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                  required
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Description (Optional)
                </label>
                <textarea
                  value={documentDescription}
                  onChange={(e) => setDocumentDescription(e.target.value)}
                  placeholder="Enter document description"
                  rows={3}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors resize-none"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Select File *
                </label>
                <input
                  ref={fileInputRef}
                  type="file"
                  onChange={handleFileSelect}
                  accept=".pdf,.doc,.docx,.jpg,.jpeg,.png,.txt"
                  className="hidden"
                  required
                />
                <div
                  onClick={() => fileInputRef.current?.click()}
                  className="border-2 border-dashed border-gray-300 rounded-lg p-8 text-center cursor-pointer hover:border-blue-500 transition-colors"
                >
                  <Upload className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                  <p className="text-gray-600">
                    {selectedFile ? selectedFile.name : 'Click to select a file'}
                  </p>
                  <p className="text-sm text-gray-500">Supported formats: PDF, DOC, DOCX, JPG, PNG, TXT</p>
                </div>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Password (Optional)
                </label>
                <input
                  type="password"
                  value={documentPassword}
                  onChange={(e) => setDocumentPassword(e.target.value)}
                  placeholder="Enter password to protect document (min 6 characters)"
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                  minLength={6}
                />
                {documentPassword && (
                  <p className="text-xs text-gray-500 mt-1">
                    Password must be at least 6 characters long
                  </p>
                )}
              </div>

              {documentPassword && (
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Password Hint (Optional)
                  </label>
                  <input
                    type="text"
                    value={documentPasswordHint}
                    onChange={(e) => setDocumentPasswordHint(e.target.value)}
                    placeholder="Enter a hint to help remember the password"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                    maxLength={255}
                  />
                </div>
              )}

                  <div className="flex justify-end space-x-3 pt-4">
                    <button
                      type="button"
                      onClick={() => setShowUploadModal(false)}
                      className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200 transition-colors"
                    >
                      Cancel
                    </button>
                    <button
                      type="submit"
                      style={{
                        backgroundColor: '#1e40af',
                        color: 'white',
                        padding: '12px 24px',
                        borderRadius: '8px',
                        fontSize: '18px',
                        fontWeight: 'bold',
                        border: 'none',
                        boxShadow: '0 4px 14px 0 rgba(30, 64, 175, 0.39)',
                        transition: 'all 0.2s ease',
                        cursor: 'pointer'
                      }}
                      onMouseEnter={(e) => {
                        e.currentTarget.style.backgroundColor = '#1d4ed8';
                        e.currentTarget.style.transform = 'scale(1.05)';
                      }}
                      onMouseLeave={(e) => {
                        e.currentTarget.style.backgroundColor = '#1e40af';
                        e.currentTarget.style.transform = 'scale(1)';
                      }}
                    >
                      Submit
                    </button>
                  </div>
                </form>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Password Verification Modal */}
      {showPasswordModal && passwordModalDocument && (
        <div className="fixed inset-0 flex items-center justify-center z-50 p-4 overflow-hidden">
          {/* Backdrop with blur */}
          <div className="absolute inset-0 backdrop-blur-sm" onClick={() => {
            setShowPasswordModal(false);
            setPasswordModalDocument(null);
            setEnteredPassword('');
          }}></div>
          
          {/* Modal content */}
          <div className="relative bg-white rounded-xl max-w-md w-full shadow-lg" onClick={(e) => e.stopPropagation()}>
            <div className="p-6">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-gray-900">
                  Password Required
                </h3>
                <button
                  onClick={() => {
                    setShowPasswordModal(false);
                    setPasswordModalDocument(null);
                    setEnteredPassword('');
                  }}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <X size={24} />
                </button>
              </div>
              
              <div className="space-y-4">
                <div>
                  <p className="text-sm text-gray-600 mb-2">
                    This document is password protected. Please enter the password to {passwordModalAction} it.
                  </p>
                  <p className="text-sm font-medium text-gray-900 mb-3">
                    Document: {passwordModalDocument.name}
                  </p>
                  {passwordModalDocument.password_hint && (
                    <p className="text-xs text-blue-600 mb-3">
                      Hint: {passwordModalDocument.password_hint}
                    </p>
                  )}
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Password
                  </label>
                  <input
                    type="password"
                    value={enteredPassword}
                    onChange={(e) => setEnteredPassword(e.target.value)}
                    placeholder="Enter document password"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                    onKeyPress={(e) => {
                      if (e.key === 'Enter') {
                        handlePasswordSubmit();
                      }
                    }}
                    autoFocus
                  />
                </div>
                
                <div className="flex justify-end space-x-3 pt-4">
                  <button
                    onClick={() => {
                      setShowPasswordModal(false);
                      setPasswordModalDocument(null);
                      setEnteredPassword('');
                    }}
                    className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200 transition-colors"
                  >
                    Cancel
                  </button>
                  <button
                    onClick={handlePasswordSubmit}
                    style={{
                      backgroundColor: '#1e40af',
                      color: 'white',
                      padding: '8px 16px',
                      borderRadius: '8px',
                      border: 'none',
                      fontWeight: '500',
                      cursor: 'pointer'
                    }}
                  >
                    {passwordModalAction === 'view' ? 'View Document' : 'Download Document'}
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
    </div>
    </div>
    </>
  );
}