// src/pages/services/LegalLoanAssistancePremium.tsx
 
import React, { useState, useRef, useEffect } from 'react';
import { ArrowRight } from 'lucide-react';
import { Button } from '../../components/ui/button';
import { useNavigate } from 'react-router-dom';
import { legalAiApi } from '../../api/legalAiApi';
 
interface ChatMessage {
  type: 'user' | 'ai';
  content: string;
  timestamp: Date;
}
 
export function LegalLoanAssistancePremium() {
  console.log('🚀 LegalLoanAssistancePremium component is loading!');
  const navigate = useNavigate();
  const [messages, setMessages] = useState<ChatMessage[]>([]);
  const [inputValue, setInputValue] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);
 
  useEffect(() => {
    console.log('✅ LegalLoanAssistancePremium component mounted!');
    window.scrollTo(0, 0);
  }, []);
 
  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };
 
  useEffect(() => {
    if (messages.length > 0) {
      scrollToBottom();
    }
  }, [messages]);
 
  const sendMessage = async () => {
    if (!inputValue.trim()) return;
 
    const userMessage: ChatMessage = {
      type: 'user',
      content: inputValue,
      timestamp: new Date()
    };
 
    setMessages(prev => [...prev, userMessage]);
    const currentInput = inputValue;
    setInputValue('');
    setIsLoading(true);
 
    try {
      const response = await legalAiApi.askLegalQuestion({
        question: currentInput,
        property_type: 'residential',
        location: 'India'
      });
 
      const aiMessage: ChatMessage = {
        type: 'ai',
        content: response.advice || 'Sorry, I could not process your question.',
        timestamp: new Date()
      };
 
      setMessages(prev => [...prev, aiMessage]);
    } catch (error) {
      const errorMessage: ChatMessage = {
        type: 'ai',
        content: 'Sorry, there was an error processing your question. Please try again.',
        timestamp: new Date()
      };
      setMessages(prev => [...prev, errorMessage]);
    } finally {
      setIsLoading(false);
    }
  };
 
  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  };
 
  const quickQuestions = [
    "What documents do I need for property registration?",
    "How to check property title verification?",
    "What are the legal requirements for home loan?",
    "How to calculate EMI for my home loan?",
    "What is the loan eligibility criteria?"
  ];
 
  return (
    <>
      <style>{`
        .legal-loan-hero {
          background: #2563eb !important;
          padding: 8rem 0 !important;
          color: white !important;
          text-align: center !important;
        }
        .legal-loan-hero h1 {
          font-size: 3.5rem !important;
          font-weight: 700 !important;
          margin-bottom: 1.5rem !important;
          color: white !important;
          letter-spacing: -0.025em !important;
        }
        .legal-loan-hero p {
          font-size: 1.25rem !important;
          margin-bottom: 2.5rem !important;
          color: #e0e7ff !important;
          max-width: 50rem !important;
          margin-left: auto !important;
          margin-right: auto !important;
          line-height: 1.7 !important;
        }
        .legal-loan-cta {
          padding: 8rem 2rem !important;
          background: linear-gradient(135deg, #0f172a 0%, #1e293b 50%, #334155 100%) !important;
          color: white !important;
          text-align: center !important;
          position: relative !important;
          overflow: hidden !important;
        }
        .legal-loan-cta-card {
          background: rgba(255, 255, 255, 0.05) !important;
          backdrop-filter: blur(20px) !important;
          border: 1px solid rgba(255, 255, 255, 0.1) !important;
          border-radius: 2rem !important;
          padding: 4rem !important;
          max-width: 50rem !important;
          margin: 0 auto !important;
          position: relative !important;
          box-shadow: 0 25px 50px rgba(0, 0, 0, 0.25) !important;
        }
        .legal-loan-cta h2 {
          font-size: 2.5rem !important;
          font-weight: 700 !important;
          margin-bottom: 1.5rem !important;
          color: white !important;
        }
        .legal-loan-cta p {
          font-size: 1.25rem !important;
          margin-bottom: 2.5rem !important;
          color: #e2e8f0 !important;
        }
        .legal-loan-cta-button {
          background: linear-gradient(135deg, #1e293b 0%, #0f172a 100%) !important;
          color: white !important;
          padding: 1.25rem 3rem !important;
          font-size: 1.125rem !important;
          font-weight: 600 !important;
          border-radius: 12px !important;
          box-shadow: 0 8px 25px rgba(30, 41, 59, 0.25) !important;
          transition: all 0.3s ease !important;
          border: none !important;
          cursor: pointer !important;
        }
      `}</style>
      <div className="min-h-screen bg-white">
        {/* Hero Section */}
        <header className="legal-loan-hero">
          <div>
            <h1>Legal & Loan Assistance</h1>
            <p>
              Get instant AI-powered legal advice and loan assistance. Our intelligent assistant provides 24/7 support for property law, documentation, loan procedures, and expert guidance from documentation to disbursement.
            </p>
          </div>
        </header>
 
        {/* AI Assistant Section */}
        <section style={{ padding: '4rem 2rem', background: 'white' }}>
          <div style={{ maxWidth: '80rem', margin: '0 auto' }}>
            <div style={{ textAlign: 'center', marginBottom: '3rem' }}>
              <h2 style={{ fontSize: '2.75rem', fontWeight: '700', color: '#1e293b', marginBottom: '1rem' }}>
                🤖 AI Legal & Loan Assistant
              </h2>
              <p style={{ fontSize: '1.125rem', color: '#64748b', maxWidth: '48rem', margin: '0 auto' }}>
                Get instant answers to your legal and loan questions. Our AI assistant is available 24/7 to help with property law, documentation, and loan procedures.
              </p>
            </div>
           
            {/* Functional AI Chat */}
            <div style={{
              width: '100%',
              maxWidth: '64rem',
              margin: '0 auto',
              height: '600px',
              backgroundColor: 'white',
              borderRadius: '1rem',
              border: 'none',
              boxShadow: '0 25px 50px rgba(0, 0, 0, 0.15)',
              display: 'flex',
              flexDirection: 'column',
              overflow: 'hidden'
            }}>
              {/* Header */}
              <div style={{
                padding: '1.5rem',
                borderBottom: 'none',
                background: 'linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%)'
              }}>
                <h3 style={{
                  margin: 0,
                  fontSize: '1.25rem',
                  fontWeight: '600',
                  color: 'white',
                  display: 'flex',
                  alignItems: 'center',
                  gap: '0.5rem'
                }}>
                  🤖 AI Legal & Loan Assistant
                </h3>
                <p style={{
                  margin: '0.5rem 0 0 0',
                  fontSize: '0.875rem',
                  color: 'rgba(255, 255, 255, 0.8)'
                }}>
                  Get instant legal advice and loan assistance for property-related questions
                </p>
              </div>
 
              {/* Messages Area */}
              <div style={{
                flex: 1,
                overflowY: 'auto',
                padding: '1rem',
                display: 'flex',
                flexDirection: 'column',
                gap: '1rem'
              }}>
                {messages.length === 0 && (
                  <div style={{
                    textAlign: 'center',
                    padding: '2rem',
                    display: 'flex',
                    flexDirection: 'column',
                    alignItems: 'center',
                    justifyContent: 'center',
                    flex: 1
                  }}>
                    <div style={{ fontSize: '4rem', marginBottom: '1rem' }}>🤖</div>
                    <h3 style={{
                      fontSize: '1.125rem',
                      fontWeight: '500',
                      color: '#374151',
                      marginBottom: '0.5rem'
                    }}>
                      Welcome to AI Legal & Loan Assistant
                    </h3>
                    <p style={{
                      color: '#6b7280',
                      marginBottom: '2rem'
                    }}>
                      Ask me anything about property law, documentation, legal procedures, or loan assistance
                    </p>
                   
                    {/* Quick Questions */}
                    <div>
                      <p style={{
                        fontSize: '0.875rem',
                        fontWeight: '500',
                        color: '#374151',
                        marginBottom: '1rem'
                      }}>
                        Quick Questions:
                      </p>
                      <div style={{
                        display: 'grid',
                        gap: '0.5rem',
                        maxWidth: '28rem'
                      }}>
                        {quickQuestions.map((question, index) => (
                          <button
                            key={index}
                            onClick={() => setInputValue(question)}
                            style={{
                              textAlign: 'left',
                              padding: '0.75rem',
                              fontSize: '0.875rem',
                              backgroundColor: '#dbeafe',
                              border: 'none',
                              borderRadius: '0.5rem',
                              cursor: 'pointer'
                            }}
                          >
                            {question}
                          </button>
                        ))}
                      </div>
                    </div>
                  </div>
                )}
               
                {messages.map((message, index) => (
                  <div
                    key={index}
                    style={{
                      display: 'flex',
                      gap: '0.75rem',
                      justifyContent: message.type === 'user' ? 'flex-end' : 'flex-start'
                    }}
                  >
                    {message.type === 'ai' && (
                      <div style={{
                        width: '2rem',
                        height: '2rem',
                        backgroundColor: '#dbeafe',
                        borderRadius: '50%',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        flexShrink: 0,
                        fontSize: '1rem'
                      }}>
                        🤖
                      </div>
                    )}
                   
                    <div
                      style={{
                        maxWidth: '70%',
                        padding: '0.75rem 1rem',
                        borderRadius: '0.75rem',
                        backgroundColor: message.type === 'user' ? '#2563eb' : '#f3f4f6',
                        color: message.type === 'user' ? 'white' : '#374151'
                      }}
                    >
                      <div
                        style={{ fontSize: '0.875rem', whiteSpace: 'pre-wrap' }}
                        dangerouslySetInnerHTML={{
                          __html: message.content
                            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
                            .replace(/\n/g, '<br/>')
                        }}
                      />
                      <p style={{
                        fontSize: '0.75rem',
                        marginTop: '0.25rem',
                        margin: 0,
                        color: message.type === 'user' ? 'rgba(255,255,255,0.7)' : '#6b7280'
                      }}>
                        {message.timestamp.toLocaleTimeString()}
                      </p>
                    </div>
                   
                    {message.type === 'user' && (
                      <div style={{
                        width: '2rem',
                        height: '2rem',
                        backgroundColor: '#e5e7eb',
                        borderRadius: '50%',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        flexShrink: 0,
                        fontSize: '1rem'
                      }}>
                        👤
                      </div>
                    )}
                  </div>
                ))}
               
                {isLoading && (
                  <div style={{ display: 'flex', gap: '0.75rem', justifyContent: 'flex-start' }}>
                    <div style={{
                      width: '2rem',
                      height: '2rem',
                      backgroundColor: '#dbeafe',
                      borderRadius: '50%',
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'center',
                      flexShrink: 0
                    }}>
                      🤖
                    </div>
                    <div style={{
                      backgroundColor: '#f3f4f6',
                      color: '#374151',
                      maxWidth: '70%',
                      padding: '0.75rem 1rem',
                      borderRadius: '0.75rem'
                    }}>
                      <div style={{ display: 'flex', alignItems: 'center', gap: '0.5rem' }}>
                        <span style={{ fontSize: '0.875rem' }}>Thinking...</span>
                      </div>
                    </div>
                  </div>
                )}
               
                <div ref={messagesEndRef} />
              </div>
             
              {/* Input Area */}
              <div style={{
                padding: '1rem',
                borderTop: '1px solid #e2e8f0',
                backgroundColor: '#f8fafc'
              }}>
                <div style={{ display: 'flex', gap: '0.5rem' }}>
                  <input
                    type="text"
                    value={inputValue}
                    onChange={(e) => setInputValue(e.target.value)}
                    placeholder="Ask your legal or loan question..."
                    onKeyPress={handleKeyPress}
                    disabled={isLoading}
                    style={{
                      flex: 1,
                      padding: '12px 16px',
                      border: '1px solid #d1d5db',
                      borderRadius: '12px',
                      fontSize: '1rem',
                      outline: 'none',
                      transition: 'all 0.2s ease'
                    }}
                  />
                  <button
                    onClick={sendMessage}
                    disabled={isLoading || !inputValue.trim()}
                    style={{
                      padding: '12px',
                      background: 'linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%)',
                      color: 'white',
                      border: 'none',
                      borderRadius: '12px',
                      cursor: 'pointer',
                      fontSize: '1.25rem',
                      opacity: (isLoading || !inputValue.trim()) ? 0.5 : 1,
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'center',
                      boxShadow: '0 4px 12px rgba(37, 99, 235, 0.3)',
                      transition: 'all 0.2s ease'
                    }}
                  >
                    ➤
                  </button>
                </div>
                <p style={{
                  fontSize: '0.75rem',
                  color: '#6b7280',
                  marginTop: '0.5rem',
                  margin: '0.5rem 0 0 0'
                }}>
                  Press Enter to send • This is AI-generated advice, consult a lawyer for complex matters
                </p>
              </div>
            </div>
          </div>
        </section>
 
        {/* CTA Section */}
        <section className="legal-loan-cta">
          <div className="legal-loan-cta-card">
            <h2>Get Expert Assistance</h2>
            <p>
              Professional legal and loan support for your property transaction
            </p>
            <Button className="legal-loan-cta-button">
              Start Your Application
              <ArrowRight className="w-5 h-5 ml-2" />
            </Button>
          </div>
        </section>
      </div>
    </>
  );
}