import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { TrendingUp, MapPin, BarChart3, Target, ArrowRight, Search, Building, DollarSign, Star, CheckCircle, Users, Award, AlertCircle } from 'lucide-react';
import { Button } from '../../components/ui/button';
import { api } from '../../utils/api';

interface MarketData {
  locality: string;
  avgPrice: number;
  pricePerSqft: number;
  totalProperties: number;
  volatilityStatus: string;
  simpleMessage: string;
  ctaAction: string;
}

interface BuyerInputs {
  city: string;
  locality: string;
  propertyType: string;
  budgetMin: string;
  budgetMax: string;
  purpose: string;
  investmentHorizon: string;
  riskPreference: string;
}

export function MarketAnalysis() {
  const navigate = useNavigate();
  const [marketData, setMarketData] = useState<MarketData[]>([]);
  const [loading, setLoading] = useState(true);
  const [volatilityData, setVolatilityData] = useState<any>(null);
  const [showInputForm, setShowInputForm] = useState(false);
  const [buyerInputs, setBuyerInputs] = useState<BuyerInputs>({
    city: '',
    locality: '',
    propertyType: '',
    budgetMin: '',
    budgetMax: '',
    purpose: '',
    investmentHorizon: '',
    riskPreference: ''
  });
  const [personalizedInsights, setPersonalizedInsights] = useState<any>(null);

  useEffect(() => {
    fetchMarketData();
  }, []);

  const fetchMarketData = async () => {
    try {
      setLoading(true);
      const properties = await api.fetchProperties();
      
      // Group by locality and calculate metrics
      const localityMap = new Map<string, any[]>();
      properties.forEach((prop: any) => {
        const locality = prop.locality || prop.city || 'Unknown';
        if (!localityMap.has(locality)) {
          localityMap.set(locality, []);
        }
        localityMap.get(locality)?.push(prop);
      });

      // Calculate market data for each locality
      const data: MarketData[] = [];
      for (const [locality, props] of localityMap.entries()) {
        const totalPrice = props.reduce((sum, p) => sum + (p.sale_price || 0), 0);
        const avgPrice = totalPrice / props.length;
        const totalArea = props.reduce((sum, p) => sum + (p.built_up_area || 0), 0);
        const pricePerSqft = totalArea > 0 ? totalPrice / totalArea : 0;

        data.push({
          locality,
          avgPrice,
          pricePerSqft,
          totalProperties: props.length,
          volatilityStatus: 'Stable',
          simpleMessage: 'Market currently stable',
          ctaAction: 'Good time to buy'
        });
      }

      // Sort by total properties and take top 3
      const topLocalities = data.sort((a, b) => b.totalProperties - a.totalProperties).slice(0, 3);
      setMarketData(topLocalities);

      // Fetch volatility data for first locality
      if (topLocalities.length > 0) {
        try {
          const response = await fetch(`http://localhost:8090/api/buyer-market-volatility/safety-check?locality=${encodeURIComponent(topLocalities[0].locality)}`);
          if (response.ok) {
            const result = await response.json();
            if (result.success) {
              setVolatilityData(result.data);
            }
          }
        } catch (error) {
          console.error('Error fetching volatility data:', error);
        }
      }
    } catch (error) {
      console.error('Error fetching market data:', error);
    } finally {
      setLoading(false);
    }
  };

  const formatPrice = (price: number) => {
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(1)}Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(1)}L`;
    return `₹${price.toLocaleString()}`;
  };

  const handleInputChange = (field: keyof BuyerInputs, value: string) => {
    setBuyerInputs(prev => ({ ...prev, [field]: value }));
  };

  const generatePersonalizedInsights = async () => {
    if (!buyerInputs.city || !buyerInputs.propertyType) return;
    
    const filtered = marketData.filter(d => 
      d.locality.toLowerCase().includes(buyerInputs.city.toLowerCase()) ||
      d.locality.toLowerCase().includes(buyerInputs.locality.toLowerCase())
    );

    if (filtered.length > 0) {
      const avgPrice = filtered.reduce((sum, d) => sum + d.avgPrice, 0) / filtered.length;
      const avgPricePerSqft = filtered.reduce((sum, d) => sum + d.pricePerSqft, 0) / filtered.length;
      
      const totalProps = filtered.reduce((sum, d) => sum + d.totalProperties, 0);
      const growthScore = Math.min(10, (avgPricePerSqft / 10000) * 10);
      const investmentScore = Math.min(10, (totalProps / 10) + 5);
      const riskScore = Math.max(1, 10 - investmentScore);
      const demandScore = Math.min(10, (totalProps / 5) + 3);
      const liquidityScore = Math.min(10, (totalProps / 8) + 4);
      const livabilityScore = Math.min(10, (filtered.length * 2) + 5);
      
      setPersonalizedInsights({
        matchedLocalities: filtered,
        avgPrice,
        avgPricePerSqft,
        growthScore: Number(growthScore.toFixed(1)),
        investmentScore: Number(investmentScore.toFixed(1)),
        riskScore: Number(riskScore.toFixed(1)),
        demandScore: Number(demandScore.toFixed(1)),
        liquidityScore: Number(liquidityScore.toFixed(1)),
        livabilityScore: Number(livabilityScore.toFixed(1))
      });
    }
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="relative bg-gradient-to-br from-blue-600 via-blue-700 to-blue-800 text-white overflow-hidden">
        <div className="absolute inset-0 bg-black/20"></div>
        <div className="relative max-w-7xl mx-auto px-4 py-20 sm:py-24">
          <div className="text-center">
            <div className="flex justify-center mb-6">
              <div className="bg-white/10 backdrop-blur-sm rounded-full p-4">
                <BarChart3 className="w-12 h-12 text-white" />
              </div>
            </div>
            <h1 className="text-4xl sm:text-6xl font-bold mb-6 leading-tight">
              Data-Driven Real Estate
              <span className="block text-blue-200">Intelligence Platform</span>
            </h1>
            <p className="text-xl text-blue-100 mb-8 max-w-3xl mx-auto leading-relaxed">
              Make informed property decisions with comprehensive market analysis. Understand trends, identify opportunities, and evaluate risks before you invest.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center mb-12">
              <Button 
                onClick={() => setShowInputForm(true)}
                className="bg-white text-blue-700 hover:bg-gray-100 px-8 py-4 text-lg font-semibold rounded-lg transition-all duration-300 transform hover:scale-105"
              >
                <Search className="w-5 h-5 mr-2" />
                Get Market Insights
              </Button>
              <Button 
                onClick={() => navigate('/services/market-analysis-premium')}
                variant="outline" 
                className="border-white text-white hover:bg-white hover:text-blue-700 px-8 py-4 text-lg font-semibold rounded-lg transition-all duration-300"
              >
                <Target className="w-5 h-5 mr-2" />
                Request Advisory
              </Button>
            </div>
            
            {/* Stats */}
            <div className="grid grid-cols-2 md:grid-cols-4 gap-6 max-w-4xl mx-auto">
              <div className="text-center">
                <div className="text-3xl font-bold text-white mb-1">
                  {loading ? '...' : marketData.reduce((sum, d) => sum + d.totalProperties, 0)}
                </div>
                <div className="text-blue-200 text-sm">Properties Analyzed</div>
              </div>
              <div className="text-center">
                <div className="text-3xl font-bold text-white mb-1">
                  {loading ? '...' : marketData.length}
                </div>
                <div className="text-blue-200 text-sm">Localities Covered</div>
              </div>
              <div className="text-center">
                <div className="text-3xl font-bold text-white mb-1">
                  {loading ? '...' : marketData.length > 0 ? `₹${Math.round(marketData.reduce((sum, d) => sum + d.pricePerSqft, 0) / marketData.length / 1000)}K` : '...'}
                </div>
                <div className="text-blue-200 text-sm">Avg Price/Sq.ft</div>
              </div>
              <div className="text-center">
                <div className="text-3xl font-bold text-white mb-1">
                  {loading ? '...' : volatilityData ? volatilityData.volatility_status : 'Stable'}
                </div>
                <div className="text-blue-200 text-sm">Market Status</div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Value Proposition Section */}
      <div className="py-20 bg-white">
        <div className="max-w-7xl mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">Why Market Analysis Matters</h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">Transform property buying from guesswork into strategic, data-backed decision-making</p>
          </div>
          
          <div className="grid md:grid-cols-3 gap-8">
            {[
              {
                icon: TrendingUp,
                title: "Understand Price Trends",
                description: "Analyze historical data and market cycles to predict future price movements and identify the right time to buy."
              },
              {
                icon: Target,
                title: "Identify Growth Areas",
                description: "Discover high-potential localities with strong infrastructure development and investment opportunities."
              },
              {
                icon: AlertCircle,
                title: "Evaluate Risks",
                description: "Assess market volatility, legal risks, and oversupply concerns before making your investment decision."
              },
              {
                icon: MapPin,
                title: "Compare Localities",
                description: "Side-by-side comparison of multiple areas based on price, growth rate, rental yield, and infrastructure."
              },
              {
                icon: DollarSign,
                title: "Predict Returns",
                description: "Estimate capital appreciation and rental income potential to calculate your total return on investment."
              },
              {
                icon: BarChart3,
                title: "Personalized Insights",
                description: "Get customized recommendations based on your budget, purpose, investment horizon, and risk preference."
              }
            ].map((item, index) => {
              const IconComponent = item.icon;
              return (
                <div key={index} className="bg-gray-50 rounded-xl p-6 hover:shadow-lg transition-all duration-300">
                  <div className="bg-[#2B256D]/10 rounded-lg p-3 w-fit mb-4">
                    <IconComponent className="w-8 h-8 text-[#2B256D]" />
                  </div>
                  <h3 className="text-xl font-bold text-gray-900 mb-3">{item.title}</h3>
                  <p className="text-gray-600">{item.description}</p>
                </div>
              );
            })}
          </div>
        </div>
      </div>

      {/* Buyer Input Section */}
      {showInputForm && (
        <div className="py-20 bg-gray-50">
          <div className="max-w-4xl mx-auto px-4">
            <div className="bg-white rounded-2xl shadow-xl p-8">
              <h2 className="text-3xl font-bold text-gray-900 mb-2">Get Personalized Market Insights</h2>
              <p className="text-gray-600 mb-8">Tell us about your requirements to receive customized market analysis</p>
              
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">City *</label>
                  <input
                    type="text"
                    value={buyerInputs.city}
                    onChange={(e) => handleInputChange('city', e.target.value)}
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
                    placeholder="e.g., Bangalore"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Locality (Optional)</label>
                  <input
                    type="text"
                    value={buyerInputs.locality}
                    onChange={(e) => handleInputChange('locality', e.target.value)}
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
                    placeholder="e.g., Whitefield"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Property Type *</label>
                  <select
                    value={buyerInputs.propertyType}
                    onChange={(e) => handleInputChange('propertyType', e.target.value)}
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
                  >
                    <option value="">Select type</option>
                    <option value="apartment">Apartment</option>
                    <option value="villa">Villa</option>
                    <option value="plot">Plot</option>
                    <option value="commercial">Commercial</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Purpose</label>
                  <select
                    value={buyerInputs.purpose}
                    onChange={(e) => handleInputChange('purpose', e.target.value)}
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
                  >
                    <option value="">Select purpose</option>
                    <option value="self-use">Self Use</option>
                    <option value="investment">Investment</option>
                    <option value="rental">Rental Income</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Budget Min</label>
                  <input
                    type="text"
                    value={buyerInputs.budgetMin}
                    onChange={(e) => handleInputChange('budgetMin', e.target.value)}
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
                    placeholder="e.g., 5000000"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Budget Max</label>
                  <input
                    type="text"
                    value={buyerInputs.budgetMax}
                    onChange={(e) => handleInputChange('budgetMax', e.target.value)}
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
                    placeholder="e.g., 10000000"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Investment Horizon</label>
                  <select
                    value={buyerInputs.investmentHorizon}
                    onChange={(e) => handleInputChange('investmentHorizon', e.target.value)}
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
                  >
                    <option value="">Select horizon</option>
                    <option value="short-term">Short-term (1-3 years)</option>
                    <option value="mid-term">Mid-term (3-5 years)</option>
                    <option value="long-term">Long-term (5+ years)</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Risk Preference</label>
                  <select
                    value={buyerInputs.riskPreference}
                    onChange={(e) => handleInputChange('riskPreference', e.target.value)}
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
                  >
                    <option value="">Select risk level</option>
                    <option value="low">Low Risk</option>
                    <option value="medium">Medium Risk</option>
                    <option value="high">High Risk</option>
                  </select>
                </div>
              </div>
              
              <div className="mt-8 flex gap-4">
                <Button
                  onClick={generatePersonalizedInsights}
                  disabled={!buyerInputs.city || !buyerInputs.propertyType}
                  className="bg-[#2B256D] hover:bg-[#1f1a5a] text-white px-8 py-3 rounded-lg font-semibold"
                >
                  Generate Insights
                </Button>
                <Button
                  onClick={() => setShowInputForm(false)}
                  variant="outline"
                  className="px-8 py-3 rounded-lg font-semibold"
                >
                  Cancel
                </Button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Personalized Insights Section */}
      {personalizedInsights && (
        <div className="py-20 bg-white">
          <div className="max-w-7xl mx-auto px-4">
            <h2 className="text-4xl font-bold text-gray-900 mb-8 text-center">Your Personalized Market Insights</h2>
            
            {/* Intelligence Scores */}
            <div className="grid md:grid-cols-3 lg:grid-cols-6 gap-6 mb-12">
              {[
                { label: 'Growth Score', value: personalizedInsights.growthScore, color: 'green' },
                { label: 'Investment Score', value: personalizedInsights.investmentScore, color: 'blue' },
                { label: 'Risk Score', value: personalizedInsights.riskScore, color: 'red' },
                { label: 'Demand Score', value: personalizedInsights.demandScore, color: 'purple' },
                { label: 'Liquidity Score', value: personalizedInsights.liquidityScore, color: 'orange' },
                { label: 'Livability Score', value: personalizedInsights.livabilityScore, color: 'teal' }
              ].map((score, index) => (
                <div key={index} className="bg-gray-50 rounded-xl p-6 text-center">
                  <div className="text-4xl font-bold text-[#2B256D] mb-2">{score.value}/10</div>
                  <div className="text-sm text-gray-600">{score.label}</div>
                </div>
              ))}
            </div>

            {/* Market Snapshot */}
            <div className="grid md:grid-cols-3 gap-6 mb-12">
              <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl p-6">
                <div className="text-sm text-gray-600 mb-2">Average Price</div>
                <div className="text-3xl font-bold text-gray-900">{formatPrice(personalizedInsights.avgPrice)}</div>
              </div>
              <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl p-6">
                <div className="text-sm text-gray-600 mb-2">Price per Sq.ft</div>
                <div className="text-3xl font-bold text-gray-900">₹{Math.round(personalizedInsights.avgPricePerSqft).toLocaleString()}</div>
              </div>
              <div className="bg-gradient-to-br from-purple-50 to-pink-50 rounded-xl p-6">
                <div className="text-sm text-gray-600 mb-2">Matched Localities</div>
                <div className="text-3xl font-bold text-gray-900">{personalizedInsights.matchedLocalities.length}</div>
              </div>
            </div>

            <div className="text-center">
              <p className="text-gray-600 mb-6">Want detailed analysis with price forecasts, risk assessment, and investment recommendations?</p>
              <Button
                onClick={() => navigate('/services/market-analysis-premium')}
                className="bg-[#2B256D] hover:bg-[#1f1a5a] text-white px-8 py-4 text-lg font-semibold rounded-lg"
              >
                Get Premium Market Report
              </Button>
            </div>
          </div>
        </div>
      )}

      {/* Premium Packages Section */}
      <div className="py-20 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">Choose Your Market Intelligence Package</h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">Professional market analysis tailored to your investment needs</p>
          </div>
          
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8">
            {[
              {
                title: "City Market Report",
                price: "₹8,999",
                description: "Complete city-wide analysis with growth forecasts and investment opportunities",
                features: [
                  "15-year historical data",
                  "AI-powered 18-month forecast",
                  "Infrastructure impact analysis",
                  "Investment hotspot mapping"
                ],
                popular: false
              },
              {
                title: "Locality Analysis",
                price: "₹5,999",
                description: "Deep-dive into specific neighborhoods and micro-markets",
                features: [
                  "Micro-market price trends",
                  "Demand-supply dynamics",
                  "Rental yield projections",
                  "Risk assessment matrix"
                ],
                popular: true
              },
              {
                title: "Custom Analysis",
                price: "₹18,999",
                description: "Personalized research based on your budget and investment goals",
                features: [
                  "Tailored recommendations",
                  "ROI optimization strategy",
                  "Portfolio diversification",
                  "Expert consultation call"
                ],
                popular: false
              },
              {
                title: "Investment Advisory",
                price: "₹29,999",
                description: "Complete investment strategy with ongoing support",
                features: [
                  "Professional consultation",
                  "Market timing strategy",
                  "Risk mitigation plan",
                  "3-month support included"
                ],
                popular: false
              }
            ].map((service, index) => (
              <div key={index} className={`relative bg-white rounded-2xl border-2 p-8 transition-all duration-300 hover:shadow-2xl hover:-translate-y-1 ${
                service.popular ? 'border-[#2B256D] shadow-lg' : 'border-gray-200 hover:border-[#2B256D]'
              }`}>
                {service.popular && (
                  <div className="absolute -top-4 left-1/2 transform -translate-x-1/2">
                    <div className="bg-[#2B256D] text-white px-4 py-2 rounded-full text-sm font-semibold flex items-center">
                      <Star className="w-4 h-4 mr-1" />
                      Most Popular
                    </div>
                  </div>
                )}
                
                <div className="text-center mb-6">
                  <h3 className="text-2xl font-bold text-gray-900 mb-2">{service.title}</h3>
                  <div className="text-4xl font-bold text-[#2B256D] mb-3">{service.price}</div>
                  <p className="text-gray-600">{service.description}</p>
                </div>
                
                <ul className="space-y-3 mb-8">
                  {service.features.map((feature, idx) => (
                    <li key={idx} className="flex items-start">
                      <CheckCircle className="w-5 h-5 text-green-500 mr-3 mt-0.5 flex-shrink-0" />
                      <span className="text-gray-700">{feature}</span>
                    </li>
                  ))}
                </ul>
                
                <Button className={`w-full py-3 font-semibold rounded-lg transition-all duration-300 ${
                  service.popular 
                    ? 'bg-[#2B256D] hover:bg-[#1f1a5a] text-white' 
                    : 'bg-gray-100 hover:bg-[#2B256D] hover:text-white text-gray-900'
                }`}>
                  Order Report
                </Button>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Market Preview Section */}
      <div className="py-20 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">Market Intelligence Preview</h2>
            <p className="text-xl text-gray-600">See what insights you'll get in our comprehensive reports</p>
          </div>
          
          {/* Volatility Status Card */}
          {volatilityData && (
            <div className="bg-white rounded-2xl shadow-xl p-8 mb-8 max-w-4xl mx-auto">
              <div className="flex items-center justify-between mb-6">
                <h3 className="text-2xl font-bold text-gray-900">Market Safety Status</h3>
                <div className={`px-4 py-2 rounded-full font-semibold ${
                  volatilityData.volatility_status === 'Stable' ? 'bg-green-100 text-green-700' :
                  volatilityData.volatility_status === 'Volatile' ? 'bg-red-100 text-red-700' :
                  'bg-yellow-100 text-yellow-700'
                }`}>
                  {volatilityData.volatility_status}
                </div>
              </div>
              <p className="text-lg text-gray-700 mb-4">{volatilityData.simple_message}</p>
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-4">
                <p className="text-blue-800 font-semibold">{volatilityData.cta_action}</p>
              </div>
              {volatilityData.market_insights && (
                <div className="grid md:grid-cols-2 gap-4">
                  <div className="bg-gray-50 rounded-lg p-4">
                    <div className="text-sm text-gray-600 mb-1">Total Properties</div>
                    <div className="text-2xl font-bold text-gray-900">{volatilityData.market_insights.total_properties}</div>
                  </div>
                  <div className="bg-gray-50 rounded-lg p-4">
                    <div className="text-sm text-gray-600 mb-1">Price Range</div>
                    <div className="text-lg font-bold text-gray-900">{volatilityData.market_insights.avg_price_range}</div>
                  </div>
                </div>
              )}
            </div>
          )}
          
          {/* Sample Data Cards */}
          <div className="bg-white rounded-2xl shadow-xl p-8 mb-16 max-w-4xl mx-auto">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Live Market Snapshot</h3>
            {loading ? (
              <div className="flex items-center justify-center py-12">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-[#2B256D]"></div>
              </div>
            ) : marketData.length > 0 ? (
              <div className="grid md:grid-cols-3 gap-6">
                {marketData.map((data, index) => (
                  <div key={index} className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl p-6 text-center">
                    <div className="text-lg font-bold text-[#2B256D] mb-2">{data.locality}</div>
                    <div className="text-2xl font-bold text-gray-900 mb-1">
                      ₹{Math.round(data.pricePerSqft).toLocaleString()}/sq.ft
                    </div>
                    <div className="text-sm text-gray-600 mb-2">
                      Avg: {formatPrice(data.avgPrice)}
                    </div>
                    <div className="text-green-600 font-semibold flex items-center justify-center">
                      <TrendingUp className="w-4 h-4 mr-1" />
                      {data.totalProperties} properties
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <div className="text-center py-12">
                <AlertCircle className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <p className="text-gray-600">No market data available. Please check back later.</p>
              </div>
            )}
            <div className="text-center mt-8">
              <Button 
                onClick={() => navigate('/services/market-analysis-premium')}
                className="bg-[#2B256D] hover:bg-[#1f1a5a] text-white px-8 py-3 rounded-lg font-semibold transition-all duration-300"
              >
                View Detailed Analysis
                <ArrowRight className="w-5 h-5 ml-2" />
              </Button>
            </div>
          </div>
          
          {/* Features Grid */}
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {[
              {
                icon: TrendingUp,
                title: "Price Trend Analysis",
                features: [
                  "15-year historical data",
                  "AI-powered forecasting",
                  "Market cycle analysis",
                  "Seasonal impact study"
                ]
              },
              {
                icon: BarChart3,
                title: "Supply & Demand",
                features: [
                  "Real-time inventory",
                  "New project tracking",
                  "Buyer demand metrics",
                  "Market status indicator"
                ]
              },
              {
                icon: MapPin,
                title: "Locality Comparison",
                features: [
                  "Side-by-side analysis",
                  "Price benchmarking",
                  "Growth rate comparison",
                  "Infrastructure scoring"
                ]
              },
              {
                icon: Target,
                title: "Investment Intelligence",
                features: [
                  "ROI projections",
                  "Risk assessment",
                  "Suitability scoring",
                  "Custom recommendations"
                ]
              },
              {
                icon: Building,
                title: "Infrastructure Impact",
                features: [
                  "Metro connectivity",
                  "IT hub proximity",
                  "Smart city projects",
                  "Social infrastructure"
                ]
              },
              {
                icon: DollarSign,
                title: "Market Scores",
                features: [
                  "Growth Score (1-10)",
                  "Investment Score (1-10)",
                  "Risk Score (1-10)",
                  "Livability Score (1-10)"
                ]
              }
            ].map((item, index) => {
              const IconComponent = item.icon;
              return (
                <div key={index} className="bg-white rounded-xl p-6 shadow-lg hover:shadow-xl transition-all duration-300 hover:-translate-y-1">
                  <div className="flex items-center mb-4">
                    <div className="bg-[#2B256D]/10 rounded-lg p-3 mr-4">
                      <IconComponent className="w-6 h-6 text-[#2B256D]" />
                    </div>
                    <h3 className="text-xl font-bold text-gray-900">{item.title}</h3>
                  </div>
                  <ul className="space-y-2">
                    {item.features.map((feature, idx) => (
                      <li key={idx} className="flex items-center text-gray-600">
                        <div className="w-2 h-2 bg-[#2B256D] rounded-full mr-3"></div>
                        {feature}
                      </li>
                    ))}
                  </ul>
                </div>
              );
            })}
          </div>
        </div>
      </div>

      {/* Trust Indicators */}
      <div className="py-16 bg-white">
        <div className="max-w-7xl mx-auto px-4">
          {!loading && marketData.length > 0 && (
            <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-2xl p-8">
              <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Current Market Overview</h3>
              <div className="grid md:grid-cols-4 gap-6">
                <div className="text-center">
                  <div className="bg-white rounded-lg p-4">
                    <div className="bg-[#2B256D]/10 rounded-full p-4 w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                      <Building className="w-8 h-8 text-[#2B256D]" />
                    </div>
                    <div className="text-3xl font-bold text-gray-900 mb-2">
                      {marketData.reduce((sum, d) => sum + d.totalProperties, 0)}
                    </div>
                    <div className="text-gray-600">Active Properties</div>
                  </div>
                </div>
                <div className="text-center">
                  <div className="bg-white rounded-lg p-4">
                    <div className="bg-[#2B256D]/10 rounded-full p-4 w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                      <MapPin className="w-8 h-8 text-[#2B256D]" />
                    </div>
                    <div className="text-3xl font-bold text-gray-900 mb-2">
                      {marketData.length}
                    </div>
                    <div className="text-gray-600">Active Localities</div>
                  </div>
                </div>
                <div className="text-center">
                  <div className="bg-white rounded-lg p-4">
                    <div className="bg-[#2B256D]/10 rounded-full p-4 w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                      <DollarSign className="w-8 h-8 text-[#2B256D]" />
                    </div>
                    <div className="text-3xl font-bold text-gray-900 mb-2">
                      {formatPrice(marketData.reduce((sum, d) => sum + d.avgPrice, 0) / marketData.length)}
                    </div>
                    <div className="text-gray-600">Average Price</div>
                  </div>
                </div>
                <div className="text-center">
                  <div className="bg-white rounded-lg p-4">
                    <div className="bg-[#2B256D]/10 rounded-full p-4 w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                      <TrendingUp className="w-8 h-8 text-[#2B256D]" />
                    </div>
                    <div className="text-3xl font-bold text-gray-900 mb-2">
                      ₹{Math.round(marketData.reduce((sum, d) => sum + d.pricePerSqft, 0) / marketData.length).toLocaleString()}
                    </div>
                    <div className="text-gray-600">Avg Price/Sq.ft</div>
                  </div>
                </div>
              </div>
            </div>
          )}
        </div>
      </div>

      {/* CTA Section */}
      <div className="py-20 bg-gradient-to-r from-[#2B256D] to-[#1f1a5a] text-white">
        <div className="max-w-4xl mx-auto px-4 text-center">
          <h2 className="text-4xl font-bold mb-6">Ready to Make Smarter Investment Decisions?</h2>
          <p className="text-xl text-blue-100 mb-8 leading-relaxed">
            Join thousands of successful investors who rely on our market intelligence. Get your comprehensive market report delivered in 48 hours.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button 
              onClick={() => navigate('/services/market-analysis-premium')}
              className="bg-white text-[#2B256D] hover:bg-gray-100 px-8 py-4 text-lg font-semibold rounded-lg transition-all duration-300 transform hover:scale-105"
            >
              <Search className="w-5 h-5 mr-2" />
              Start Market Analysis
            </Button>
            <Button 
              variant="outline" 
              className="border-white text-white hover:bg-white hover:text-[#2B256D] px-8 py-4 text-lg font-semibold rounded-lg transition-all duration-300"
            >
              <Target className="w-5 h-5 mr-2" />
              Talk to Expert
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
}