import React, { useState, useEffect } from 'react';
import { AlertCircle } from 'lucide-react';
import { Button } from '../../components/ui/button';
import { useNavigate } from 'react-router-dom';
import { api } from '../../utils/api';

interface MarketData {
  locality: string;
  avgPrice: number;
  pricePerSqft: number;
  totalProperties: number;
}

export function MarketAnalysisPremium() {
  const navigate = useNavigate();
  const [selectedCity, setSelectedCity] = useState('');
  const [selectedPropertyType, setSelectedPropertyType] = useState('all');
  const [selectedBedrooms, setSelectedBedrooms] = useState('all');
  const [appliedFilters, setAppliedFilters] = useState({ city: '', type: '', bedrooms: '' });
  const [demandSupplyData, setDemandSupplyData] = useState<any>(null);
  const [priceComparison, setPriceComparison] = useState<any>(null);
  const [timeOnMarket, setTimeOnMarket] = useState<any>(null);
  const [showDetailedInsights, setShowDetailedInsights] = useState(false);
  const [marketData, setMarketData] = useState<MarketData[]>([]);
  const [loading, setLoading] = useState(true);
  const [volatilityData, setVolatilityData] = useState<any>(null);
  const [allProperties, setAllProperties] = useState<any[]>([]);
  const [cityGrowth, setCityGrowth] = useState<any[]>([]);
  const [propertyTypeData, setPropertyTypeData] = useState<any>(null);

  useEffect(() => {
    setShowDetailedInsights(true);
  }, []);

  useEffect(() => {
    if (showDetailedInsights && appliedFilters.city) {
      fetchCitySpecificData();
    }
  }, [appliedFilters, showDetailedInsights]);

  const fetchCitySpecificData = async () => {
    try {
      setLoading(true);
      const properties = await api.fetchProperties();
      
      const cityFiltered = properties.filter((prop: any) => {
        const city = (prop.city || '').toLowerCase();
        const locality = (prop.locality || '').toLowerCase();
        const searchCity = appliedFilters.city.toLowerCase();
        
        const cityMatch = !appliedFilters.city || city.includes(searchCity) || locality.includes(searchCity) ||
               (searchCity === 'bangalore' && (city.includes('bengaluru') || locality.includes('bengaluru'))) ||
               (searchCity === 'bengaluru' && (city.includes('bangalore') || locality.includes('bangalore')));
        
        const typeMatch = appliedFilters.type === 'all' || 
                         (prop.property_type || '').toLowerCase() === appliedFilters.type.toLowerCase();
        
        const bedroomMatch = appliedFilters.bedrooms === 'all' || 
                            String(prop.bedrooms) === appliedFilters.bedrooms;
        
        return cityMatch && typeMatch && bedroomMatch;
      });

      // Calculate demand-supply metrics
      const totalViews = cityFiltered.reduce((sum, p) => sum + (p.views || 0), 0);
      const totalInquiries = cityFiltered.reduce((sum, p) => sum + (p.inquiries || 0), 0);
      const avgViews = cityFiltered.length > 0 ? totalViews / cityFiltered.length : 0;
      const avgInquiries = cityFiltered.length > 0 ? totalInquiries / cityFiltered.length : 0;
      
      setDemandSupplyData({
        totalListings: cityFiltered.length,
        totalViews,
        totalInquiries,
        avgViews: Math.round(avgViews),
        avgInquiries: Math.round(avgInquiries),
        demandScore: avgViews > 50 ? 'High' : avgViews > 20 ? 'Medium' : 'Low',
        marketStatus: avgViews > 50 && cityFiltered.length < 100 ? 'Hot Market' : 
                      avgViews > 20 ? 'Normal Market' : 'Slow Market'
      });

      // Calculate price comparison
      const prices = cityFiltered.map(p => p.sale_price || 0).filter(p => p > 0);
      const avgPrice = prices.length > 0 ? prices.reduce((a, b) => a + b, 0) / prices.length : 0;
      const minPrice = prices.length > 0 ? Math.min(...prices) : 0;
      const maxPrice = prices.length > 0 ? Math.max(...prices) : 0;
      
      setPriceComparison({
        avgPrice,
        minPrice,
        maxPrice,
        priceRange: maxPrice - minPrice,
        fairPriceRange: { min: avgPrice * 0.85, max: avgPrice * 1.15 }
      });

      // Calculate time on market
      const now = new Date();
      const daysOnMarket = cityFiltered.map(p => {
        const created = new Date(p.created_at || now);
        return Math.floor((now.getTime() - created.getTime()) / (1000 * 60 * 60 * 24));
      });
      const avgDaysOnMarket = daysOnMarket.length > 0 ? 
        daysOnMarket.reduce((a, b) => a + b, 0) / daysOnMarket.length : 0;
      
      setTimeOnMarket({
        avgDays: Math.round(avgDaysOnMarket),
        fastSelling: daysOnMarket.filter(d => d < 30).length,
        slowSelling: daysOnMarket.filter(d => d > 90).length,
        status: avgDaysOnMarket < 30 ? 'Fast Moving' : avgDaysOnMarket < 60 ? 'Normal' : 'Slow Moving'
      });

      const localityMap = new Map<string, any[]>();
      cityFiltered.forEach((prop: any) => {
        const locality = prop.locality || prop.city || 'Unknown';
        if (!localityMap.has(locality)) {
          localityMap.set(locality, []);
        }
        localityMap.get(locality)?.push(prop);
      });

      const data: MarketData[] = [];
      for (const [locality, props] of localityMap.entries()) {
        const totalPrice = props.reduce((sum, p) => sum + (p.sale_price || 0), 0);
        const avgPrice = totalPrice / props.length;
        const totalArea = props.reduce((sum, p) => sum + (p.built_up_area || 0), 0);
        const pricePerSqft = totalArea > 0 ? totalPrice / totalArea : 0;

        data.push({
          locality,
          avgPrice,
          pricePerSqft,
          totalProperties: props.length
        });
      }

      const topLocalities = data.sort((a, b) => b.totalProperties - a.totalProperties).slice(0, 3);
      setMarketData(topLocalities);

      if (topLocalities.length > 0) {
        try {
          const response = await fetch(`http://localhost:8090/api/buyer-market-volatility/safety-check?locality=${encodeURIComponent(topLocalities[0].locality)}`);
          if (response.ok) {
            const result = await response.json();
            if (result.success) {
              setVolatilityData(result.data);
            }
          }
        } catch (error) {
          console.error('Error fetching volatility data:', error);
        }
      }
    } catch (error) {
      console.error('Error fetching city data:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchMarketData = async () => {
    try {
      setLoading(true);
      const properties = await api.fetchProperties();
      setAllProperties(properties);
      
      // Calculate city-wise growth
      const cityMap = new Map<string, any[]>();
      properties.forEach((prop: any) => {
        const city = (prop.city || 'Unknown').toLowerCase();
        if (!cityMap.has(city)) {
          cityMap.set(city, []);
        }
        cityMap.get(city)?.push(prop);
      });

      const cityStats = Array.from(cityMap.entries()).map(([city, props]) => {
        const avgPrice = props.reduce((sum, p) => sum + (p.sale_price || 0), 0) / props.length;
        return { city, count: props.length, avgPrice };
      }).sort((a, b) => b.count - a.count).slice(0, 3);
      setCityGrowth(cityStats);

      // Calculate property type data
      const typeMap = new Map<string, any[]>();
      properties.forEach((prop: any) => {
        const bedrooms = prop.bedrooms || 0;
        let type = 'Other';
        if (bedrooms === 1) type = '1BHK';
        else if (bedrooms === 2) type = '2BHK';
        else if (bedrooms === 3) type = '3BHK';
        else if (bedrooms >= 4) type = 'Villa';
        
        if (!typeMap.has(type)) {
          typeMap.set(type, []);
        }
        typeMap.get(type)?.push(prop);
      });

      const typeStats: any = {};
      for (const [type, props] of typeMap.entries()) {
        const prices = props.map(p => p.sale_price || 0).filter(p => p > 0);
        if (prices.length > 0) {
          typeStats[type] = {
            min: Math.min(...prices),
            max: Math.max(...prices),
            avg: prices.reduce((a, b) => a + b, 0) / prices.length,
            count: props.length
          };
        }
      }
      setPropertyTypeData(typeStats);
      
      const localityMap = new Map<string, any[]>();
      properties.forEach((prop: any) => {
        const locality = prop.locality || prop.city || 'Unknown';
        if (!localityMap.has(locality)) {
          localityMap.set(locality, []);
        }
        localityMap.get(locality)?.push(prop);
      });

      const data: MarketData[] = [];
      for (const [locality, props] of localityMap.entries()) {
        const totalPrice = props.reduce((sum, p) => sum + (p.sale_price || 0), 0);
        const avgPrice = totalPrice / props.length;
        const totalArea = props.reduce((sum, p) => sum + (p.built_up_area || 0), 0);
        const pricePerSqft = totalArea > 0 ? totalPrice / totalArea : 0;

        data.push({
          locality,
          avgPrice,
          pricePerSqft,
          totalProperties: props.length
        });
      }

      const topLocalities = data.sort((a, b) => b.totalProperties - a.totalProperties).slice(0, 3);
      setMarketData(topLocalities);

      if (topLocalities.length > 0) {
        try {
          const response = await fetch(`http://localhost:8090/api/buyer-market-volatility/safety-check?locality=${encodeURIComponent(topLocalities[0].locality)}`);
          if (response.ok) {
            const result = await response.json();
            if (result.success) {
              setVolatilityData(result.data);
            }
          }
        } catch (error) {
          console.error('Error fetching volatility data:', error);
        }
      }
    } catch (error) {
      console.error('Error fetching market data:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleSearch = () => {
    setAppliedFilters({ city: selectedCity, type: selectedPropertyType, bedrooms: selectedBedrooms });
    setShowDetailedInsights(true);
  };

  const handleClear = () => {
    setSelectedCity('');
    setSelectedPropertyType('all');
    setSelectedBedrooms('all');
    setAppliedFilters({ city: '', type: '', bedrooms: '' });
    setDemandSupplyData(null);
    setPriceComparison(null);
    setTimeOnMarket(null);
    setMarketData([]);
    setVolatilityData(null);
  };

  const formatPrice = (price: number) => {
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(1)}Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(1)}L`;
    return `₹${price.toLocaleString()}`;
  };

  return (
    <>
      <style>{`
        .market-hero {
          background: linear-gradient(135deg, #1e40af 0%, #3b82f6 100%) !important;
          padding: 4rem 0 !important;
          color: white !important;
          text-align: center !important;
        }
        .market-hero h1 {
          font-size: 3.5rem !important;
          font-weight: 700 !important;
          margin-bottom: 1.5rem !important;
          color: white !important;
          letter-spacing: -0.025em !important;
        }
        .market-hero p {
          font-size: 1.25rem !important;
          margin-bottom: 2.5rem !important;
          color: #e2e8f0 !important;
          max-width: 50rem !important;
          margin-left: auto !important;
          margin-right: auto !important;
          line-height: 1.7 !important;
        }
        .market-pricing {
          background: rgba(255, 255, 255, 0.95) !important;
          color: #1e293b !important;
          padding: 1.5rem 3rem !important;
          border-radius: 1rem !important;
          font-weight: 600 !important;
          font-size: 1.1rem !important;
          display: inline-block !important;
          box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1) !important;
          backdrop-filter: blur(10px) !important;
        }
        .market-features {
          padding: 6rem 2rem !important;
          background: #f8fafc !important;
        }
        .market-features h2 {
          font-size: 2rem !important;
          font-weight: 700 !important;
          text-align: center !important;
          margin-bottom: 4rem !important;
          color: #1e293b !important;
          letter-spacing: -0.025em !important;
        }
        .market-feature-grid {
          display: grid !important;
          grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)) !important;
          gap: 2rem !important;
          max-width: 72rem !important;
          margin: 0 auto !important;
        }
        .market-feature-card {
          background: white !important;
          padding: 0 !important;
          border-radius: 1rem !important;
          box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08) !important;
          text-align: center !important;
          transition: all 0.3s ease !important;
          border: 1px solid #e2e8f0 !important;
          overflow: hidden !important;
        }
        .market-feature-card:hover {
          transform: translateY(-8px) !important;
          box-shadow: 0 20px 40px rgba(0, 0, 0, 0.12) !important;
          border-color: #cbd5e1 !important;
        }
        .market-feature-card h3 {
          font-size: 1.5rem !important;
          font-weight: 600 !important;
          margin-bottom: 1rem !important;
          color: #1e293b !important;
        }
        .market-cta {
          padding: 8rem 2rem !important;
          background: linear-gradient(135deg, #0f172a 0%, #1e293b 50%, #334155 100%) !important;
          color: white !important;
          text-align: center !important;
          position: relative !important;
          overflow: hidden !important;
        }
        .market-cta-card {
          background: rgba(255, 255, 255, 0.05) !important;
          backdrop-filter: blur(20px) !important;
          border: 1px solid rgba(255, 255, 255, 0.1) !important;
          border-radius: 2rem !important;
          padding: 4rem !important;
          max-width: 50rem !important;
          margin: 0 auto !important;
          position: relative !important;
          box-shadow: 0 25px 50px rgba(0, 0, 0, 0.25) !important;
        }
        .market-cta h2 {
          font-size: 2.5rem !important;
          font-weight: 700 !important;
          margin-bottom: 1.5rem !important;
          color: white !important;
          letter-spacing: -0.025em !important;
        }
        .market-cta p {
          font-size: 1.25rem !important;
          margin-bottom: 2.5rem !important;
          color: #e2e8f0 !important;
          line-height: 1.6 !important;
        }
        .market-cta-button {
          background: linear-gradient(135deg, #1e293b 0%, #0f172a 100%) !important;
          color: white !important;
          padding: 1.25rem 3rem !important;
          font-size: 1.125rem !important;
          font-weight: 600 !important;
          border-radius: 12px !important;
          box-shadow: 0 8px 25px rgba(30, 41, 59, 0.25) !important;
          transition: all 0.3s ease !important;
          border: none !important;
          cursor: pointer !important;
          position: relative !important;
          overflow: hidden !important;
        }
        .market-cta-button:hover {
          transform: translateY(-2px) !important;
          box-shadow: 0 12px 35px rgba(30, 41, 59, 0.35) !important;
          background: linear-gradient(135deg, #334155 0%, #1e293b 100%) !important;
        }
      `}</style>
      <div className="min-h-screen bg-white">


        {/* Hero Section */}
        <header style={{ padding: '4rem 2rem', background: 'linear-gradient(135deg, #1e40af 0%, #3b82f6 100%)', textAlign: 'center', color: 'white' }}>
          <div style={{ maxWidth: '64rem', margin: '0 auto' }}>
            <h1 style={{ fontSize: '2.5rem', fontWeight: '700', marginBottom: '1rem', color: 'white' }}>Market Analysis – Smart Insights for Property Buyers</h1>
            <p style={{ fontSize: '1.125rem', color: 'rgba(255, 255, 255, 0.9)', maxWidth: '50rem', margin: '0 auto 1.5rem' }}>
              Understand real estate markets before making a property decision. Get clear insights into price trends, demand-supply conditions, growth potential, and future forecasts.
            </p>
            <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(200px, 1fr))', gap: '1rem', maxWidth: '48rem', margin: '2rem auto 0' }}>
              <div style={{ background: 'rgba(255,255,255,0.1)', padding: '1rem', borderRadius: '0.5rem', backdropFilter: 'blur(10px)' }}>
                <div style={{ fontSize: '1.5rem', fontWeight: '700', marginBottom: '0.25rem' }}>City-Level Data</div>
                <div style={{ fontSize: '0.875rem', opacity: '0.9' }}>Compare markets across cities</div>
              </div>
              <div style={{ background: 'rgba(255,255,255,0.1)', padding: '1rem', borderRadius: '0.5rem', backdropFilter: 'blur(10px)' }}>
                <div style={{ fontSize: '1.5rem', fontWeight: '700', marginBottom: '0.25rem' }}>Locality Insights</div>
                <div style={{ fontSize: '0.875rem', opacity: '0.9' }}>Area-wise detailed analysis</div>
              </div>
              <div style={{ background: 'rgba(255,255,255,0.1)', padding: '1rem', borderRadius: '0.5rem', backdropFilter: 'blur(10px)' }}>
                <div style={{ fontSize: '1.5rem', fontWeight: '700', marginBottom: '0.25rem' }}>Risk Assessment</div>
                <div style={{ fontSize: '0.875rem', opacity: '0.9' }}>Evaluate investment risks</div>
              </div>
            </div>
          </div>
        </header>

        {/* Market Snapshot with City Filter */}
        <section style={{ padding: '4rem 2rem', background: '#f8fafc' }}>
          <div style={{ maxWidth: '72rem', margin: '0 auto' }}>
            <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '3rem', flexWrap: 'wrap', gap: '1rem' }}>
              <h2 style={{ fontSize: '2rem', fontWeight: '700', color: '#1e293b' }}>
                Market Snapshot
                {appliedFilters.city && ` - ${appliedFilters.city.charAt(0).toUpperCase() + appliedFilters.city.slice(1)}`}
                {appliedFilters.bedrooms && appliedFilters.bedrooms !== 'all' && ` | ${appliedFilters.bedrooms} BHK`}
                {appliedFilters.type && appliedFilters.type !== 'all' && ` | ${appliedFilters.type.charAt(0).toUpperCase() + appliedFilters.type.slice(1)}`}
              </h2>
              
              {/* City Filter */}
              <div style={{ display: 'flex', gap: '1rem', alignItems: 'center', flexWrap: 'wrap' }}>
                <div style={{ display: 'flex', gap: '0.5rem', alignItems: 'center' }}>
                  <label style={{ fontSize: '0.875rem', fontWeight: '500', color: '#475569' }}>City:</label>
                  <select 
                    value={selectedCity} 
                    onChange={(e) => setSelectedCity(e.target.value)}
                    style={{ padding: '0.5rem 1rem', borderRadius: '0.5rem', border: '1px solid #d1d5db', fontSize: '0.875rem', background: 'white' }}
                  >
                    <option value="">All Cities</option>
                    <option value="bangalore">Bangalore</option>
                    <option value="hyderabad">Hyderabad</option>
                    <option value="pune">Pune</option>
                    <option value="mumbai">Mumbai</option>
                    <option value="delhi">Delhi NCR</option>
                    <option value="chennai">Chennai</option>
                    <option value="kolkata">Kolkata</option>
                    <option value="ahmedabad">Ahmedabad</option>
                    <option value="kochi">Kochi</option>
                    <option value="coimbatore">Coimbatore</option>
                  </select>
                </div>
                
                <div style={{ display: 'flex', gap: '0.5rem', alignItems: 'center' }}>
                  <label style={{ fontSize: '0.875rem', fontWeight: '500', color: '#475569' }}>Type:</label>
                  <select 
                    value={selectedPropertyType}
                    onChange={(e) => setSelectedPropertyType(e.target.value)}
                    style={{ padding: '0.5rem 1rem', borderRadius: '0.5rem', border: '1px solid #d1d5db', fontSize: '0.875rem', background: 'white' }}
                  >
                    <option value="all">All Types</option>
                    <option value="apartment">Apartment</option>
                    <option value="villa">Villa</option>
                    <option value="plot">Plot</option>
                    <option value="commercial">Commercial</option>
                  </select>
                </div>
                
                <div style={{ display: 'flex', gap: '0.5rem', alignItems: 'center' }}>
                  <label style={{ fontSize: '0.875rem', fontWeight: '500', color: '#475569' }}>BHK:</label>
                  <select 
                    value={selectedBedrooms}
                    onChange={(e) => setSelectedBedrooms(e.target.value)}
                    style={{ padding: '0.5rem 1rem', borderRadius: '0.5rem', border: '1px solid #d1d5db', fontSize: '0.875rem', background: 'white' }}
                  >
                    <option value="all">All BHK</option>
                    <option value="1">1 BHK</option>
                    <option value="2">2 BHK</option>
                    <option value="3">3 BHK</option>
                    <option value="4">4+ BHK</option>
                  </select>
                </div>
                
                <Button 
                  onClick={handleSearch}
                  style={{ padding: '0.5rem 1.5rem', background: '#2B256D', color: 'white', borderRadius: '0.5rem', fontSize: '0.875rem', fontWeight: '600', border: 'none', cursor: 'pointer' }}
                >
                  Search
                </Button>
                
                <Button 
                  onClick={handleClear}
                  style={{ padding: '0.5rem 1.5rem', background: '#dc2626', color: 'white', borderRadius: '0.5rem', fontSize: '0.875rem', fontWeight: '600', border: 'none', cursor: 'pointer' }}
                >
                  Clear
                </Button>
              </div>
            </div>
              
              {loading ? (
                <div style={{ textAlign: 'center', padding: '3rem' }}>
                  <div style={{ display: 'inline-block', width: '3rem', height: '3rem', border: '4px solid #e2e8f0', borderTopColor: '#2B256D', borderRadius: '50%', animation: 'spin 1s linear infinite' }}></div>
                </div>
              ) : !appliedFilters.city && appliedFilters.type === '' && appliedFilters.bedrooms === '' ? (
                <div style={{ textAlign: 'center', padding: '3rem' }}>
                  <AlertCircle style={{ width: '3rem', height: '3rem', color: '#9ca3af', margin: '0 auto 1rem' }} />
                  <p style={{ color: '#6b7280', fontSize: '1.125rem', fontWeight: '500' }}>Please select filters and click Search to view market data</p>
                </div>
              ) : marketData.length > 0 ? (
                <>
                  <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(250px, 1fr))', gap: '2rem', marginBottom: '3rem' }}>
                    <div style={{ background: 'white', padding: '2rem', borderRadius: '1rem', textAlign: 'center', boxShadow: '0 4px 20px rgba(0,0,0,0.1)' }}>
                      <h3 style={{ fontSize: '1.125rem', fontWeight: '600', color: '#1e293b', marginBottom: '0.5rem' }}>Average Price</h3>
                      <div style={{ fontSize: '1.5rem', fontWeight: '700', color: '#2B256D' }}>{formatPrice(priceComparison?.avgPrice || marketData[0].avgPrice)}</div>
                      <div style={{ fontSize: '0.875rem', color: '#64748b', marginTop: '0.5rem' }}>Range: {formatPrice(priceComparison?.minPrice || 0)} - {formatPrice(priceComparison?.maxPrice || 0)}</div>
                    </div>
                    
                    <div style={{ background: 'white', padding: '2rem', borderRadius: '1rem', textAlign: 'center', boxShadow: '0 4px 20px rgba(0,0,0,0.1)' }}>
                      <h3 style={{ fontSize: '1.125rem', fontWeight: '600', color: '#1e293b', marginBottom: '0.5rem' }}>Price per Sq.ft</h3>
                      <div style={{ fontSize: '1.5rem', fontWeight: '700', color: '#2B256D' }}>₹{Math.round(marketData[0].pricePerSqft).toLocaleString()}</div>
                      <div style={{ fontSize: '0.875rem', color: '#64748b', marginTop: '0.5rem' }}>Fair Range: ₹{Math.round((priceComparison?.fairPriceRange?.min || 0) / (marketData[0].totalProperties || 1)).toLocaleString()} - ₹{Math.round((priceComparison?.fairPriceRange?.max || 0) / (marketData[0].totalProperties || 1)).toLocaleString()}</div>
                    </div>
                    
                    <div style={{ background: 'white', padding: '2rem', borderRadius: '1rem', textAlign: 'center', boxShadow: '0 4px 20px rgba(0,0,0,0.1)' }}>
                      <h3 style={{ fontSize: '1.125rem', fontWeight: '600', color: '#1e293b', marginBottom: '0.5rem' }}>Total Listings</h3>
                      <div style={{ fontSize: '1.5rem', fontWeight: '700', color: '#dc2626' }}>{demandSupplyData?.totalListings || marketData[0].totalProperties}</div>
                      <div style={{ fontSize: '0.875rem', color: '#64748b', marginTop: '0.5rem' }}>Supply: {demandSupplyData?.totalListings > 100 ? 'High' : demandSupplyData?.totalListings > 50 ? 'Medium' : 'Low'}</div>
                    </div>
                    
                    <div style={{ background: 'white', padding: '2rem', borderRadius: '1rem', textAlign: 'center', boxShadow: '0 4px 20px rgba(0,0,0,0.1)' }}>
                      <h3 style={{ fontSize: '1.125rem', fontWeight: '600', color: '#1e293b', marginBottom: '0.5rem' }}>Market Status</h3>
                      <div style={{ fontSize: '1.5rem', fontWeight: '700', color: '#059669' }}>{demandSupplyData?.marketStatus || volatilityData?.volatility_status || 'Stable'}</div>
                      <div style={{ fontSize: '0.875rem', color: '#64748b', marginTop: '0.5rem' }}>Demand: {demandSupplyData?.demandScore || 'Medium'}</div>
                    </div>
                  </div>

                  {/* Demand Supply Analysis */}
                  {demandSupplyData && (
                    <div style={{ background: 'white', padding: '2rem', borderRadius: '1rem', marginBottom: '2rem', boxShadow: '0 4px 20px rgba(0,0,0,0.1)' }}>
                      <h3 style={{ fontSize: '1.5rem', fontWeight: '600', marginBottom: '1.5rem', color: '#1e293b' }}>Demand & Supply Analysis</h3>
                      <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(200px, 1fr))', gap: '1.5rem' }}>
                        <div style={{ padding: '1rem', background: '#f0fdf4', borderRadius: '0.5rem', borderLeft: '3px solid #059669' }}>
                          <div style={{ fontSize: '0.875rem', fontWeight: '600', color: '#059669', marginBottom: '0.25rem' }}>Total Views</div>
                          <div style={{ fontSize: '1.75rem', fontWeight: '700', color: '#1e293b' }}>{demandSupplyData.totalViews.toLocaleString()}</div>
                          <div style={{ fontSize: '0.75rem', color: '#64748b', marginTop: '0.25rem' }}>Avg: {demandSupplyData.avgViews} per property</div>
                        </div>
                        <div style={{ padding: '1rem', background: '#fef3c7', borderRadius: '0.5rem', borderLeft: '3px solid #f59e0b' }}>
                          <div style={{ fontSize: '0.875rem', fontWeight: '600', color: '#f59e0b', marginBottom: '0.25rem' }}>Total Inquiries</div>
                          <div style={{ fontSize: '1.75rem', fontWeight: '700', color: '#1e293b' }}>{demandSupplyData.totalInquiries.toLocaleString()}</div>
                          <div style={{ fontSize: '0.75rem', color: '#64748b', marginTop: '0.25rem' }}>Avg: {demandSupplyData.avgInquiries} per property</div>
                        </div>
                        <div style={{ padding: '1rem', background: '#dbeafe', borderRadius: '0.5rem', borderLeft: '3px solid #2563eb' }}>
                          <div style={{ fontSize: '0.875rem', fontWeight: '600', color: '#2563eb', marginBottom: '0.25rem' }}>Demand Score</div>
                          <div style={{ fontSize: '1.75rem', fontWeight: '700', color: '#1e293b' }}>{demandSupplyData.demandScore}</div>
                          <div style={{ fontSize: '0.75rem', color: '#64748b', marginTop: '0.25rem' }}>Based on buyer activity</div>
                        </div>
                        <div style={{ padding: '1rem', background: '#fce7f3', borderRadius: '0.5rem', borderLeft: '3px solid #ec4899' }}>
                          <div style={{ fontSize: '0.875rem', fontWeight: '600', color: '#ec4899', marginBottom: '0.25rem' }}>Market Type</div>
                          <div style={{ fontSize: '1.75rem', fontWeight: '700', color: '#1e293b' }}>{demandSupplyData.marketStatus}</div>
                          <div style={{ fontSize: '0.75rem', color: '#64748b', marginTop: '0.25rem' }}>Current condition</div>
                        </div>
                      </div>
                    </div>
                  )}

                  {/* Time on Market Analysis */}
                  {timeOnMarket && (
                    <div style={{ background: 'white', padding: '2rem', borderRadius: '1rem', marginBottom: '2rem', boxShadow: '0 4px 20px rgba(0,0,0,0.1)' }}>
                      <h3 style={{ fontSize: '1.5rem', fontWeight: '600', marginBottom: '1.5rem', color: '#1e293b' }}>Time-on-Market Analysis</h3>
                      <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(200px, 1fr))', gap: '1.5rem' }}>
                        <div style={{ padding: '1rem', background: '#f8fafc', borderRadius: '0.5rem', border: '1px solid #e2e8f0' }}>
                          <div style={{ fontSize: '0.875rem', fontWeight: '600', color: '#64748b', marginBottom: '0.25rem' }}>Avg Days on Market</div>
                          <div style={{ fontSize: '1.75rem', fontWeight: '700', color: '#2B256D' }}>{timeOnMarket.avgDays} days</div>
                          <div style={{ fontSize: '0.75rem', color: '#64748b', marginTop: '0.25rem' }}>Status: {timeOnMarket.status}</div>
                        </div>
                        <div style={{ padding: '1rem', background: '#f8fafc', borderRadius: '0.5rem', border: '1px solid #e2e8f0' }}>
                          <div style={{ fontSize: '0.875rem', fontWeight: '600', color: '#64748b', marginBottom: '0.25rem' }}>Fast Selling</div>
                          <div style={{ fontSize: '1.75rem', fontWeight: '700', color: '#059669' }}>{timeOnMarket.fastSelling}</div>
                          <div style={{ fontSize: '0.75rem', color: '#64748b', marginTop: '0.25rem' }}>Sold within 30 days</div>
                        </div>
                        <div style={{ padding: '1rem', background: '#f8fafc', borderRadius: '0.5rem', border: '1px solid #e2e8f0' }}>
                          <div style={{ fontSize: '0.875rem', fontWeight: '600', color: '#64748b', marginBottom: '0.25rem' }}>Slow Selling</div>
                          <div style={{ fontSize: '1.75rem', fontWeight: '700', color: '#dc2626' }}>{timeOnMarket.slowSelling}</div>
                          <div style={{ fontSize: '0.75rem', color: '#64748b', marginTop: '0.25rem' }}>Over 90 days unsold</div>
                        </div>
                      </div>
                    </div>
                  )}

                  {/* Price Comparison & Risk Assessment */}
                  {priceComparison && (
                    <div style={{ background: 'white', padding: '2rem', borderRadius: '1rem', marginBottom: '2rem', boxShadow: '0 4px 20px rgba(0,0,0,0.1)' }}>
                      <h3 style={{ fontSize: '1.5rem', fontWeight: '600', marginBottom: '1.5rem', color: '#1e293b' }}>Price Comparison & Risk Assessment</h3>
                      <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(300px, 1fr))', gap: '1.5rem' }}>
                        <div>
                          <h4 style={{ fontSize: '1.125rem', fontWeight: '600', color: '#2B256D', marginBottom: '1rem' }}>Fair Price Range</h4>
                          <div style={{ padding: '1rem', background: '#f0fdf4', borderRadius: '0.5rem', marginBottom: '0.5rem' }}>
                            <div style={{ fontSize: '0.875rem', color: '#64748b', marginBottom: '0.25rem' }}>Lower Bound (15% below avg)</div>
                            <div style={{ fontSize: '1.25rem', fontWeight: '700', color: '#059669' }}>{formatPrice(priceComparison.fairPriceRange.min)}</div>
                          </div>
                          <div style={{ padding: '1rem', background: '#fef3c7', borderRadius: '0.5rem' }}>
                            <div style={{ fontSize: '0.875rem', color: '#64748b', marginBottom: '0.25rem' }}>Upper Bound (15% above avg)</div>
                            <div style={{ fontSize: '1.25rem', fontWeight: '700', color: '#f59e0b' }}>{formatPrice(priceComparison.fairPriceRange.max)}</div>
                          </div>
                        </div>
                        <div>
                          <h4 style={{ fontSize: '1.125rem', fontWeight: '600', color: '#2B256D', marginBottom: '1rem' }}>Investment Signals</h4>
                          <div style={{ padding: '1rem', background: demandSupplyData?.demandScore === 'High' ? '#f0fdf4' : '#fef3c7', borderRadius: '0.5rem', marginBottom: '0.5rem', borderLeft: '3px solid ' + (demandSupplyData?.demandScore === 'High' ? '#059669' : '#f59e0b') }}>
                            <div style={{ fontSize: '0.875rem', fontWeight: '600', marginBottom: '0.25rem' }}>Demand Level</div>
                            <div style={{ fontSize: '1rem', color: '#475569' }}>{demandSupplyData?.demandScore === 'High' ? 'High buyer interest' : demandSupplyData?.demandScore === 'Medium' ? 'Moderate interest' : 'Low interest'}</div>
                          </div>
                          <div style={{ padding: '1rem', background: timeOnMarket?.status === 'Fast Moving' ? '#f0fdf4' : '#fee2e2', borderRadius: '0.5rem', borderLeft: '3px solid ' + (timeOnMarket?.status === 'Fast Moving' ? '#059669' : '#dc2626') }}>
                            <div style={{ fontSize: '0.875rem', fontWeight: '600', marginBottom: '0.25rem' }}>Liquidity</div>
                            <div style={{ fontSize: '1rem', color: '#475569' }}>{timeOnMarket?.status === 'Fast Moving' ? 'Properties sell quickly' : timeOnMarket?.status === 'Normal' ? 'Normal selling pace' : 'Slow market movement'}</div>
                          </div>
                        </div>
                        <div>
                          <h4 style={{ fontSize: '1.125rem', fontWeight: '600', color: '#2B256D', marginBottom: '1rem' }}>Buyer Recommendation</h4>
                          <div style={{ padding: '1.5rem', background: 'linear-gradient(135deg, #2B256D 0%, #1f1a5a 100%)', borderRadius: '0.5rem', color: 'white' }}>
                            <div style={{ fontSize: '1.125rem', fontWeight: '700', marginBottom: '0.75rem' }}>
                              {demandSupplyData?.demandScore === 'High' && timeOnMarket?.status === 'Fast Moving' ? 'Strong Buy Signal' :
                               demandSupplyData?.demandScore === 'Medium' ? 'Balanced Market' : 'Wait & Watch'}
                            </div>
                            <div style={{ fontSize: '0.875rem', opacity: '0.9', lineHeight: '1.5' }}>
                              {demandSupplyData?.demandScore === 'High' && timeOnMarket?.status === 'Fast Moving' ? 'High demand + fast selling = good investment opportunity' :
                               demandSupplyData?.demandScore === 'Medium' ? 'Moderate activity, negotiate for better deals' : 'Low demand, consider waiting or strong negotiation'}
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                  )}

                </>
              ) : (
                <div style={{ textAlign: 'center', padding: '3rem' }}>
                  <AlertCircle style={{ width: '3rem', height: '3rem', color: '#9ca3af', margin: '0 auto 1rem' }} />
                  <p style={{ color: '#6b7280' }}>No properties found for {appliedFilters.city.charAt(0).toUpperCase() + appliedFilters.city.slice(1)}
                    {appliedFilters.type !== 'all' && ` - ${appliedFilters.type}`}
                    {appliedFilters.bedrooms !== 'all' && ` - ${appliedFilters.bedrooms} BHK`}
                  </p>
                </div>
              )}

              <div style={{ textAlign: 'center', marginTop: '2rem' }}>
                <Button 
                  style={{ background: '#2B256D', color: 'white', padding: '1rem 2rem', fontSize: '1.125rem', fontWeight: '600', borderRadius: '0.5rem', marginRight: '1rem' }}
                >
                  Get Professional Report - ₹8,999
                </Button>
                <Button 
                  style={{ background: 'transparent', color: '#2B256D', border: '2px solid #2B256D', padding: '1rem 2rem', fontSize: '1.125rem', fontWeight: '600', borderRadius: '0.5rem' }}
                >
                  Consult Market Expert
                </Button>
              </div>
            </div>
          </section>

        {/* Key Features Section */}
        <section style={{ padding: '4rem 2rem', background: 'white' }}>
          <div style={{ maxWidth: '72rem', margin: '0 auto' }}>
            <h2 style={{ fontSize: '2rem', fontWeight: '700', textAlign: 'center', marginBottom: '1rem', color: '#1e293b' }}>What You Get with Market Analysis</h2>
            <p style={{ textAlign: 'center', color: '#64748b', marginBottom: '3rem', maxWidth: '48rem', margin: '0 auto 3rem' }}>Professional market reports and expert-backed analysis to help you make confident, data-driven property decisions</p>
            
            <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(280px, 1fr))', gap: '2rem' }}>
              <div style={{ background: '#f8fafc', padding: '2rem', borderRadius: '1rem', border: '1px solid #e2e8f0' }}>
                <h3 style={{ fontSize: '1.25rem', fontWeight: '600', marginBottom: '0.75rem', color: '#1e293b' }}>Price Trend Analysis</h3>
                <p style={{ color: '#64748b', fontSize: '0.9375rem', lineHeight: '1.6' }}>Historical price movements, current market rates, and future price forecasts for informed decision-making</p>
              </div>

              <div style={{ background: '#f8fafc', padding: '2rem', borderRadius: '1rem', border: '1px solid #e2e8f0' }}>
                <h3 style={{ fontSize: '1.25rem', fontWeight: '600', marginBottom: '0.75rem', color: '#1e293b' }}>Demand & Supply Insights</h3>
                <p style={{ color: '#64748b', fontSize: '0.9375rem', lineHeight: '1.6' }}>Real-time inventory levels, buyer demand patterns, and market liquidity indicators</p>
              </div>

              <div style={{ background: '#f8fafc', padding: '2rem', borderRadius: '1rem', border: '1px solid #e2e8f0' }}>
                <h3 style={{ fontSize: '1.25rem', fontWeight: '600', marginBottom: '0.75rem', color: '#1e293b' }}>Growth Potential Evaluation</h3>
                <p style={{ color: '#64748b', fontSize: '0.9375rem', lineHeight: '1.6' }}>Infrastructure development, connectivity improvements, and area appreciation forecasts</p>
              </div>

              <div style={{ background: '#f8fafc', padding: '2rem', borderRadius: '1rem', border: '1px solid #e2e8f0' }}>
                <h3 style={{ fontSize: '1.25rem', fontWeight: '600', marginBottom: '0.75rem', color: '#1e293b' }}>Area Comparison</h3>
                <p style={{ color: '#64748b', fontSize: '0.9375rem', lineHeight: '1.6' }}>Side-by-side comparison of localities based on price, amenities, connectivity, and growth</p>
              </div>

              <div style={{ background: '#f8fafc', padding: '2rem', borderRadius: '1rem', border: '1px solid #e2e8f0' }}>
                <h3 style={{ fontSize: '1.25rem', fontWeight: '600', marginBottom: '0.75rem', color: '#1e293b' }}>Risk Assessment</h3>
                <p style={{ color: '#64748b', fontSize: '0.9375rem', lineHeight: '1.6' }}>Market volatility analysis, legal risks, and investment safety scores for each location</p>
              </div>

              <div style={{ background: '#f8fafc', padding: '2rem', borderRadius: '1rem', border: '1px solid #e2e8f0' }}>
                <h3 style={{ fontSize: '1.25rem', fontWeight: '600', marginBottom: '0.75rem', color: '#1e293b' }}>Personalized Recommendations</h3>
                <p style={{ color: '#64748b', fontSize: '0.9375rem', lineHeight: '1.6' }}>Strategic insights based on your budget, goals, and preferences to identify best opportunities</p>
              </div>
            </div>
          </div>
        </section>

        {/* Market Trends & Analytics */}
        <section style={{ padding: '4rem 2rem', background: 'linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%)' }}>
          <div style={{ maxWidth: '72rem', margin: '0 auto' }}>
            <h2 style={{ fontSize: '2rem', fontWeight: '700', textAlign: 'center', marginBottom: '3rem', color: '#1e293b' }}>Live Market Trends & Analytics</h2>
            
            <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(250px, 1fr))', gap: '2rem', marginBottom: '3rem' }}>
              <div style={{ background: 'white', padding: '2rem', borderRadius: '1rem', boxShadow: '0 4px 20px rgba(0,0,0,0.1)', border: '1px solid #e2e8f0' }}>
                <h3 style={{ fontSize: '1.25rem', fontWeight: '600', color: '#2B256D', marginBottom: '1rem' }}>Top Cities by Properties</h3>
                <div style={{ space: '1rem' }}>
                  {cityGrowth.slice(0, 3).map((city, idx) => (
                    <div key={idx} style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '0.75rem', padding: '0.5rem', background: '#f1f5f9', borderRadius: '0.5rem' }}>
                      <span style={{ fontWeight: '500', textTransform: 'capitalize' }}>{city.city}</span>
                      <span style={{ color: '#059669', fontWeight: '600' }}>{city.count} props</span>
                    </div>
                  ))}
                </div>
              </div>
              
              <div style={{ background: 'white', padding: '2rem', borderRadius: '1rem', boxShadow: '0 4px 20px rgba(0,0,0,0.1)', border: '1px solid #e2e8f0' }}>
                <h3 style={{ fontSize: '1.25rem', fontWeight: '600', color: '#2B256D', marginBottom: '1rem' }}>Market Status</h3>
                <div style={{ textAlign: 'center' }}>
                  <div style={{ fontSize: '2rem', fontWeight: '700', color: '#059669', marginBottom: '0.5rem' }}>{volatilityData?.volatility_status || 'Stable'}</div>
                  <div style={{ fontSize: '0.875rem', color: '#64748b' }}>{volatilityData?.simple_message || 'Market analysis in progress'}</div>
                </div>
              </div>
              
              <div style={{ background: 'white', padding: '2rem', borderRadius: '1rem', boxShadow: '0 4px 20px rgba(0,0,0,0.1)', border: '1px solid #e2e8f0' }}>
                <h3 style={{ fontSize: '1.25rem', fontWeight: '600', color: '#2B256D', marginBottom: '1rem' }}>Top Localities</h3>
                <div style={{ space: '1rem' }}>
                  {marketData.slice(0, 3).map((data, idx) => (
                    <div key={idx} style={{ marginBottom: '0.75rem', padding: '0.5rem', background: '#f1f5f9', borderRadius: '0.5rem' }}>
                      <div style={{ fontWeight: '600', color: '#1e293b', fontSize: '0.875rem' }}>{data.locality}</div>
                      <div style={{ fontSize: '0.75rem', color: '#64748b' }}>{data.totalProperties} properties</div>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          </div>
        </section>

        {/* Detailed Market Data */}
        <section style={{ padding: '4rem 2rem', background: 'white' }}>
          <div style={{ maxWidth: '72rem', margin: '0 auto' }}>
            <h2 style={{ fontSize: '2rem', fontWeight: '700', textAlign: 'center', marginBottom: '3rem', color: '#1e293b' }}>Comprehensive Market Data</h2>
            
            {propertyTypeData && Object.keys(propertyTypeData).length > 0 && (
              <div style={{ background: '#f8fafc', padding: '2rem', borderRadius: '1rem', border: '1px solid #e2e8f0', marginBottom: '3rem' }}>
                <h3 style={{ fontSize: '1.5rem', fontWeight: '600', color: '#1e293b', marginBottom: '2rem' }}>Price Analysis by Property Type</h3>
                <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(250px, 1fr))', gap: '1.5rem' }}>
                  {Object.entries(propertyTypeData).map(([type, data]: [string, any]) => (
                    <div key={type} style={{ marginBottom: '1rem' }}>
                      <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '0.5rem' }}>
                        <span style={{ fontWeight: '500' }}>{type} Apartments</span>
                        <span style={{ fontWeight: '600', color: '#2B256D' }}>{formatPrice(data.min)}-{formatPrice(data.max)}</span>
                      </div>
                      <div style={{ fontSize: '0.875rem', color: '#64748b' }}>Average: {formatPrice(data.avg)} | Count: {data.count}</div>
                    </div>
                  ))}
                </div>
              </div>
            )}
            
            <div style={{ background: 'linear-gradient(135deg, #2B256D 0%, #1f1a5a 100%)', padding: '3rem', borderRadius: '1rem', color: 'white', textAlign: 'center' }}>
              <h3 style={{ fontSize: '1.75rem', fontWeight: '600', marginBottom: '1rem' }}>Market Intelligence Advantage</h3>
              <p style={{ fontSize: '1.125rem', marginBottom: '2rem', opacity: '0.9' }}>Real-time data from live property transactions</p>
              <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(150px, 1fr))', gap: '2rem' }}>
                <div>
                  <div style={{ fontSize: '2rem', fontWeight: '700', marginBottom: '0.5rem' }}>{loading ? '...' : allProperties.length}</div>
                  <div style={{ fontSize: '0.875rem', opacity: '0.8' }}>Total Properties</div>
                </div>
                <div>
                  <div style={{ fontSize: '2rem', fontWeight: '700', marginBottom: '0.5rem' }}>{loading ? '...' : marketData.length}</div>
                  <div style={{ fontSize: '0.875rem', opacity: '0.8' }}>Localities</div>
                </div>
                <div>
                  <div style={{ fontSize: '2rem', fontWeight: '700', marginBottom: '0.5rem' }}>{loading ? '...' : cityGrowth.length}</div>
                  <div style={{ fontSize: '0.875rem', opacity: '0.8' }}>Cities</div>
                </div>
                <div>
                  <div style={{ fontSize: '2rem', fontWeight: '700', marginBottom: '0.5rem' }}>Live</div>
                  <div style={{ fontSize: '0.875rem', opacity: '0.8' }}>Data</div>
                </div>
              </div>
            </div>
          </div>
        </section>








      </div>
    </>
  );
}