import React, { useState, useEffect, useRef } from "react";
import { useNavigate } from "react-router-dom";
import { CheckCircle, TrendingUp, MapPin, Star, Clock } from "lucide-react";
import { Button } from "../../components/ui/button";

interface Testimonial {
  name: string;
  location: string;
  rating: number;
  text: string;
}

export function PropertyManagement() {
  const navigate = useNavigate();

  // Dynamic state for live statistics
  const [liveStats, setLiveStats] = useState({
    propertiesManaged: 0,
    nriClients: 0,
    avgRentalYield: 0,
    clientSatisfaction: 0,
    citiesCovered: 0,
    activeContracts: 0,
  });
  const [isLoading, setIsLoading] = useState(true);
  const [testimonials, setTestimonials] = useState<Testimonial[]>([]);

  // Auth modal and auth state
  const [showAuthModal, setShowAuthModal] = useState(false);
  const [isLoggedIn, setIsLoggedIn] = useState(false); // Replace with actual auth state

  // ref to keep a stable onChoosePlan inside global handlers if needed
  const onChoosePlanRef = useRef<(planName: string) => void>(() => {});
  useEffect(() => {
    onChoosePlanRef.current = (planName: string) => {
      if (!isLoggedIn) {
        setShowAuthModal(true);
        return;
      }
      console.log(`${planName} plan selected`);
      alert(`You selected the ${planName} plan!`);
    };
  }, [isLoggedIn]);

  // Fetch live statistics
  useEffect(() => {
    let mounted = true;
    const fetchLiveStats = async () => {
      setIsLoading(true);
      try {
        // Simulate API call with dynamic data
        await new Promise((resolve) => setTimeout(resolve, 800));

        if (!mounted) return;
        setLiveStats({
          propertiesManaged: Math.floor(Math.random() * 500) + 2500,
          nriClients: Math.floor(Math.random() * 200) + 800,
          avgRentalYield: parseFloat((Math.random() * 5 + 25).toFixed(1)),
          clientSatisfaction: parseFloat((Math.random() * 5 + 92).toFixed(1)),
          citiesCovered: Math.floor(Math.random() * 10) + 45,
          activeContracts: Math.floor(Math.random() * 100) + 1200,
        });

        setTestimonials([
          {
            name: "Rajesh Kumar",
            location: "NRI - Singapore",
            rating: 5,
            text: "NAL's property management has been exceptional. My rental yield increased by 28% within 6 months.",
          },
          {
            name: "Priya Sharma",
            location: "Bangalore",
            rating: 5,
            text: "Professional service with complete transparency. No more tenant headaches!",
          },
          {
            name: "David Chen",
            location: "NRI - USA",
            rating: 5,
            text: "Managing my Mumbai property from the US was impossible until I found NAL. Highly recommended!",
          },
        ] as Testimonial[]);
      } catch (error) {
        console.error("Failed to fetch stats:", error);
      } finally {
        if (mounted) setIsLoading(false);
      }
    };

    fetchLiveStats();

    // Update stats every 30 seconds
    const interval = setInterval(fetchLiveStats, 30000);
    return () => {
      mounted = false;
      clearInterval(interval);
    };
  }, []);

  // Centralized onChoosePlan wrapper that uses the ref
  const onChoosePlan = (planName: string) => {
    onChoosePlanRef.current(planName);
  };

  const showAuthModalHandler = () => {
    setShowAuthModal(true);
  };

  /**
   * Defensive UI fixer:
   * - Ensures plan CTAs are not disabled by removing disabled attr/classes.
   * - Applies safe inline styles to CTAs to avoid global CSS accidentally greying them out.
   * - Installs a short-lived MutationObserver that re-enables CTAs if something re-disables them.
   *
   * Remove the MutationObserver block once you have fixed the root cause.
   */
  useEffect(() => {
    // Selector that targets plan CTA buttons inside this component
    const selector = "[data-plan]";

    // Helper to enable a button element
    const enableButton = (el: HTMLElement | null) => {
      if (!el) return;
      try {
        // remove disabled attr if present
        if ("disabled" in el && (el as any).disabled) {
          (el as any).disabled = false;
        }
        el.removeAttribute("disabled");
        // remove common disabling classes that projects often use
        el.classList.remove(
          "disabled",
          "opacity-50",
          "pointer-events-none",
          "btn-disabled"
        );

        // Scoped inline style to guarantee interactivity (keeps CSS manageable)
        // NOTE: keep these minimal and scoped to elements with data-plan
        (el as HTMLElement).style.opacity = "1";
        (el as HTMLElement).style.pointerEvents = "auto";
        (el as HTMLElement).style.cursor = "pointer";
        // keep visual styling minimal here; main styles still come from className
      } catch (err) {
        // ignore
      }
    };

    // First pass enable
    document
      .querySelectorAll(selector)
      .forEach((node) => enableButton(node as HTMLElement));

    // Install a MutationObserver to detect if something re-disables our buttons.
    // This is defensive and short-lived: it will auto-disconnect after 10s.
    const observer = new MutationObserver((mutations) => {
      for (const m of mutations) {
        // If attributes changed on a plan button, re-enable it
        if (
          m.type === "attributes" &&
          m.target &&
          (m.target as Element).matches &&
          (m.target as Element).matches(selector)
        ) {
          enableButton(m.target as HTMLElement);
        }
        // If new nodes added, check inside them
        if (m.addedNodes && m.addedNodes.length) {
          m.addedNodes.forEach((n) => {
            if (n instanceof Element) {
              if (n.matches && n.matches(selector))
                enableButton(n as HTMLElement);
              (
                (n.querySelectorAll && n.querySelectorAll(selector)) ||
                []
              ).forEach((sub: Element) => enableButton(sub as HTMLElement));
            }
          });
        }
      }
    });

    observer.observe(document.documentElement, {
      attributes: true,
      childList: true,
      subtree: true,
    });

    // Auto-disconnect observer after 8 seconds to avoid long-running background work.
    const t = setTimeout(() => {
      try {
        observer.disconnect();
      } catch (e) {}
    }, 8000);

    return () => {
      clearTimeout(t);
      try {
        observer.disconnect();
      } catch (e) {}
    };
  }, []);

  // Add a document-level click handler to catch any [data-plan] activations
  // (This is defensive; buttons already have onClick handlers — this ensures clicks from other paths are captured)
  useEffect(() => {
    function handler(e: MouseEvent) {
      const target = e.target as HTMLElement | null;
      if (!target) return;
      const btn =
        target.closest && (target.closest("[data-plan]") as HTMLElement | null);
      if (!btn) return;
      e.preventDefault();
      const plan =
        btn.getAttribute("data-plan") ||
        btn.getAttribute("aria-label") ||
        btn.textContent ||
        "plan";
      onChoosePlan(plan.trim());
    }
    document.addEventListener("click", handler);
    return () => document.removeEventListener("click", handler);
  }, []);

  return (
    <>
      <style>{`
        /* Scoped component styles (clean, no global disabled overrides) */
        .property-hero {
          background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
          padding: 6rem 0;
          color: white;
          text-align: center;
        }
        .property-hero h1 {
          font-size: 3rem;
          font-weight: 700;
          margin-bottom: 1.5rem;
          color: white;
          letter-spacing: -0.02em;
        }
        .property-hero p { font-size: 1.125rem; margin-bottom: 1.75rem; color: #e2e8f0; max-width: 60rem; margin-left: auto; margin-right: auto; line-height: 1.6; }

        /* Plans area */
        .plan-card {
          display: flex;
          flex-direction: column;
          height: 100%;
          border-radius: 1rem;
          overflow: visible;
        }
        .plan-card .plan-inner { background: #ffffff; border-radius: 1rem; border: 1px solid #e6edf3; padding: 2rem; display:flex; flex-direction: column; height: 100%; }
        .plan-card .plan-features { flex: 1 1 auto; margin-top: 0.5rem; margin-bottom: 1.25rem; }
        .plan-card .plan-footer { padding-top: 0.5rem; }

        /* Badge should never block pointer events */
        .most-popular-badge { pointer-events: none; }

        /* Buttons: class-based primary styles */
        .cta-primary {
          display: inline-block;
          width: 100%;
          padding: 1rem 1.25rem;
          border-radius: 0.75rem;
          font-weight: 600;
          border: none;
          box-shadow: 0 6px 18px rgba(37,99,235,0.12);
          transition: transform .12s ease, box-shadow .12s ease;
        }
        .cta-primary:focus { outline: none; box-shadow: 0 0 0 4px rgba(37,99,235,0.12); }
        .cta-primary:hover { transform: translateY(-2px); }

        /* Keep NRI style separate */
        .cta-nri { background-color: #7C3AED; color: #fff; }
        .cta-blue { background-color: #2563EB; color: #fff; }
        /* Accessible small link under button */
        .plan-link { background: none; border: none; color: inherit; text-decoration: underline; cursor: pointer; padding: 0; }

        /* simple responsive tweaks */
        @media (max-width: 768px) {
          .property-hero h1 { font-size: 2rem; }
        }
      `}</style>

      <div className="min-h-screen bg-white">
        {/* Hero Section */}
        <div className="relative bg-gradient-to-br from-blue-600 via-blue-700 to-indigo-800 py-20 mt-3 overflow-hidden property-hero">
          <div className="absolute top-20 left-10 w-72 h-72 bg-white/5 rounded-full blur-3xl"></div>
          <div className="absolute bottom-20 right-10 w-96 h-96 bg-blue-400/10 rounded-full blur-3xl"></div>
          <div className="relative max-w-5xl mx-auto px-4 text-center">
            <h1 className="text-5xl font-bold text-white mb-6">
              Property Management Services
            </h1>
            <p className="text-xl text-blue-100 mb-8 leading-relaxed">
              Complete Peace of Mind for Owners & NRIs. NAL India manages your
              property professionally with services designed for NRIs
              contributing 15-20% of real estate investments (projected to reach
              20% by 2025). Our clients report 25-30% higher rent yields and
              significantly fewer tenant disputes through tech-enabled
              oversight.
            </p>

            {/* Live Statistics */}
            <div className="grid grid-cols-2 md:grid-cols-3 gap-4 mb-8 max-w-4xl mx-auto">
              <div className="bg-white/10 backdrop-blur-sm rounded-lg p-4 text-center border border-white/20">
                <div className="text-2xl font-bold text-white">
                  {isLoading
                    ? "..."
                    : liveStats.propertiesManaged.toLocaleString()}
                </div>
                <div className="text-blue-200 text-sm">Properties Managed</div>
              </div>
              <div className="bg-white/10 backdrop-blur-sm rounded-lg p-4 text-center border border-white/20">
                <div className="text-2xl font-bold text-white">
                  {isLoading ? "..." : liveStats.nriClients.toLocaleString()}
                </div>
                <div className="text-blue-200 text-sm">NRI Clients</div>
              </div>
              <div className="bg-white/10 backdrop-blur-sm rounded-lg p-4 text-center border border-white/20">
                <div className="text-2xl font-bold text-white">
                  {isLoading ? "..." : `${liveStats.avgRentalYield}%`}
                </div>
                <div className="text-blue-200 text-sm">Avg Rental Yield</div>
              </div>
            </div>

            <div className="inline-block bg-white/20 backdrop-blur-sm text-white px-8 py-4 rounded-xl font-semibold text-lg shadow-lg border border-white/30">
              Standard: 8% of rental income | Premium: 10% | NRI Special: 12%
            </div>
          </div>
        </div>

        {/* Service Images Section (unchanged) */}
        <div className="bg-gradient-to-b from-white to-gray-50 py-20">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
              <div className="text-center bg-white p-8 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1 group">
                <img
                  src="https://images.unsplash.com/photo-1582407947304-fd86f028f716?w=400&h=300&fit=crop"
                  alt="Property Care"
                  className="w-full h-48 object-cover rounded-xl mb-6 shadow-md group-hover:scale-105 transition-transform duration-300"
                />
                <h3 className="text-xl font-bold text-gray-900 mb-3">
                  Property Care
                </h3>
                <p className="text-gray-600">
                  {isLoading
                    ? "Loading..."
                    : `${liveStats.avgRentalYield}% higher rental yields`}
                </p>
                <div className="flex items-center justify-center gap-2 mt-2">
                  <TrendingUp className="w-4 h-4 text-green-500" />
                  <span className="text-sm text-green-600 font-medium">
                    Live Performance
                  </span>
                </div>
              </div>

              <div className="text-center bg-white p-8 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1 group">
                <img
                  src="https://images.unsplash.com/photo-1551434678-e076c223a692?w=400&h=300&fit=crop"
                  alt="Team Coordination"
                  className="w-full h-48 object-cover rounded-xl mb-6 shadow-md group-hover:scale-105 transition-transform duration-300"
                />
                <h3 className="text-xl font-bold text-gray-900 mb-3">
                  Team Coordination
                </h3>
                <p className="text-gray-600">
                  {isLoading
                    ? "Loading..."
                    : `Professional oversight across ${liveStats.citiesCovered}+ cities`}
                </p>
                <div className="flex items-center justify-center gap-2 mt-2">
                  <MapPin className="w-4 h-4 text-blue-500" />
                  <span className="text-sm text-blue-600 font-medium">
                    Pan-India Coverage
                  </span>
                </div>
              </div>

              <div className="text-center bg-white p-8 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1 group">
                <img
                  src="https://images.unsplash.com/photo-1448630360428-65456885c650?w=400&h=300&fit=crop"
                  alt="Peace of Mind"
                  className="w-full h-48 object-cover rounded-xl mb-6 shadow-md group-hover:scale-105 transition-transform duration-300"
                />
                <h3 className="text-xl font-bold text-gray-900 mb-3">
                  Peace of Mind
                </h3>
                <p className="text-gray-600">
                  {isLoading
                    ? "Loading..."
                    : `${liveStats.clientSatisfaction}% client satisfaction rate`}
                </p>
                <div className="flex items-center justify-center gap-2 mt-2">
                  <Star className="w-4 h-4 text-yellow-500" />
                  <span className="text-sm text-yellow-600 font-medium">
                    Verified Reviews
                  </span>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* features/benefits/process/testimonials omitted for brevity but preserved in your original file - kept below unchanged */}
        <div className="max-w-6xl mx-auto px-4 py-16 -mt-6">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <div className="bg-white p-8 rounded-2xl border border-gray-200 shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
              <h2 className="text-2xl font-bold text-gray-900 mb-6">
                Key Features
              </h2>
              <ul className="space-y-4">
                <li className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">
                    Complete tenant management and screening services with KYC
                    verification and background checks
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">
                    Automated rent collection and payment processing with
                    digital payment integration
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">
                    Repairs and maintenance coordination with trusted vendor
                    network across 50+ cities
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">
                    Regular property inspections and condition reports with
                    photo documentation
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">
                    Emergency response and 24/7 support services—critical for
                    NRI owners across time zones
                  </span>
                </li>
              </ul>
            </div>

            <div className="bg-white p-8 rounded-2xl border border-gray-200 shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
              <h2 className="text-2xl font-bold text-gray-900 mb-6">
                Benefits
              </h2>
              <ul className="space-y-4">
                <li className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">
                    Ideal solution for NRIs managing properties
                    remotely—supporting 15-20% of investment market
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">
                    Perfect for investors with multiple property portfolios
                    seeking 25-30% higher rental yields
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">
                    Hassle-free management with professional oversight reducing
                    disputes significantly
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">
                    Property value preservation through proper maintenance and
                    timely repairs
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">
                    Significant time savings and stress reduction—average client
                    satisfaction rate above 90%
                  </span>
                </li>
              </ul>
            </div>

            <div className="bg-white p-8 rounded-2xl border border-gray-200 shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
              <h2 className="text-2xl font-bold text-gray-900 mb-6">
                How It Works
              </h2>
              <ol className="space-y-4">
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">
                    1
                  </div>
                  <span className="text-gray-700">
                    Comprehensive property assessment and evaluation with market
                    rate analysis
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">
                    2
                  </div>
                  <span className="text-gray-700">
                    Detailed service agreement and terms establishment aligned
                    with industry best practices
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">
                    3
                  </div>
                  <span className="text-gray-700">
                    Professional tenant placement and screening using digital
                    KYC and credit verification
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">
                    4
                  </div>
                  <span className="text-gray-700">
                    Ongoing property management with monthly inspections and
                    maintenance coordination
                  </span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">
                    5
                  </div>
                  <span className="text-gray-700">
                    Regular reporting with transparent communication through
                    digital dashboard and monthly statements
                  </span>
                </li>
              </ol>
            </div>
          </div>

          {/* Live Testimonials Section */}
          <div className="mt-16 bg-gray-50 rounded-2xl p-8">
            <h3 className="text-2xl font-bold text-center mb-8 text-gray-900">
              What Our Clients Say
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              {testimonials.map((testimonial, index) => (
                <div
                  key={index}
                  className="bg-white p-6 rounded-xl shadow-md hover:shadow-lg transition-all duration-300"
                >
                  <div className="flex items-center mb-4">
                    <div className="flex text-yellow-400">
                      {[...Array(testimonial.rating)].map((_, i) => (
                        <Star key={i} className="w-4 h-4 fill-current" />
                      ))}
                    </div>
                  </div>
                  <p className="text-gray-600 mb-4 italic">
                    "{testimonial.text}"
                  </p>
                  <div className="border-t pt-4">
                    <p className="font-semibold text-gray-900">
                      {testimonial.name}
                    </p>
                    <p className="text-sm text-gray-500">
                      {testimonial.location}
                    </p>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {/* Choose Your Service Plan - Redesigned */}
          <section className="py-16 bg-gradient-to-b from-gray-50 to-white">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
              {/* Section Header */}
              <div className="text-center mb-12">
                <h2 className="text-4xl font-bold text-gray-900 mb-4">
                  Choose Your Service Plan
                </h2>
                <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                  Flexible pricing options tailored for every property owner
                </p>
              </div>

              {/* Plans Grid */}
              <div className="grid grid-cols-1 md:grid-cols-3 gap-8 max-w-6xl mx-auto">
                {/* Basic Plan */}
                <div className="relative plan-card">
                  <div className="plan-inner">
                    <div className="p-0 plan-content">
                      <div className="p-6">
                        <h3 className="text-2xl font-bold text-gray-900 mb-2">
                          Basic Plan
                        </h3>
                        <p className="text-gray-600 mb-6">
                          8% of rental income
                        </p>

                        <ul className="space-y-4 plan-features">
                          <li className="flex items-start gap-3">
                            <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                            <span className="text-gray-700">
                              Professional rent collection
                            </span>
                          </li>
                          <li className="flex items-start gap-3">
                            <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                            <span className="text-gray-700">
                              Comprehensive tenant screening
                            </span>
                          </li>
                          <li className="flex items-start gap-3">
                            <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                            <span className="text-gray-700">
                              Basic maintenance coordination
                            </span>
                          </li>
                          <li className="flex items-start gap-3">
                            <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                            <span className="text-gray-700">
                              Monthly performance reports
                            </span>
                          </li>
                        </ul>
                      </div>
                    </div>

                    <div className="plan-footer p-6">
                      {/* add data-plan attribute and inline styles as defensive measure */}
                      <button
                        data-plan="Basic Plan"
                        onClick={() => onChoosePlan("Basic Plan")}
                        className="cta-primary cta-blue"
                        aria-label="Choose Basic Plan"
                        type="button"
                        style={{
                          opacity: 1,
                          pointerEvents: "auto",
                        }}
                      >
                        Get Started
                      </button>

                      <p className="text-center text-sm text-gray-500 mt-3">
                        <button
                          onClick={showAuthModalHandler}
                          className="plan-link text-blue-600 hover:text-blue-700"
                        >
                          Compare plans
                        </button>
                      </p>
                    </div>
                  </div>
                </div>

                {/* Premium Plan - Most Popular */}
                <div className="relative plan-card">
                  <div className="absolute -top-4 left-1/2 transform -translate-x-1/2 z-10 most-popular-badge">
                    <span className="bg-blue-600 text-white text-xs px-3 py-1 rounded-full shadow-lg font-medium">
                      Most Popular
                    </span>
                  </div>

                  <div className="plan-inner border-2 border-blue-200">
                    <div className="p-6 pt-12">
                      <div className="mb-6 text-center">
                        <h3 className="text-2xl font-bold text-gray-900 mb-2">
                          Premium Plan
                        </h3>
                        <p className="text-gray-600">10% of rental income</p>
                      </div>

                      <ul className="space-y-4 mb-8 plan-features">
                        <li className="flex items-start gap-3">
                          <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                          <span className="text-gray-700">
                            Everything in Basic Plan
                          </span>
                        </li>
                        <li className="flex items-start gap-3">
                          <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                          <span className="text-gray-700">
                            24/7 maintenance hotline
                          </span>
                        </li>
                        <li className="flex items-start gap-3">
                          <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                          <span className="text-gray-700">
                            Legal support & documentation
                          </span>
                        </li>
                        <li className="flex items-start gap-3">
                          <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                          <span className="text-gray-700">
                            Owner portal access
                          </span>
                        </li>
                        <li className="flex items-start gap-3">
                          <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                          <span className="text-gray-700">
                            Dedicated account manager
                          </span>
                        </li>
                      </ul>
                    </div>

                    <div className="plan-footer p-6 pt-0">
                      <button
                        data-plan="Premium Plan"
                        onClick={() => onChoosePlan("Premium Plan")}
                        className="cta-primary cta-blue"
                        aria-label="Choose Premium Plan"
                        type="button"
                        style={{
                          opacity: 1,
                          pointerEvents: "auto",
                        }}
                      >
                        Get Started
                      </button>

                      <p className="text-center text-sm text-gray-500 mt-3">
                        <button
                          onClick={showAuthModalHandler}
                          className="plan-link text-blue-600 hover:text-blue-700"
                        >
                          View details
                        </button>
                      </p>
                    </div>
                  </div>
                </div>

                {/* NRI Special Plan */}
                <div className="relative plan-card">
                  <div className="plan-inner">
                    <div className="p-6">
                      <h3 className="text-2xl font-bold text-gray-900 mb-2">
                        NRI Special
                      </h3>
                      <p className="text-gray-600 mb-6">12% of rental income</p>

                      <ul className="space-y-4 mb-8 plan-features">
                        <li className="flex items-start gap-3">
                          <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                          <span className="text-gray-700">
                            Everything in Premium Plan
                          </span>
                        </li>
                        <li className="flex items-start gap-3">
                          <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                          <span className="text-gray-700">
                            NRI tax compliance support
                          </span>
                        </li>
                        <li className="flex items-start gap-3">
                          <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                          <span className="text-gray-700">
                            International banking assistance
                          </span>
                        </li>
                        <li className="flex items-start gap-3">
                          <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                          <span className="text-gray-700">
                            Multi-timezone support
                          </span>
                        </li>
                        <li className="flex items-start gap-3">
                          <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                          <span className="text-gray-700">
                            Priority service guarantee
                          </span>
                        </li>
                      </ul>
                    </div>

                    <div className="plan-footer p-6 pt-0">
                      <button
                        data-plan="NRI Special"
                        onClick={() => onChoosePlan("NRI Special")}
                        className="cta-primary cta-nri"
                        aria-label="Choose NRI Special Plan"
                        type="button"
                        style={{
                          opacity: 1,
                          pointerEvents: "auto",
                        }}
                      >
                        Get Started
                      </button>

                      <p className="text-center text-sm text-gray-500 mt-3">
                        <button
                          onClick={showAuthModalHandler}
                          className="plan-link text-purple-600 hover:text-purple-700"
                        >
                          NRI benefits
                        </button>
                      </p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </section>

          {/* Auth Modal */}
          {showAuthModal && (
            <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
              <div className="bg-white rounded-2xl p-8 max-w-md w-full shadow-2xl relative">
                <h3 className="text-2xl font-bold text-gray-900 mb-4">
                  Get Started Today
                </h3>
                <p className="text-gray-600 mb-6">
                  Sign up or log in to choose your property management plan and
                  start maximizing your rental income.
                </p>

                <div className="space-y-3">
                  <button
                    onClick={() => {
                      setShowAuthModal(false);
                      navigate("/signup");
                    }}
                    className="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-6 rounded-lg transition-colors"
                  >
                    Create Account
                  </button>
                  <button
                    onClick={() => {
                      setShowAuthModal(false);
                      navigate("/login");
                    }}
                    className="w-full border border-gray-300 text-gray-700 hover:bg-gray-50 font-semibold py-3 px-6 rounded-lg transition-colors"
                  >
                    Sign In
                  </button>
                </div>

                <button
                  onClick={() => setShowAuthModal(false)}
                  className="absolute top-4 right-4 text-gray-400 hover:text-gray-600"
                  aria-label="Close modal"
                >
                  ✕
                </button>
              </div>
            </div>
          )}

          {/* Live Stats Dashboard (preserved) */}
          <div className="mt-16 bg-white rounded-2xl p-8 border border-gray-200">
            <h3 className="text-2xl font-bold text-center mb-8 text-gray-900">
              Live Performance Dashboard
            </h3>
            <div className="grid grid-cols-2 md:grid-cols-3 gap-6">
              <div className="text-center p-4 bg-blue-50 rounded-lg">
                <div className="text-3xl font-bold text-blue-600 mb-2">
                  {isLoading
                    ? "..."
                    : liveStats.activeContracts.toLocaleString()}
                </div>
                <div className="text-gray-600">Active Contracts</div>
                <div className="flex items-center justify-center gap-1 mt-2">
                  <Clock className="w-3 h-3 text-green-500" />
                  <span className="text-xs text-green-600">Live</span>
                </div>
              </div>
              <div className="text-center p-4 bg-green-50 rounded-lg">
                <div className="text-3xl font-bold text-green-600 mb-2">
                  {isLoading ? "..." : `${liveStats.clientSatisfaction}%`}
                </div>
                <div className="text-gray-600">Satisfaction Rate</div>
                <div className="flex items-center justify-center gap-1 mt-2">
                  <TrendingUp className="w-3 h-3 text-green-500" />
                  <span className="text-xs text-green-600">Trending Up</span>
                </div>
              </div>
              <div className="text-center p-4 bg-purple-50 rounded-lg">
                <div className="text-3xl font-bold text-purple-600 mb-2">
                  {isLoading ? "..." : liveStats.citiesCovered}
                </div>
                <div className="text-gray-600">Cities Covered</div>
                <div className="flex items-center justify-center gap-1 mt-2">
                  <MapPin className="w-3 h-3 text-purple-500" />
                  <span className="text-xs text-purple-600">Pan-India</span>
                </div>
              </div>
            </div>
          </div>

          {/* CTA Section (preserved) */}
          <div className="mt-16 bg-gradient-to-r from-blue-600 to-blue-700 rounded-2xl p-12 text-white text-center shadow-2xl relative overflow-hidden">
            <div className="absolute top-10 right-10 w-32 h-32 bg-white/10 rounded-full blur-2xl"></div>
            <div className="absolute bottom-10 left-10 w-40 h-40 bg-blue-400/20 rounded-full blur-2xl"></div>
            <div className="relative">
              <h2 className="text-4xl font-bold mb-6">Ready to Get Started?</h2>
              <p className="text-blue-100 mb-6 max-w-2xl mx-auto text-lg leading-relaxed">
                Join{" "}
                {isLoading ? "1000+" : liveStats.nriClients.toLocaleString()}{" "}
                satisfied clients who trust NAL with their property management
                needs.
              </p>
              <div className="bg-white/10 backdrop-blur-sm rounded-lg p-4 mb-8 max-w-md mx-auto">
                <div className="text-2xl font-bold">
                  {isLoading ? "..." : `${liveStats.avgRentalYield}%`}
                </div>
                <div className="text-blue-200">
                  Average rental yield increase
                </div>
              </div>
              <div className="flex flex-col sm:flex-row gap-4 justify-center">
                <Button
                  variant="outline"
                  className="border-2 border-white text-white hover:bg-white/10 px-8 py-4 font-semibold text-lg rounded-xl transition-all duration-300"
                >
                  Schedule Consultation
                </Button>
                <Button className="bg-white text-blue-600 hover:bg-gray-100 px-8 py-4 font-semibold text-lg rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:scale-105">
                  Get Started Now
                </Button>
              </div>
            </div>
          </div>
        </div>
      </div>
    </>
  );
}