import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { CheckCircle, Shield, Search, FileText, Upload, ArrowRight, Clock, AlertTriangle } from 'lucide-react';
import { Button } from '../../components/ui/button';
import { titleSearchService, TitleSearchRequest } from '../../services/titleSearchService';

export function TitleSearch() {
  const navigate = useNavigate();
  const [searchForm, setSearchForm] = useState<TitleSearchRequest>({
    property_id: '',
    survey_number: '',
    owner_name: '',
    address: '',
    search_type: 'comprehensive'
  });
  const [isLoading, setIsLoading] = useState(false);
  const [showInstantVerification, setShowInstantVerification] = useState(false);
  const [instantFiles, setInstantFiles] = useState<{
    titleDeed?: File;
    saleDeed?: File;
    encumbrance?: File;
  }>({});

  const handleTitleSearchSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);

    try {
      const result = await titleSearchService.initiateTitleSearch(searchForm);
      
      if (result.success) {
        // Navigate to document verification with title search context
        navigate(`/docverification?title_search_id=${result.search_id}&property_id=${searchForm.property_id}`);
      } else {
        alert('Failed to initiate title search. Please try again.');
      }
    } catch (error) {
      console.error('Title search initiation failed:', error);
      alert('Failed to initiate title search. Please check your details and try again.');
    } finally {
      setIsLoading(false);
    }
  };

  const handleInstantVerification = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!instantFiles.titleDeed || !instantFiles.saleDeed || !searchForm.property_id) {
      alert('Please provide property ID and upload both title deed and sale deed.');
      return;
    }

    setIsLoading(true);

    try {
      const result = await titleSearchService.instantTitleVerification(
        searchForm.property_id,
        instantFiles.titleDeed,
        instantFiles.saleDeed,
        instantFiles.encumbrance
      );

      if (result.success) {
        // Navigate to instant results page
        navigate(`/title-search/instant-results`, { state: { results: result } });
      } else {
        alert('Instant verification failed. Please try again.');
      }
    } catch (error) {
      console.error('Instant verification failed:', error);
      alert('Instant verification failed. Please check your files and try again.');
    } finally {
      setIsLoading(false);
      setShowInstantVerification(false);
    }
  };

  return (
    <div className="min-h-screen bg-white">
      {/* Hero Section */}
      <div className="relative bg-gradient-to-br from-blue-600 via-blue-700 to-indigo-800 py-20 mt-3 overflow-hidden">
        <div className="absolute top-20 left-10 w-72 h-72 bg-white/5 rounded-full blur-3xl"></div>
        <div className="absolute bottom-20 right-10 w-96 h-96 bg-blue-400/10 rounded-full blur-3xl"></div>
        <div className="relative max-w-5xl mx-auto px-4 text-center">
          <h1 className="text-5xl font-bold text-black mb-6">Title Search & Property History Check</h1>
          <p className="text-xl text-blue-100 mb-8 leading-relaxed">
            Know the Property Before You Sign. NAL performs a deep background check to ensure the property has a clean history and clear ownership chain. With document fraud accounting for 12-18% of property disputes and over 6.5 million people affected by land conflicts, our comprehensive verification protects your investment.
          </p>
          <div className="inline-block bg-white/20 backdrop-blur-sm text-black px-8 py-4 rounded-xl font-semibold text-lg shadow-lg border border-white/30 mb-8">
            Basic Title Search: ₹4,999 | Comprehensive Report: ₹7,999
          </div>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button 
              variant="outline" 
              className="border-2 border-white text-white hover:bg-white/10 px-8 py-4 font-semibold text-lg rounded-xl transition-all duration-300"
              onClick={() => setShowInstantVerification(true)}
            >
              <Upload className="w-5 h-5 mr-2" />
              Instant Verification
            </Button>
            <Button 
              className="bg-white text-blue-600 hover:bg-gray-100 px-8 py-4 font-semibold text-lg rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:scale-105"
              onClick={() => document.getElementById('title-search-form')?.scrollIntoView({ behavior: 'smooth' })}
            >
              <Search className="w-5 h-5 mr-2" />
              Start Comprehensive Search
            </Button>
          </div>
        </div>
      </div>

      {/* Service Images Section */}
      <div className="bg-gradient-to-b from-white to-gray-50 py-20">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="text-center bg-white p-8 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1 group">
              <img 
                src="https://images.unsplash.com/photo-1436450412740-6b988f486c6b?w=400&h=300&fit=crop"
                alt="Legal Verification" 
                className="w-full h-48 object-cover rounded-xl mb-6 shadow-md group-hover:scale-105 transition-transform duration-300"
              />
              <h3 className="text-xl font-bold text-gray-900 mb-3">Legal Verification</h3>
              <p className="text-gray-600">Prevent fraud in 12-18% dispute cases</p>
            </div>
            
            <div className="text-center bg-white p-8 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1 group">
              <img 
                src="https://images.unsplash.com/photo-1554224155-6726b3ff858f?w=400&h=300&fit=crop"
                alt="Digital Verification" 
                className="w-full h-48 object-cover rounded-xl mb-6 shadow-md group-hover:scale-105 transition-transform duration-300"
              />
              <h3 className="text-xl font-bold text-gray-900 mb-3">Digital Verification</h3>
              <p className="text-gray-600">30-year history with blockchain validation</p>
            </div>
            
            <div className="text-center bg-white p-8 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1 group">
              <img 
                src="https://images.unsplash.com/photo-1560472354-b33ff0c44a43?w=400&h=300&fit=crop"
                alt="Fraud Prevention" 
                className="w-full h-48 object-cover rounded-xl mb-6 shadow-md group-hover:scale-105 transition-transform duration-300"
              />
              <h3 className="text-xl font-bold text-gray-900 mb-3">Fraud Prevention</h3>
              <p className="text-gray-600">Reduce risk with 30-year verification</p>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-6xl mx-auto px-4 py-16 -mt-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Features */}
          <div className="bg-white p-8 rounded-2xl border border-gray-200 shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Key Features</h2>
            <ul className="space-y-4">
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Complete ownership chain verification with 30-year historical record analysis</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Encumbrance certificate analysis and validation through government registry systems</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Mortgage and lien verification checks across banking databases</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Litigation and legal dispute investigation—critical as 1.25 lakh RERA complaints were resolved in recent years</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">RERA records validation with instant verification of project registration authenticity</span>
              </li>
            </ul>
          </div>

          {/* Benefits */}
          <div className="bg-white p-8 rounded-2xl border border-gray-200 shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Benefits</h2>
            <ul className="space-y-4">
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Prevents fraud and protects against legal complications—critical as forgery can attract 7 years imprisonment under IPC Section 468</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Ensures safe buying with complete legal transparency in line with Registration Bill 2025</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Eliminates legal surprises and hidden liabilities through comprehensive background checks</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Provides complete peace of mind backed by verification of 943 crore+ digital documents ecosystem</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Ensures clear and marketable title for future transactions with blockchain validation</span>
              </li>
            </ul>
          </div>

          {/* Process */}
          <div className="bg-white p-8 rounded-2xl border border-gray-200 shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">How It Works</h2>
            <ol className="space-y-4">
              <li className="flex items-start gap-3">
                <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">1</div>
                <span className="text-gray-700">Comprehensive document collection and initial verification with AI-powered fraud detection</span>
              </li>
              <li className="flex items-start gap-3">
                <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">2</div>
                <span className="text-gray-700">Registry verification and official record checks across state government databases</span>
              </li>
              <li className="flex items-start gap-3">
                <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">3</div>
                <span className="text-gray-700">Detailed legal investigation including 30-year encumbrance certificate and court record search</span>
              </li>
              <li className="flex items-start gap-3">
                <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">4</div>
                <span className="text-gray-700">Property history compilation analyzing ownership chain and transaction patterns</span>
              </li>
              <li className="flex items-start gap-3">
                <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">5</div>
                <span className="text-gray-700">Clear title report with legal certification delivered within 7-10 days with risk assessment</span>
              </li>
            </ol>
          </div>
        </div>

        {/* Title Search Form */}
        <div id="title-search-form" className="mt-16 bg-white rounded-2xl p-8 shadow-2xl border">
          <div className="text-center mb-8">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Start Your Title Search</h2>
            <p className="text-gray-600 max-w-2xl mx-auto">
              Provide property details to begin comprehensive title verification and property history analysis.
            </p>
          </div>

          <form onSubmit={handleTitleSearchSubmit} className="max-w-2xl mx-auto space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Property ID *
                </label>
                <input
                  type="text"
                  required
                  value={searchForm.property_id}
                  onChange={(e) => setSearchForm(prev => ({ ...prev, property_id: e.target.value }))}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter property ID"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Survey Number
                </label>
                <input
                  type="text"
                  value={searchForm.survey_number}
                  onChange={(e) => setSearchForm(prev => ({ ...prev, survey_number: e.target.value }))}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter survey number"
                />
              </div>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Owner Name
              </label>
              <input
                type="text"
                value={searchForm.owner_name}
                onChange={(e) => setSearchForm(prev => ({ ...prev, owner_name: e.target.value }))}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="Enter current owner name"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Property Address
              </label>
              <textarea
                value={searchForm.address}
                onChange={(e) => setSearchForm(prev => ({ ...prev, address: e.target.value }))}
                rows={3}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="Enter complete property address"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Search Type
              </label>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <label className="flex items-center p-4 border rounded-lg cursor-pointer hover:bg-gray-50">
                  <input
                    type="radio"
                    name="search_type"
                    value="basic"
                    checked={searchForm.search_type === 'basic'}
                    onChange={(e) => setSearchForm(prev => ({ ...prev, search_type: e.target.value as 'basic' | 'comprehensive' }))}
                    className="mr-3"
                  />
                  <div>
                    <div className="font-medium text-gray-900">Basic Search - ₹4,999</div>
                    <div className="text-sm text-gray-600">Essential title verification</div>
                  </div>
                </label>
                
                <label className="flex items-center p-4 border rounded-lg cursor-pointer hover:bg-gray-50 border-blue-200 bg-blue-50">
                  <input
                    type="radio"
                    name="search_type"
                    value="comprehensive"
                    checked={searchForm.search_type === 'comprehensive'}
                    onChange={(e) => setSearchForm(prev => ({ ...prev, search_type: e.target.value as 'basic' | 'comprehensive' }))}
                    className="mr-3"
                  />
                  <div>
                    <div className="font-medium text-gray-900">Comprehensive - ₹7,999</div>
                    <div className="text-sm text-gray-600">Complete analysis + history</div>
                  </div>
                </label>
              </div>
            </div>

            <div className="flex justify-center pt-4">
              <button
                type="submit"
                disabled={isLoading || !searchForm.property_id}
                className="flex items-center px-8 py-4 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed font-semibold text-lg shadow-lg transition-all duration-300"
              >
                {isLoading ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-3"></div>
                    Initiating Search...
                  </>
                ) : (
                  <>
                    <Search className="w-5 h-5 mr-3" />
                    Start Title Search
                    <ArrowRight className="w-5 h-5 ml-3" />
                  </>
                )}
              </button>
            </div>
          </form>
        </div>

        {/* Instant Verification Modal */}
        {showInstantVerification && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
            <div className="bg-white rounded-2xl p-8 max-w-2xl w-full max-h-[90vh] overflow-y-auto">
              <div className="flex items-center justify-between mb-6">
                <h3 className="text-2xl font-bold text-gray-900">Instant Title Verification</h3>
                <button
                  onClick={() => setShowInstantVerification(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  ✕
                </button>
              </div>
              
              <div className="space-y-6">
                <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                  <div className="flex items-start gap-3">
                    <Clock className="w-5 h-5 text-blue-600 mt-0.5" />
                    <div>
                      <h4 className="font-medium text-blue-900">Quick 5-Minute Analysis</h4>
                      <p className="text-sm text-blue-700 mt-1">
                        Upload your key documents for immediate AI-powered title verification and risk assessment.
                      </p>
                    </div>
                  </div>
                </div>

                <form onSubmit={handleInstantVerification} className="space-y-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Property ID *
                    </label>
                    <input
                      type="text"
                      required
                      value={searchForm.property_id}
                      onChange={(e) => setSearchForm(prev => ({ ...prev, property_id: e.target.value }))}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      placeholder="Enter property ID"
                    />
                  </div>

                  <div className="grid gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Title Deed / Mother Deed *
                      </label>
                      <input
                        type="file"
                        required
                        accept=".pdf,.jpg,.jpeg,.png"
                        onChange={(e) => setInstantFiles(prev => ({ ...prev, titleDeed: e.target.files?.[0] }))}
                        className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Sale Deed *
                      </label>
                      <input
                        type="file"
                        required
                        accept=".pdf,.jpg,.jpeg,.png"
                        onChange={(e) => setInstantFiles(prev => ({ ...prev, saleDeed: e.target.files?.[0] }))}
                        className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Encumbrance Certificate (Optional)
                      </label>
                      <input
                        type="file"
                        accept=".pdf,.jpg,.jpeg,.png"
                        onChange={(e) => setInstantFiles(prev => ({ ...prev, encumbrance: e.target.files?.[0] }))}
                        className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                  </div>

                  <div className="flex gap-4 pt-4">
                    <button
                      type="button"
                      onClick={() => setShowInstantVerification(false)}
                      className="flex-1 px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50"
                    >
                      Cancel
                    </button>
                    <button
                      type="submit"
                      disabled={isLoading || !instantFiles.titleDeed || !instantFiles.saleDeed}
                      className="flex-1 px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      {isLoading ? 'Analyzing...' : 'Start Instant Verification'}
                    </button>
                  </div>
                </form>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}