import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { CheckCircle, Shield, Search, FileText, Upload, ArrowRight, Clock, AlertTriangle } from 'lucide-react';
import { titleSearchService, TitleSearchRequest } from '../../services/titleSearchService';
import { getProperties } from '../../api/properties';

// Helper function to get authentication headers for all user types
const getAuthHeaders = (): HeadersInit => {
  // Check for buyer authentication first (access_token)
  const accessToken = localStorage.getItem('access_token');
  if (accessToken) {
    return { 'Authorization': `Bearer ${accessToken}` };
  }

  // Fallback to auth_token for backward compatibility
  const authToken = localStorage.getItem('auth_token');
  if (authToken) {
    return { 'Authorization': `Bearer ${authToken}` };
  }

  // Check for seller authentication
  const sellerToken = localStorage.getItem('sellerToken');
  if (sellerToken) {
    return { 'Authorization': `Bearer ${sellerToken}` };
  }

  // Check for agent authentication
  const agentToken = localStorage.getItem('agentToken');
  if (agentToken) {
    return { 'Authorization': `Bearer ${agentToken}` };
  }

  return {};
};

// Helper function to get user ID for all user types
const getUserId = (): string => {
  // Check for buyer authentication first
  const userStr = localStorage.getItem('user');
  if (userStr) {
    try {
      const user = JSON.parse(userStr);
      const userId = user?.user_id || user?.id;
      if (userId) return userId;
    } catch (e) {
      console.error('Error parsing user data:', e);
    }
  }

  // Check for seller authentication
  const sellerId = localStorage.getItem('sellerId');
  if (sellerId) return sellerId;

  // Check for agent authentication
  const agentId = localStorage.getItem('agentId');
  if (agentId) return agentId;

  return 'guest';
};

export function TitleSearchPremium() {
  const navigate = useNavigate();
  const [searchForm, setSearchForm] = useState<TitleSearchRequest>({
    property_id: '',
    survey_number: '',
    owner_name: '',
    address: '',
    search_type: 'comprehensive'
  });
  const [propertyName, setPropertyName] = useState('');
  const [selectedPropertyId, setSelectedPropertyId] = useState('');
  const [propertySuggestions, setPropertySuggestions] = useState<Array<{ id: number; title: string }>>([]);
  const [propertySearchLoading, setPropertySearchLoading] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [showInstantVerification, setShowInstantVerification] = useState(false);
  const [instantFiles, setInstantFiles] = useState<{
    titleDeed?: File;
    saleDeed?: File;
    encumbrance?: File;
  }>({});

  const resolvePropertyId = async (): Promise<string> => {
    if (selectedPropertyId) {
      return selectedPropertyId;
    }

    const name = propertyName.trim();
    if (!name) {
      throw new Error('Please enter a property name.');
    }

    const result = await getProperties({ search: name, limit: 20 });
    const normalized = name.toLowerCase();
    const match = result.properties.find(
      (property) => property.title?.trim().toLowerCase() === normalized
    );

    if (!match?.id) {
      throw new Error('Property not found. Please select an exact listed property name.');
    }

    const id = match.id.toString();
    setSelectedPropertyId(id);
    setSearchForm((prev) => ({ ...prev, property_id: id }));
    return id;
  };

  useEffect(() => {
    const name = propertyName.trim();
    setSelectedPropertyId('');
    if (!name) {
      setPropertySuggestions([]);
      return;
    }

    const handle = window.setTimeout(async () => {
      try {
        setPropertySearchLoading(true);
        const result = await getProperties({ search: name, limit: 8 });
        const suggestions = result.properties.map((property) => ({
          id: property.id,
          title: property.title
        }));
        setPropertySuggestions(suggestions);
      } catch {
        setPropertySuggestions([]);
      } finally {
        setPropertySearchLoading(false);
      }
    }, 250);

    return () => window.clearTimeout(handle);
  }, [propertyName]);

  const selectProperty = (id: number, title: string) => {
    setPropertyName(title);
    const idStr = id.toString();
    setSelectedPropertyId(idStr);
    setSearchForm((prev) => ({ ...prev, property_id: idStr }));
    setPropertySuggestions([]);
  };

  const handleTitleSearchSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);

    try {
      const propertyId = await resolvePropertyId();
      const requestPayload = { ...searchForm, property_id: propertyId };
      console.log('Submitting title search with data:', searchForm);
      // Debug authentication
      const accessToken = localStorage.getItem('access_token');
      const authToken = localStorage.getItem('auth_token');
      const userStr = localStorage.getItem('user');
      console.log('Auth debug - Access token exists:', !!accessToken);
      console.log('Auth debug - Auth token exists:', !!authToken);
      console.log('Auth debug - User exists:', !!userStr);
      if (userStr) {
        try {
          const user = JSON.parse(userStr);
          console.log('Auth debug - User ID:', user?.user_id || user?.id);
        } catch (e) {
          console.error('Auth debug - User parse error:', e);
        }
      }
      // Try the API call first, let the service handle auth
      const result = await titleSearchService.initiateTitleSearch(requestPayload);
      console.log('Title search result:', result);
      
      if (result && result.success) {
        // Navigate to document verification with title search context
        navigate(`/docverification?title_search_id=${result.search_id}&property_id=${propertyId}`);
      } else {
        const errorMsg = result?.message || 'Failed to initiate title search. Please try again.';
        console.error('Title search failed:', errorMsg);
        alert(errorMsg);
      }
    } catch (error) {
      console.error('Title search initiation failed:', error);
      
      // Handle different error types
      if (error instanceof Error) {
        if (error.message.includes('401')) {
          alert('Please login to continue with title search.');
          navigate('/login');
        } else if (error.message.includes('403')) {
          alert('You do not have permission to perform title search.');
        } else if (error.message.includes('404')) {
          alert('Title search service is currently unavailable.');
        } else {
          alert(`Error: ${error.message}`);
        }
      } else {
        alert('Failed to initiate title search. Please try again.');
      }
    } finally {
      setIsLoading(false);
    }
  };

  const handleInstantVerification = async (e: React.FormEvent) => {
    e.preventDefault();

    setIsLoading(true);

    try {
      // Require logged-in user (buyer/seller/agent)
      const resolvedUserId = getUserId();
      if (resolvedUserId === 'guest') {
        alert('Please login to continue instant verification.');
        navigate('/login');
        return;
      }

      // Require property selection / exact match to resolve ID
      const propertyId = await resolvePropertyId();

      if (!instantFiles.titleDeed || !instantFiles.saleDeed) {
        alert('Please upload both title deed and sale deed.');
        return;
      }
      // Generate proper UUID for session_id
      const sessionId = crypto.randomUUID();

      // Step 1: Upload title deed using regular endpoint with full OCR
      const titleFormData = new FormData();
      titleFormData.append('file', instantFiles.titleDeed);
      titleFormData.append('session_id', sessionId);
      titleFormData.append('user_id', resolvedUserId);
      titleFormData.append('doc_type', 'titleDeed');

      const titleResponse = await fetch('/api/documents/verification/upload', {
        method: 'POST',
        headers: getAuthHeaders(),
        body: titleFormData
      });

      if (!titleResponse.ok) {
        const errorData = await titleResponse.json().catch(() => ({}));
        throw new Error(errorData.detail || 'Failed to upload title deed');
      }

      // Step 2: Upload sale deed using regular endpoint with full OCR
      const saleFormData = new FormData();
      saleFormData.append('file', instantFiles.saleDeed);
      saleFormData.append('session_id', sessionId);
      saleFormData.append('user_id', resolvedUserId);
      saleFormData.append('doc_type', 'saleDeed');

      const saleResponse = await fetch('/api/documents/verification/upload', {
        method: 'POST',
        headers: getAuthHeaders(),
        body: saleFormData
      });

      if (!saleResponse.ok) {
        const errorData = await saleResponse.json().catch(() => ({}));
        throw new Error(errorData.detail || 'Failed to upload sale deed');
      }

      // Step 3: Upload encumbrance if provided using regular endpoint with full OCR
      if (instantFiles.encumbrance) {
        const encFormData = new FormData();
        encFormData.append('file', instantFiles.encumbrance);
        encFormData.append('session_id', sessionId);
        encFormData.append('user_id', resolvedUserId);
        encFormData.append('doc_type', 'encumbrance');

        await fetch('/api/documents/verification/upload', {
          method: 'POST',
          headers: getAuthHeaders(),
          body: encFormData
        });
      }

      // Step 4: Wait for OCR processing to complete (5 seconds)
      await new Promise(resolve => setTimeout(resolve, 5000));

      // Step 5: Call instant verification with session_id and user_id
      const verifyFormData = new FormData();
      verifyFormData.append('session_id', sessionId);
      verifyFormData.append('user_id', resolvedUserId);
      verifyFormData.append('property_id', propertyId);

      const response = await fetch('/api/title-search/instant-verification', {
        method: 'POST',
        headers: getAuthHeaders(),
        body: verifyFormData
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.detail || `HTTP error! status: ${response.status}`);
      }

      const result = await response.json();

      if (result.success) {
        // Navigate to instant results page with session_id
        navigate(`/title-search/instant-results?session_id=${sessionId}`);
      } else {
        alert('Instant verification failed. Please try again.');
      }
    } catch (error) {
      console.error('Instant verification failed:', error);
      alert(`Instant verification failed: ${error}`);
    } finally {
      setIsLoading(false);
      setShowInstantVerification(false);
    }
  };

  return (
    <div className="min-h-screen bg-white">
      <style>{`
        .esign-hero {
          background: linear-gradient(135deg, rgba(30, 41, 59, 0.92) 0%, rgba(51, 65, 85, 0.92) 100%), var(--hero-bg) !important;
          background-size: cover !important;
          background-position: center !important;
          padding: 6rem 0 !important;
          color: white !important;
          text-align: center !important;
        }
        .esign-hero h1 {
          font-size: 3.5rem !important;
          font-weight: 700 !important;
          margin-bottom: 1.5rem !important;
          color: white !important;
          letter-spacing: -0.025em !important;
        }
        .esign-hero p {
          font-size: 1.25rem !important;
          margin-bottom: 2.5rem !important;
          color: #e2e8f0 !important;
          max-width: 50rem !important;
          margin-left: auto !important;
          margin-right: auto !important;
          line-height: 1.7 !important;
        }
        .esign-pricing {
          background: rgba(255, 255, 255, 0.95) !important;
          color: var(--pricing-text) !important;
          padding: 1.25rem 2.5rem !important;
          border-radius: 0.75rem !important;
          font-weight: 600 !important;
          font-size: 1rem !important;
          display: inline-block !important;
          box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15) !important;
        }
      `}</style>

      {/* Hero Section */}
      <header
        className="esign-hero"
        style={
          {
            ['--pricing-text' as string]: '#1e293b',
            ['--hero-bg' as string]:
              "url('https://images.unsplash.com/photo-1450101499163-c8848c66ca85?w=1600&h=900&fit=crop')",
          } as React.CSSProperties
        }
      >
        <div>
          <h1>Title Search &amp; Property History Check</h1>
          <p>
            Know the Property Before You Sign. NAL performs a deep background check to ensure the property has a clean history and clear ownership chain. With document fraud accounting for 12-18% of property disputes and over 6.5 million people affected by land conflicts, our comprehensive verification protects your investment.
          </p>
          <div className="esign-pricing">
            Basic Title Search: ₹4,999 | Comprehensive Report: ₹7,999
          </div>
          <div className="mt-8 flex flex-col sm:flex-row gap-4 justify-center">
            <button
              type="button"
              onClick={() => setShowInstantVerification(true)}
              className="inline-flex items-center justify-center bg-white hover:bg-slate-100 px-8 py-4 font-semibold text-lg rounded-lg shadow-md transition-all duration-300 ring-1 ring-white/40"
              style={{ color: 'var(--pricing-text)' }}
            >
              <Upload className="w-5 h-5 mr-2" />
              Instant Verification
            </button>
            <button
              type="button"
              onClick={() => document.getElementById('title-search-form')?.scrollIntoView({ behavior: 'smooth' })}
              className="inline-flex items-center justify-center bg-white hover:bg-blue-50 px-8 py-4 font-semibold text-lg rounded-lg shadow-md transition-all duration-300"
              style={{ color: 'var(--pricing-text)' }}
            >
              <Search className="w-5 h-5 mr-2" />
              Start Comprehensive Search
            </button>
          </div>
        </div>
      </header>

      {showInstantVerification && (
        <div className="fixed inset-0 z-50 bg-black/50 flex items-center justify-center p-4">
          <div className="w-full max-w-2xl bg-white rounded-lg p-8 shadow-xl border border-gray-200 max-h-[90vh] overflow-y-auto">
            <div className="flex items-center justify-between mb-6">
              <h3 className="text-2xl font-bold text-gray-900">Instant Title Verification</h3>
              <button
                onClick={() => setShowInstantVerification(false)}
                className="text-gray-400 hover:text-gray-600"
                aria-label="Close instant verification"
                type="button"
              >
                ✕
              </button>
            </div>

            <div className="space-y-6">
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <div className="flex items-start gap-3">
                  <Clock className="w-5 h-5 text-blue-600 mt-0.5" />
                  <div>
                    <h4 className="font-medium text-blue-900">Quick 5-Minute Analysis</h4>
                    <p className="text-sm text-blue-700 mt-1">
                      Upload your key documents for immediate AI-powered title verification and risk assessment.
                    </p>
                  </div>
                </div>
              </div>

              <form onSubmit={handleInstantVerification} className="space-y-6">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Property Name *
                  </label>
                  <div className="relative">
                    <input
                      type="text"
                      required
                      value={propertyName}
                      onChange={(e) => setPropertyName(e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      placeholder="Enter property name"
                    />
                    {(propertySearchLoading || propertySuggestions.length > 0) && (
                      <div className="absolute z-10 mt-2 w-full rounded-lg border border-gray-200 bg-white shadow-lg">
                        {propertySearchLoading && (
                          <div className="px-4 py-2 text-sm text-gray-500">Searching...</div>
                        )}
                        {!propertySearchLoading && propertySuggestions.map((property) => (
                          <button
                            key={property.id}
                            type="button"
                            onClick={() => selectProperty(property.id, property.title)}
                            className="w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-50"
                          >
                            {property.title}
                          </button>
                        ))}
                        {!propertySearchLoading && propertySuggestions.length === 0 && (
                          <div className="px-4 py-2 text-sm text-gray-500">No matches found</div>
                        )}
                      </div>
                    )}
                  </div>
                </div>

                <div className="grid gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Title Deed / Mother Deed *
                    </label>
                    <input
                      type="file"
                      required
                      accept=".pdf,.jpg,.jpeg,.png"
                      onChange={(e) => setInstantFiles(prev => ({ ...prev, titleDeed: e.target.files?.[0] }))}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Sale Deed *
                    </label>
                    <input
                      type="file"
                      required
                      accept=".pdf,.jpg,.jpeg,.png"
                      onChange={(e) => setInstantFiles(prev => ({ ...prev, saleDeed: e.target.files?.[0] }))}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Encumbrance Certificate (Optional)
                    </label>
                    <input
                      type="file"
                      accept=".pdf,.jpg,.jpeg,.png"
                      onChange={(e) => setInstantFiles(prev => ({ ...prev, encumbrance: e.target.files?.[0] }))}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    />
                  </div>
                </div>

                <div className="flex gap-4 pt-4">
                  <button
                    type="button"
                    onClick={() => setShowInstantVerification(false)}
                    className="flex-1 px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50"
                  >
                    Cancel
                  </button>
                  <button
                    type="submit"
                    disabled={isLoading || !instantFiles.titleDeed || !instantFiles.saleDeed}
                    className="flex-1 px-6 py-3 bg-blue-600 text-black rounded-lg hover:bg-blue-700 disabled:cursor-not-allowed border border-blue-700"
                  >
                    {isLoading ? 'Analyzing...' : 'Start Instant Verification'}
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      )}

      {/* Service Images Section */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="text-center bg-gray-50 p-6 rounded-lg shadow-md">
              <img 
                src="https://images.unsplash.com/photo-1436450412740-6b988f486c6b?w=400&h=300&fit=crop"
                alt="Legal Verification" 
                className="w-full h-48 object-cover rounded-lg mb-4"
              />
              <h3 className="text-xl font-bold text-gray-900 mb-3">Legal Verification</h3>
              <p className="text-gray-600">Prevent fraud in 12-18% dispute cases</p>
            </div>
            
            <div className="text-center bg-gray-50 p-6 rounded-lg shadow-md">
              <img 
                src="https://images.unsplash.com/photo-1554224155-6726b3ff858f?w=400&h=300&fit=crop"
                alt="Digital Verification" 
                className="w-full h-48 object-cover rounded-lg mb-4"
              />
              <h3 className="text-xl font-bold text-gray-900 mb-3">Digital Verification</h3>
              <p className="text-gray-600">30-year history with blockchain validation</p>
            </div>
            
            <div className="text-center bg-gray-50 p-6 rounded-lg shadow-md">
              <img 
                src="https://images.unsplash.com/photo-1560472354-b33ff0c44a43?w=400&h=300&fit=crop"
                alt="Fraud Prevention" 
                className="w-full h-48 object-cover rounded-lg mb-4"
              />
              <h3 className="text-xl font-bold text-gray-900 mb-3">Fraud Prevention</h3>
              <p className="text-gray-600">Reduce risk with 30-year verification</p>
            </div>
          </div>
        </div>
      </div>

      <div className="bg-gray-50 py-16">
        <div className="max-w-6xl mx-auto px-4">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Features */}
          <div className="bg-white p-8 rounded-lg shadow-md">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Key Features</h2>
            <ul className="space-y-4">
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Complete ownership chain verification with 30-year historical record analysis</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Encumbrance certificate analysis and validation through government registry systems</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Mortgage and lien verification checks across banking databases</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Litigation and legal dispute investigation—critical as 1.25 lakh RERA complaints were resolved in recent years</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">RERA records validation with instant verification of project registration authenticity</span>
              </li>
            </ul>
          </div>

          {/* Benefits */}
          <div className="bg-white p-8 rounded-lg shadow-md">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Benefits</h2>
            <ul className="space-y-4">
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Prevents fraud and protects against legal complications—critical as forgery can attract 7 years imprisonment under IPC Section 468</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Ensures safe buying with complete legal transparency in line with Registration Bill 2025</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Eliminates legal surprises and hidden liabilities through comprehensive background checks</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Provides complete peace of mind backed by verification of 943 crore+ digital documents ecosystem</span>
              </li>
              <li className="flex items-start gap-3">
                <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                <span className="text-gray-700">Ensures clear and marketable title for future transactions with blockchain validation</span>
              </li>
            </ul>
          </div>

          {/* Process */}
          <div className="bg-white p-8 rounded-lg shadow-md">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">How It Works</h2>
            <ol className="space-y-4">
              <li className="flex items-start gap-3">
                <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">1</div>
                <span className="text-gray-700">Comprehensive document collection and initial verification with AI-powered fraud detection</span>
              </li>
              <li className="flex items-start gap-3">
                <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">2</div>
                <span className="text-gray-700">Registry verification and official record checks across state government databases</span>
              </li>
              <li className="flex items-start gap-3">
                <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">3</div>
                <span className="text-gray-700">Detailed legal investigation including 30-year encumbrance certificate and court record search</span>
              </li>
              <li className="flex items-start gap-3">
                <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">4</div>
                <span className="text-gray-700">Property history compilation analyzing ownership chain and transaction patterns</span>
              </li>
              <li className="flex items-start gap-3">
                <div className="w-6 h-6 bg-blue-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">5</div>
                <span className="text-gray-700">Clear title report with legal certification delivered within 7-10 days with risk assessment</span>
              </li>
            </ol>
          </div>
        </div>

          {/* Title Search Form */}
          <div id="title-search-form" className="mt-16 bg-white rounded-lg p-8 shadow-md border border-gray-200">
            <div className="text-center mb-8">
              <h2 className="text-3xl font-bold text-gray-900 mb-4">Start Your Title Search</h2>
              <p className="text-gray-600 max-w-2xl mx-auto">
                Provide property details to begin comprehensive title verification and property history analysis.
              </p>
            </div>

            <form onSubmit={handleTitleSearchSubmit} className="max-w-2xl mx-auto space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Property Name *
                    </label>
                    <div className="relative">
                      <input
                        type="text"
                        required
                        value={propertyName}
                        onChange={(e) => setPropertyName(e.target.value)}
                        className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="Enter property name"
                      />
                      {(propertySearchLoading || propertySuggestions.length > 0) && (
                        <div className="absolute z-10 mt-2 w-full rounded-lg border border-gray-200 bg-white shadow-lg">
                          {propertySearchLoading && (
                            <div className="px-4 py-2 text-sm text-gray-500">Searching...</div>
                          )}
                          {!propertySearchLoading && propertySuggestions.map((property) => (
                            <button
                              key={property.id}
                              type="button"
                              onClick={() => selectProperty(property.id, property.title)}
                              className="w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-50"
                            >
                              {property.title}
                            </button>
                          ))}
                          {!propertySearchLoading && propertySuggestions.length === 0 && (
                            <div className="px-4 py-2 text-sm text-gray-500">No matches found</div>
                          )}
                        </div>
                      )}
                    </div>
                  </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Survey Number
                </label>
                <input
                  type="text"
                  value={searchForm.survey_number}
                  onChange={(e) => setSearchForm(prev => ({ ...prev, survey_number: e.target.value }))}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter survey number"
                />
              </div>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Owner Name
              </label>
              <input
                type="text"
                value={searchForm.owner_name}
                onChange={(e) => setSearchForm(prev => ({ ...prev, owner_name: e.target.value }))}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="Enter current owner name"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Property Address
              </label>
              <textarea
                value={searchForm.address}
                onChange={(e) => setSearchForm(prev => ({ ...prev, address: e.target.value }))}
                rows={3}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="Enter complete property address"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Search Type
              </label>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <label className="flex items-center p-4 border rounded-lg cursor-pointer hover:bg-gray-50">
                  <input
                    type="radio"
                    name="search_type"
                    value="basic"
                    checked={searchForm.search_type === 'basic'}
                    onChange={(e) => setSearchForm(prev => ({ ...prev, search_type: e.target.value as 'basic' | 'comprehensive' }))}
                    className="mr-3"
                  />
                  <div>
                    <div className="font-medium text-gray-900">Basic Search - ₹4,999</div>
                    <div className="text-sm text-gray-600">Essential title verification</div>
                  </div>
                </label>
                
                <label className="flex items-center p-4 border rounded-lg cursor-pointer hover:bg-gray-50 border-blue-200 bg-blue-50">
                  <input
                    type="radio"
                    name="search_type"
                    value="comprehensive"
                    checked={searchForm.search_type === 'comprehensive'}
                    onChange={(e) => setSearchForm(prev => ({ ...prev, search_type: e.target.value as 'basic' | 'comprehensive' }))}
                    className="mr-3"
                  />
                  <div>
                    <div className="font-medium text-gray-900">Comprehensive - ₹7,999</div>
                    <div className="text-sm text-gray-600">Complete analysis + history</div>
                  </div>
                </label>
              </div>
            </div>

            <div className="flex justify-center pt-4">
              <button
                type="submit"
                disabled={isLoading || !propertyName.trim()}
                className="flex items-center px-8 py-4 bg-blue-600 text-black rounded-lg hover:bg-blue-700  disabled:cursor-not-allowed font-semibold text-lg shadow-lg transition-all duration-300"
              >
                {isLoading ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-3"></div>
                    Initiating Search...
                  </>
                ) : (
                  <>
                    <Search className="w-5 h-5 mr-3" />
                    Start Title Search
                    <ArrowRight className="w-5 h-5 ml-3" />
                  </>
                )}
              </button>
            </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
}
