import React, { useState, useEffect } from 'react';
import { useSearchParams, useNavigate } from 'react-router-dom';
import { CheckCircle, AlertTriangle, FileText, Download, ArrowLeft } from 'lucide-react';
import { titleSearchService } from '../../services/titleSearchService';

export function TitleSearchResults() {
  const [searchParams] = useSearchParams();
  const navigate = useNavigate();
  const sessionId = searchParams.get('session_id');
  const [results, setResults] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    if (sessionId) {
      fetchResults();
    }
  }, [sessionId]);

  const fetchResults = async () => {
    try {
      setLoading(true);
      const data = await titleSearchService.instantTitleVerificationBySession(sessionId!);
      setResults(data);
    } catch (err: any) {
      setError(err.message || 'Failed to fetch results');
    } finally {
      setLoading(false);
    }
  };

  const getRiskColor = (riskLevel: string) => {
    switch (riskLevel?.toLowerCase()) {
      case 'low': return 'text-green-600 bg-green-50 border-green-200';
      case 'medium': return 'text-yellow-600 bg-yellow-50 border-yellow-200';
      case 'high': return 'text-orange-600 bg-orange-50 border-orange-200';
      case 'critical': return 'text-red-600 bg-red-50 border-red-200';
      default: return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading title search results...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <AlertTriangle className="h-12 w-12 text-red-500 mx-auto mb-4" />
          <h2 className="text-xl font-semibold text-gray-900 mb-2">Error Loading Results</h2>
          <p className="text-gray-600 mb-4">{error}</p>
          <button
            onClick={() => navigate('/title-search')}
            className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700"
          >
            Back to Title Search
          </button>
        </div>
      </div>
    );
  }

  if (!results) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
          <h2 className="text-xl font-semibold text-gray-900 mb-2">No Results Found</h2>
          <p className="text-gray-600 mb-4">Unable to find results for this session.</p>
          <button
            onClick={() => navigate('/title-search')}
            className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700"
          >
            Start New Search
          </button>
        </div>
      </div>
    );
  }

  const { search_results } = results;
  const reportData = search_results?.report_data;

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-6xl mx-auto px-4">
        {/* Header */}
        <div className="mb-8">
          <button
            onClick={() => navigate('/title-search')}
            className="flex items-center text-blue-600 hover:text-blue-700 mb-4"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Title Search
          </button>
          <h1 className="text-3xl font-bold text-gray-900">Title Search Results</h1>
          <p className="text-gray-600 mt-2">Comprehensive property title analysis and verification report</p>
        </div>

        {/* Enhanced Progress Indicator */}
        <div className="bg-white rounded-xl shadow-sm border p-6 mb-8">
          <h2 className="text-xl font-semibold text-gray-900 mb-4">Title Analysis Progress</h2>
          <div className="relative">
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm font-medium text-gray-700">Analysis Completion</span>
              <span className="text-sm font-medium text-gray-700">100%</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div className="bg-blue-600 h-2 rounded-full transition-all duration-1000 ease-out" style={{width: '100%'}}></div>
            </div>
            <div className="flex justify-between mt-2 text-xs text-gray-500">
              <span>✓ Documents Processed</span>
              <span>✓ Intelligence Analysis</span>
              <span>✓ Risk Assessment</span>
              <span>✓ Report Generated</span>
            </div>
          </div>
        </div>

        {/* Summary Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <div className="bg-white p-6 rounded-xl shadow-sm border">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Title Integrity Score</p>
                <p className="text-3xl font-bold text-gray-900">{results.instant_title_score}%</p>
              </div>
              <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                <CheckCircle className="w-6 h-6 text-blue-600" />
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-xl shadow-sm border">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Risk Level</p>
                <span className={`inline-block px-3 py-1 rounded-full text-sm font-medium border ${getRiskColor(results.risk_level)}`}>
                  {results.risk_level}
                </span>
              </div>
              <div className="w-12 h-12 bg-orange-100 rounded-full flex items-center justify-center">
                <AlertTriangle className="w-6 h-6 text-orange-600" />
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-xl shadow-sm border">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Documents Analyzed</p>
                <p className="text-3xl font-bold text-gray-900">{results.documents_analyzed}</p>
              </div>
              <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
                <FileText className="w-6 h-6 text-green-600" />
              </div>
            </div>
          </div>
        </div>

        {/* Professional Summary Card */}
        <div className="bg-gradient-to-r from-gray-50 to-blue-50 rounded-xl border p-6 mb-8">
          <div className="flex items-start justify-between">
            <div className="flex-1">
              <h2 className="text-xl font-semibold text-gray-900 mb-2">Professional Assessment</h2>
              <p className="text-gray-600 mb-4">
                Based on our comprehensive analysis of {results.documents_analyzed} documents, 
                this property shows a <span className="font-semibold">{results.instant_title_score}% title integrity score</span> 
                with <span className={`font-semibold ${
                  results.risk_level?.toLowerCase() === 'low' ? 'text-green-600' :
                  results.risk_level?.toLowerCase() === 'medium' ? 'text-yellow-600' :
                  'text-red-600'
                }`}>{results.risk_level?.toLowerCase()} risk level</span>.
              </p>
              <div className="flex items-center text-sm text-gray-500">
                <CheckCircle className="w-4 h-4 mr-1" />
                <span>Analysis completed on {new Date().toLocaleDateString()}</span>
              </div>
            </div>
            <div className="ml-6">
              <div className={`w-20 h-20 rounded-full flex items-center justify-center text-2xl font-bold ${
                results.instant_title_score >= 80 ? 'bg-green-100 text-green-600' :
                results.instant_title_score >= 60 ? 'bg-yellow-100 text-yellow-600' :
                'bg-red-100 text-red-600'
              }`}>
                {results.instant_title_score}%
              </div>
            </div>
          </div>
        </div>

        {/* Property Details */}
        {results.property_details && (
          <div className="bg-white rounded-xl shadow-sm border p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4">Property Details</h2>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <p className="text-sm text-gray-600">Property ID</p>
                <p className="font-medium">{results.property_details.property_id || 'N/A'}</p>
              </div>
              <div>
                <p className="text-sm text-gray-600">Survey Number</p>
                <p className="font-medium">{results.property_details.survey_number || 'N/A'}</p>
              </div>
              <div>
                <p className="text-sm text-gray-600">Owner Name</p>
                <p className="font-medium">{results.property_details.owner_name || 'N/A'}</p>
              </div>
              <div>
                <p className="text-sm text-gray-600">Document Types</p>
                <p className="font-medium">{results.property_details.document_types?.join(', ') || 'N/A'}</p>
              </div>
            </div>
          </div>
        )}

        {/* Result Interpretation Guide */}
        <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl border border-blue-200 p-6 mb-8">
          <h2 className="text-xl font-semibold text-gray-900 mb-4 flex items-center">
            <AlertTriangle className="w-5 h-5 mr-2 text-blue-600" />
            Understanding Your Results
          </h2>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="bg-white p-4 rounded-lg border border-blue-100">
              <div className="flex items-center mb-2">
                <div className="w-3 h-3 bg-green-500 rounded-full mr-2"></div>
                <span className="font-medium text-green-700">80-100% Score</span>
              </div>
              <p className="text-sm text-gray-600">Excellent title integrity. Property is safe for investment with minimal legal risks.</p>
            </div>
            <div className="bg-white p-4 rounded-lg border border-blue-100">
              <div className="flex items-center mb-2">
                <div className="w-3 h-3 bg-yellow-500 rounded-full mr-2"></div>
                <span className="font-medium text-yellow-700">60-79% Score</span>
              </div>
              <p className="text-sm text-gray-600">Good title with minor issues. Proceed with additional due diligence.</p>
            </div>
            <div className="bg-white p-4 rounded-lg border border-blue-100">
              <div className="flex items-center mb-2">
                <div className="w-3 h-3 bg-red-500 rounded-full mr-2"></div>
                <span className="font-medium text-red-700">Below 60%</span>
              </div>
              <p className="text-sm text-gray-600">Significant concerns identified. Legal consultation strongly recommended.</p>
            </div>
          </div>
        </div>

        {/* Next Steps Guidance */}
        {results.instant_title_score < 80 && (
          <div className="bg-yellow-50 border border-yellow-200 rounded-xl p-6 mb-8">
            <h2 className="text-xl font-semibold text-yellow-800 mb-4 flex items-center">
              <AlertTriangle className="w-5 h-5 mr-2" />
              Recommended Next Steps
            </h2>
            <div className="space-y-3">
              {results.instant_title_score < 60 && (
                <div className="flex items-start">
                  <CheckCircle className="w-5 h-5 text-yellow-600 mr-3 mt-0.5 flex-shrink-0" />
                  <span className="text-yellow-800">Engage a qualified property lawyer for comprehensive legal review</span>
                </div>
              )}
              <div className="flex items-start">
                <CheckCircle className="w-5 h-5 text-yellow-600 mr-3 mt-0.5 flex-shrink-0" />
                <span className="text-yellow-800">Verify all identified issues with original government records</span>
              </div>
              <div className="flex items-start">
                <CheckCircle className="w-5 h-5 text-yellow-600 mr-3 mt-0.5 flex-shrink-0" />
                <span className="text-yellow-800">Consider title insurance to protect against unforeseen legal issues</span>
              </div>
              <div className="flex items-start">
                <CheckCircle className="w-5 h-5 text-yellow-600 mr-3 mt-0.5 flex-shrink-0" />
                <span className="text-yellow-800">Request seller to resolve all pending documentation issues</span>
              </div>
            </div>
          </div>
        )}

        {/* Success Guidance */}
        {results.instant_title_score >= 80 && (
          <div className="bg-green-50 border border-green-200 rounded-xl p-6 mb-8">
            <h2 className="text-xl font-semibold text-green-800 mb-4 flex items-center">
              <CheckCircle className="w-5 h-5 mr-2" />
              Excellent Results - You're Good to Proceed!
            </h2>
            <div className="space-y-3">
              <div className="flex items-start">
                <CheckCircle className="w-5 h-5 text-green-600 mr-3 mt-0.5 flex-shrink-0" />
                <span className="text-green-800">Property shows strong title integrity with minimal legal risks</span>
              </div>
              <div className="flex items-start">
                <CheckCircle className="w-5 h-5 text-green-600 mr-3 mt-0.5 flex-shrink-0" />
                <span className="text-green-800">Standard legal review and registration process can proceed</span>
              </div>
              <div className="flex items-start">
                <CheckCircle className="w-5 h-5 text-green-600 mr-3 mt-0.5 flex-shrink-0" />
                <span className="text-green-800">Property is suitable for bank financing and investment</span>
              </div>
            </div>
          </div>
        )}

        {/* Document Analysis */}
        {results.document_analyses && results.document_analyses.length > 0 && (
          <div className="bg-white rounded-xl shadow-sm border p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4">Document Analysis</h2>
            <div className="space-y-4">
              {results.document_analyses.map((doc: any, index: number) => (
                <div key={index} className="border rounded-lg p-4">
                  <div className="flex items-center justify-between mb-3">
                    <h3 className="font-medium text-gray-900">{doc.document_type}</h3>
                    <span className="text-sm text-gray-500">{doc.filename}</span>
                  </div>
                  <div className="grid grid-cols-3 gap-4">
                    <div>
                      <p className="text-sm text-gray-600">Authenticity</p>
                      <p className="font-medium">{doc.authenticity_score}%</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">Consistency</p>
                      <p className="font-medium">{doc.consistency_score}%</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">Completeness</p>
                      <p className="font-medium">{doc.completeness_score}%</p>
                    </div>
                  </div>
                  {doc.issues_found && doc.issues_found.length > 0 && (
                    <div className="mt-3">
                      <p className="text-sm text-gray-600 mb-1">Issues Found:</p>
                      <ul className="text-sm text-red-600 list-disc list-inside">
                        {doc.issues_found.map((issue: string, i: number) => (
                          <li key={i}>{issue}</li>
                        ))}
                      </ul>
                    </div>
                  )}
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Recommendations */}
        {results.recommendation && (
          <div className="bg-white rounded-xl shadow-sm border p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4">Recommendations</h2>
            <p className="text-gray-700 leading-relaxed">{results.recommendation}</p>
          </div>
        )}

        {/* Risk Assessment */}
        {reportData?.risk_assessment && (
          <div className="bg-white rounded-xl shadow-sm border p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4">Risk Assessment</h2>
            <p className="text-gray-700 leading-relaxed">{reportData.risk_assessment}</p>
          </div>
        )}

        {/* Actions */}
        <div className="flex gap-4">
          <button
            onClick={() => navigate('/title-search')}
            className="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 flex items-center"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            New Search
          </button>
          <button
            onClick={() => window.print()}
            className="bg-gray-600 text-white px-6 py-3 rounded-lg hover:bg-gray-700 flex items-center"
          >
            <Download className="w-4 h-4 mr-2" />
            Download Report
          </button>
        </div>
      </div>
    </div>
  );
}