import React, { useState, useEffect, useCallback, useMemo } from 'react';
import { useSearchParams, useNavigate } from 'react-router-dom';
import { CheckCircle, AlertTriangle, FileText, Download, ArrowLeft, Brain, Shield, TrendingUp } from 'lucide-react';
import { titleSearchService } from '../../services/titleSearchService';

// Type definitions for better type safety
interface IntelligenceInsights {
  field_extraction_confidence?: number;
  consistency_score?: number;
  ownership_integrity?: number;
  risk_factors_identified?: number;
}

interface ReportData {
  intelligence_insights?: IntelligenceInsights;
  executive_summary?: string;
  risk_assessment?: string;
}

interface SearchResults {
  report_data?: ReportData;
}

interface PropertyDetails {
  property_id?: string;
  survey_number?: string;
  owner_name?: string;
  document_types?: string[];
}

interface DocumentAnalysis {
  document_type?: string;
  filename?: string;
  overall_confidence?: number;
  authenticity_score?: number;
  consistency_score?: number;
  completeness_score?: number;
  extracted_fields?: Record<string, { value?: string }>;
  issues_found?: string[];
}

interface RiskFlag {
  description?: string;
  category?: string;
  risk_level?: 'LOW' | 'MEDIUM' | 'HIGH' | 'CRITICAL';
}

interface ConsistencyAnalysis {
  overall_consistency_score?: number;
  risk_flags?: RiskFlag[];
}

interface OwnershipAnalysis {
  chain_length?: number;
  integrity_score?: number;
  ownership_status?: 'VALID' | 'ACCEPTABLE' | 'INVALID';
  missing_years?: string[];
}

interface TitleSearchResultsData {
  search_results?: SearchResults;
  intelligence_version?: string;
  instant_title_score?: number;
  risk_level?: string;
  documents_analyzed?: number;
  property_details?: PropertyDetails;
  document_analyses?: DocumentAnalysis[];
  consistency_analysis?: ConsistencyAnalysis;
  ownership_analysis?: OwnershipAnalysis;
  recommendation?: string;
}

type RiskLevel = 'LOW' | 'MEDIUM' | 'HIGH' | 'CRITICAL';

export function TitleSearchResults() {
  const [searchParams] = useSearchParams();
  const navigate = useNavigate();
  const sessionId = searchParams.get('session_id');
  const [results, setResults] = useState<TitleSearchResultsData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchResults = useCallback(async () => {
    if (!sessionId) {
      setError('Invalid session ID');
      setLoading(false);
      return;
    }

    try {
      setLoading(true);
      setError(null);
      const data = await titleSearchService.instantTitleVerificationBySession(sessionId);
      
      if (!data || typeof data !== 'object') {
        throw new Error('Invalid response data');
      }
      
      setResults(data);
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to fetch results';
      setError(errorMessage);
    } finally {
      setLoading(false);
    }
  }, [sessionId]);

  useEffect(() => {
    fetchResults();
  }, [fetchResults]);

  const normalizeRiskLevel = useCallback((riskLevel?: string): RiskLevel => {
    if (!riskLevel) return 'LOW';
    const normalized = riskLevel.toUpperCase() as RiskLevel;
    return ['LOW', 'MEDIUM', 'HIGH', 'CRITICAL'].includes(normalized) ? normalized : 'LOW';
  }, []);

  const getRiskColor = useCallback((riskLevel?: string) => {
    const normalized = normalizeRiskLevel(riskLevel);
    switch (normalized) {
      case 'LOW': return 'text-green-800 bg-green-100 border-green-300';
      case 'MEDIUM': return 'text-yellow-800 bg-yellow-100 border-yellow-300';
      case 'HIGH': return 'text-orange-800 bg-orange-100 border-orange-300';
      case 'CRITICAL': return 'text-red-800 bg-red-100 border-red-300';
      default: return 'text-gray-800 bg-gray-100 border-gray-300';
    }
  }, [normalizeRiskLevel]);

  const memoizedData = useMemo(() => {
    if (!results) return null;
    
    const searchResults = results.search_results;
    const reportData = searchResults?.report_data;
    const intelligenceInsights = reportData?.intelligence_insights;
    
    return {
      searchResults,
      reportData,
      intelligenceInsights,
      titleScore: results.instant_title_score ?? 0,
      riskLevel: normalizeRiskLevel(results.risk_level),
      documentsAnalyzed: results.documents_analyzed ?? 0,
      intelligenceScore: intelligenceInsights?.field_extraction_confidence 
        ? Math.round(intelligenceInsights.field_extraction_confidence * 100) 
        : 0
    };
  }, [results, normalizeRiskLevel]);

  const handleDownloadReport = useCallback(() => {
    const printElements = document.querySelectorAll('.no-print');
    printElements.forEach(el => el.classList.add('hidden'));
    window.print();
    setTimeout(() => {
      printElements.forEach(el => el.classList.remove('hidden'));
    }, 1000);
  }, []);

  const handleBackToSearch = useCallback(() => {
    navigate('/title-search');
  }, [navigate]);

  if (loading) {
    return (
      <div className="min-h-screen bg-white flex items-center justify-center p-4">
        <div className="text-center max-w-md">
          <div className="relative mb-6">
            <div className="animate-spin rounded-full h-16 w-16 border-4 border-gray-200 border-t-blue-600 mx-auto"></div>
            <div className="absolute inset-0 flex items-center justify-center">
              <Brain className="h-6 w-6 text-blue-600 animate-pulse" />
            </div>
          </div>
          <h2 className="text-xl font-semibold text-gray-900 mb-2">Analyzing Title Documents</h2>
          <p className="text-gray-700">AI is processing your property title verification...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen bg-white flex items-center justify-center p-4">
        <div className="text-center max-w-md">
          <div className="bg-red-100 rounded-full p-4 w-20 h-20 mx-auto mb-6 flex items-center justify-center">
            <AlertTriangle className="h-10 w-10 text-red-700" />
          </div>
          <h2 className="text-xl font-semibold text-gray-900 mb-3">Unable to Load Results</h2>
          <p className="text-gray-700 mb-6 leading-relaxed">{error}</p>
          <div className="space-y-3">
            <button
              onClick={fetchResults}
              className="w-full bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors font-medium"
            >
              Try Again
            </button>
            <button
              onClick={handleBackToSearch}
              className="w-full bg-gray-200 text-gray-900 px-6 py-3 rounded-lg hover:bg-gray-300 transition-colors font-medium"
            >
              Back to Title Search
            </button>
          </div>
        </div>
      </div>
    );
  }

  if (!results || !memoizedData) {
    return (
      <div className="min-h-screen bg-white flex items-center justify-center p-4">
        <div className="text-center max-w-md">
          <div className="bg-gray-200 rounded-full p-4 w-20 h-20 mx-auto mb-6 flex items-center justify-center">
            <FileText className="h-10 w-10 text-gray-600" />
          </div>
          <h2 className="text-xl font-semibold text-gray-900 mb-3">No Results Available</h2>
          <p className="text-gray-700 mb-6 leading-relaxed">
            We couldn't find any results for this session. The session may have expired or the data may not be available.
          </p>
          <button
            onClick={handleBackToSearch}
            className="w-full bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors font-medium"
          >
            Start New Search
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 py-6">
      <div className="max-w-6xl mx-auto px-4">
        {/* Header */}
        <header className="mb-8">
          <button
            onClick={handleBackToSearch}
            className="flex items-center text-blue-600 hover:text-blue-800 mb-4 transition-colors no-print font-medium"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Title Search
          </button>
          <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 mb-2">
                Enhanced Title Search Results
              </h1>
              <p className="text-gray-700">
                AI-powered comprehensive property title analysis with Phase 2 intelligence
              </p>
            </div>
            {results.intelligence_version && (
              <div className="flex items-center px-4 py-2 bg-blue-100 text-blue-800 rounded-full text-sm font-semibold">
                <Brain className="w-4 h-4 mr-2" />
                Intelligence v{results.intelligence_version}
              </div>
            )}
          </div>
        </header>

        {/* Summary Cards */}
        <section className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600 mb-1">Title Integrity Score</p>
                <p className="text-3xl font-bold text-gray-900">
                  {memoizedData.titleScore}%
                </p>
              </div>
              <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                <CheckCircle className="w-6 h-6 text-blue-600" />
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600 mb-1">Risk Level</p>
                <span className={`inline-block px-3 py-1 rounded-full text-sm font-semibold border ${getRiskColor(results.risk_level)}`}>
                  {memoizedData.riskLevel}
                </span>
              </div>
              <div className="w-12 h-12 bg-orange-100 rounded-full flex items-center justify-center">
                <Shield className="w-6 h-6 text-orange-600" />
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600 mb-1">Documents Analyzed</p>
                <p className="text-3xl font-bold text-gray-900">
                  {memoizedData.documentsAnalyzed}
                </p>
              </div>
              <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
                <FileText className="w-6 h-6 text-green-600" />
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600 mb-1">Intelligence Score</p>
                <p className="text-3xl font-bold text-gray-900">
                  {memoizedData.intelligenceScore}%
                </p>
              </div>
              <div className="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center">
                <Brain className="w-6 h-6 text-purple-600" />
              </div>
            </div>
          </div>
        </section>

        {/* Phase 2 Intelligence Overview */}
        {memoizedData.intelligenceInsights && (
          <section className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4 flex items-center">
              <Brain className="w-5 h-5 mr-2 text-blue-600" />
              Intelligence Analysis Overview
            </h2>
            <div className="grid grid-cols-2 lg:grid-cols-4 gap-4">
              <div className="text-center p-4 bg-gray-50 rounded-lg border border-gray-200">
                <p className="text-2xl font-bold text-blue-600">
                  {Math.round((memoizedData.intelligenceInsights.field_extraction_confidence ?? 0) * 100)}%
                </p>
                <p className="text-sm font-medium text-gray-700">Field Extraction</p>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg border border-gray-200">
                <p className="text-2xl font-bold text-green-600">
                  {memoizedData.intelligenceInsights.consistency_score ?? 0}%
                </p>
                <p className="text-sm font-medium text-gray-700">Consistency Score</p>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg border border-gray-200">
                <p className="text-2xl font-bold text-purple-600">
                  {memoizedData.intelligenceInsights.ownership_integrity ?? 0}%
                </p>
                <p className="text-sm font-medium text-gray-700">Ownership Integrity</p>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg border border-gray-200">
                <p className="text-2xl font-bold text-orange-600">
                  {memoizedData.intelligenceInsights.risk_factors_identified ?? 0}
                </p>
                <p className="text-sm font-medium text-gray-700">Risk Factors</p>
              </div>
            </div>
          </section>
        )}

        {/* Property Details */}
        {results.property_details && (
          <section className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4">Property Details</h2>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <p className="text-sm font-medium text-gray-600 mb-1">Property ID</p>
                <p className="font-semibold text-gray-900">{results.property_details.property_id || 'N/A'}</p>
              </div>
              <div>
                <p className="text-sm font-medium text-gray-600 mb-1">Survey Number</p>
                <p className="font-semibold text-gray-900">{results.property_details.survey_number || 'N/A'}</p>
              </div>
              <div>
                <p className="text-sm font-medium text-gray-600 mb-1">Owner Name</p>
                <p className="font-semibold text-gray-900">{results.property_details.owner_name || 'N/A'}</p>
              </div>
              <div>
                <p className="text-sm font-medium text-gray-600 mb-1">Document Types</p>
                <p className="font-semibold text-gray-900">
                  {results.property_details.document_types?.length 
                    ? results.property_details.document_types.join(', ') 
                    : 'N/A'
                  }
                </p>
              </div>
            </div>
          </section>
        )}

        {/* Executive Summary */}
        {memoizedData.reportData?.executive_summary && (
          <section className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4">Executive Summary</h2>
            <p className="text-gray-800 leading-relaxed">{memoizedData.reportData.executive_summary}</p>
          </section>
        )}

        {/* Enhanced Document Analysis */}
        {results.document_analyses?.length && (
          <section className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4">Enhanced Document Analysis</h2>
            <div className="space-y-4">
              {results.document_analyses.map((doc, index) => (
                <div key={`doc-${index}`} className="border border-gray-200 rounded-lg p-4">
                  <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between mb-3 gap-2">
                    <h3 className="font-semibold text-gray-900">{doc.document_type || 'Unknown Document'}</h3>
                    <div className="flex flex-col sm:flex-row sm:items-center gap-2">
                      <span className="text-sm text-gray-700 font-medium">{doc.filename || 'N/A'}</span>
                      {doc.overall_confidence && (
                        <span className={`px-3 py-1 rounded-full text-xs font-semibold ${
                          doc.overall_confidence > 0.8 ? 'bg-green-100 text-green-800' :
                          doc.overall_confidence > 0.6 ? 'bg-yellow-100 text-yellow-800' :
                          'bg-red-100 text-red-800'
                        }`}>
                          {Math.round(doc.overall_confidence * 100)}% Confidence
                        </span>
                      )}
                    </div>
                  </div>
                  
                  <div className="grid grid-cols-3 gap-4 mb-4">
                    <div className="text-center p-3 bg-gray-50 rounded border border-gray-200">
                      <p className="text-sm font-medium text-gray-600 mb-1">Authenticity</p>
                      <p className="font-bold text-gray-900">{Math.round(doc.authenticity_score || 0)}%</p>
                    </div>
                    <div className="text-center p-3 bg-gray-50 rounded border border-gray-200">
                      <p className="text-sm font-medium text-gray-600 mb-1">Consistency</p>
                      <p className="font-bold text-gray-900">{Math.round(doc.consistency_score || 0)}%</p>
                    </div>
                    <div className="text-center p-3 bg-gray-50 rounded border border-gray-200">
                      <p className="text-sm font-medium text-gray-600 mb-1">Completeness</p>
                      <p className="font-bold text-gray-900">{Math.round(doc.completeness_score || 0)}%</p>
                    </div>
                  </div>
                  
                  {doc.extracted_fields && Object.keys(doc.extracted_fields).length > 0 && (
                    <div className="mb-4 p-4 bg-gray-50 rounded border border-gray-200">
                      <h4 className="text-sm font-semibold text-gray-800 mb-3">Extracted Fields</h4>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                        {Object.entries(doc.extracted_fields).map(([field, data]) => (
                          data?.value && (
                            <div key={field} className="flex justify-between items-start gap-2">
                              <span className="text-gray-700 font-medium capitalize">
                                {field.replace('_', ' ')}:
                              </span>
                              <span className="font-semibold text-gray-900 text-right">
                                {data.value}
                              </span>
                            </div>
                          )
                        ))}
                      </div>
                    </div>
                  )}
                  
                  {doc.issues_found?.length && (
                    <div className="p-4 bg-red-50 rounded border border-red-200">
                      <p className="text-sm font-semibold text-red-800 mb-2">Issues Identified:</p>
                      <ul className="space-y-1">
                        {doc.issues_found.map((issue, i) => (
                          <li key={i} className="flex items-start text-sm text-red-700">
                            <span className="text-red-600 mr-2 font-bold">•</span>
                            <span>{issue}</span>
                          </li>
                        ))}
                      </ul>
                    </div>
                  )}
                </div>
              ))}
            </div>
          </section>
        )}

        {/* Phase 2: Consistency Analysis */}
        {results.consistency_analysis && (
          <section className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4 flex items-center">
              <TrendingUp className="w-5 h-5 mr-2 text-blue-600" />
              Document Consistency Analysis
            </h2>
            <div className="mb-6">
              <div className="flex items-center justify-between mb-3">
                <span className="text-gray-700 font-semibold">Overall Consistency Score</span>
                <span className={`font-bold text-xl ${
                  (results.consistency_analysis.overall_consistency_score ?? 0) >= 80 ? 'text-green-600' :
                  (results.consistency_analysis.overall_consistency_score ?? 0) >= 60 ? 'text-yellow-600' :
                  'text-red-600'
                }`}>
                  {results.consistency_analysis.overall_consistency_score ?? 0}%
                </span>
              </div>
              <div className="w-full bg-gray-300 rounded-full h-4">
                <div 
                  className={`h-4 rounded-full transition-all duration-500 ${
                    (results.consistency_analysis.overall_consistency_score ?? 0) >= 80 ? 'bg-green-500' :
                    (results.consistency_analysis.overall_consistency_score ?? 0) >= 60 ? 'bg-yellow-500' :
                    'bg-red-500'
                  }`}
                  style={{ width: `${results.consistency_analysis.overall_consistency_score ?? 0}%` }}
                ></div>
              </div>
            </div>
            
            {results.consistency_analysis.risk_flags?.length && (
              <div>
                <h3 className="font-semibold text-gray-900 mb-3">Risk Flags ({results.consistency_analysis.risk_flags.length})</h3>
                <div className="space-y-3">
                  {results.consistency_analysis.risk_flags.map((flag, index) => {
                    const normalizedRiskLevel = normalizeRiskLevel(flag.risk_level);
                    return (
                      <div key={`risk-${index}`} className={`p-4 rounded-lg border-l-4 ${
                        normalizedRiskLevel === 'CRITICAL' ? 'bg-red-50 border-red-500' :
                        normalizedRiskLevel === 'HIGH' ? 'bg-orange-50 border-orange-500' :
                        normalizedRiskLevel === 'MEDIUM' ? 'bg-yellow-50 border-yellow-500' :
                        'bg-blue-50 border-blue-500'
                      }`}>
                        <div className="flex flex-col lg:flex-row lg:items-start lg:justify-between gap-2">
                          <div className="flex-1">
                            <p className="font-semibold text-gray-900 mb-1">{flag.description || 'Risk identified'}</p>
                            <p className="text-sm font-medium text-gray-700">Category: {flag.category || 'General'}</p>
                          </div>
                          <span className={`px-3 py-1 rounded-full text-xs font-bold inline-block ${
                            normalizedRiskLevel === 'CRITICAL' ? 'bg-red-100 text-red-800' :
                            normalizedRiskLevel === 'HIGH' ? 'bg-orange-100 text-orange-800' :
                            normalizedRiskLevel === 'MEDIUM' ? 'bg-yellow-100 text-yellow-800' :
                            'bg-blue-100 text-blue-800'
                          }`}>
                            {normalizedRiskLevel}
                          </span>
                        </div>
                      </div>
                    );
                  })}
                </div>
              </div>
            )}
          </section>
        )}

        {/* Phase 2: Ownership Analysis */}
        {results.ownership_analysis && (
          <section className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4">Ownership Chain Analysis</h2>
            
            <div className="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
              <div className="text-center p-4 bg-gray-50 rounded-lg border border-gray-200">
                <p className="text-2xl font-bold text-gray-900">
                  {results.ownership_analysis.chain_length ?? 0}
                </p>
                <p className="text-sm font-medium text-gray-700">Chain Length</p>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg border border-gray-200">
                <p className="text-2xl font-bold text-gray-900">
                  {results.ownership_analysis.integrity_score ?? 0}%
                </p>
                <p className="text-sm font-medium text-gray-700">Integrity Score</p>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg border border-gray-200">
                <p className={`text-2xl font-bold ${
                  results.ownership_analysis.ownership_status === 'VALID' ? 'text-green-600' :
                  results.ownership_analysis.ownership_status === 'ACCEPTABLE' ? 'text-yellow-600' :
                  'text-red-600'
                }`}>
                  {results.ownership_analysis.ownership_status || 'UNKNOWN'}
                </p>
                <p className="text-sm font-medium text-gray-700">Status</p>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg border border-gray-200">
                <p className="text-2xl font-bold text-gray-900">
                  {results.ownership_analysis.missing_years?.length ?? 0}
                </p>
                <p className="text-sm font-medium text-gray-700">Missing Periods</p>
              </div>
            </div>
            
            {results.ownership_analysis.missing_years?.length && (
              <div className="p-4 bg-yellow-50 border border-yellow-300 rounded-lg">
                <h4 className="font-semibold text-yellow-800 mb-2">Missing Documentation Periods</h4>
                <p className="text-sm font-medium text-yellow-700">
                  {results.ownership_analysis.missing_years.join(', ')}
                </p>
              </div>
            )}
          </section>
        )}

        {/* Recommendations */}
        {results.recommendation && (
          <section className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4 flex items-center">
              <Brain className="w-5 h-5 mr-2 text-purple-600" />
              AI-Powered Recommendations
            </h2>
            <p className="text-gray-800 leading-relaxed">{results.recommendation}</p>
          </section>
        )}

        {/* Risk Assessment */}
        {memoizedData.reportData?.risk_assessment && (
          <section className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8">
            <h2 className="text-xl font-semibold text-gray-900 mb-4 flex items-center">
              <Shield className="w-5 h-5 mr-2 text-red-600" />
              Enhanced Risk Assessment
            </h2>
            <p className="text-gray-800 leading-relaxed">{memoizedData.reportData.risk_assessment}</p>
          </section>
        )}

        {/* Actions */}
        <footer className="flex flex-col sm:flex-row gap-4 no-print">
          <button
            onClick={handleBackToSearch}
            className="flex-1 sm:flex-none bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors flex items-center justify-center font-semibold focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            New Search
          </button>
          <button
            onClick={handleDownloadReport}
            className="flex-1 sm:flex-none bg-gray-700 text-white px-6 py-3 rounded-lg hover:bg-gray-800 transition-colors flex items-center justify-center font-semibold focus:outline-none focus:ring-2 focus:ring-gray-500 focus:ring-offset-2"
          >
            <Download className="w-4 h-4 mr-2" />
            Download Report
          </button>
        </footer>
      </div>
    </div>
  );
}