import axios from 'axios';

const API_BASE_URL = 'http://localhost:8090/api';

export interface Asset {
  id: string;
  title: string;
  property_type: string;
  address: string;
  city: string;
  state: string;
  pincode: string;
  area: number;
  bedrooms?: number;
  bathrooms?: number;
  purchase_date?: string;
  purchase_price?: number;
  current_value?: number;
  occupancy_status: 'self_occupied' | 'rented' | 'vacant';
  is_asset: boolean;
  image?: string;
}

export interface AssetDocument {
  id: string;
  property_id: string;
  category: string;
  file_name: string;
  file_path: string;
  file_size: number;
  description?: string;
  uploaded_at: string;
}

export interface AssetTransaction {
  id: string;
  property_id: string;
  type: 'income' | 'expense';
  category: string;
  amount: number;
  transaction_date: string;
  description?: string;
  created_at: string;
}

export interface FinancialSummary {
  total_income: number;
  total_expense: number;
  net_profit: number;
}

class AssetAPI {
  // Get all asset properties
  async getAssets(): Promise<Asset[]> {
    const response = await axios.get(`${API_BASE_URL}/assets/`);
    return response.data;
  }

  // Get single asset by ID
  async getAssetById(id: string): Promise<Asset> {
    console.log('API: Fetching asset', id);
    const response = await axios.get(`${API_BASE_URL}/assets/${id}`);
    console.log('API: Asset response', response.data);
    return response.data;
  }

  // Get dashboard stats
  async getAssetStats() {
    const response = await axios.get(`${API_BASE_URL}/assets/stats`);
    return response.data;
  }

  // Update asset
  async updateAsset(id: string, data: Partial<Asset>): Promise<Asset> {
    const sellerToken = localStorage.getItem('sellerToken');
    console.log('Updating asset:', id, data);
    const response = await axios.patch(`${API_BASE_URL}/assets/${id}`, data, {
      headers: sellerToken ? { 'Authorization': `Bearer ${sellerToken}` } : {}
    });
    console.log('Update response:', response.data);
    return response.data;
  }

  // Delete asset
  async deleteAsset(id: string): Promise<void> {
    await axios.delete(`${API_BASE_URL}/assets/${id}`);
  }

  // Document APIs
  async uploadDocument(propertyId: string, file: File, category: string, description?: string): Promise<AssetDocument> {
    const formData = new FormData();
    formData.append('file', file);
    formData.append('category', category);
    if (description) formData.append('description', description);

    const response = await axios.post(`${API_BASE_URL}/assets/${propertyId}/documents`, formData, {
      headers: { 'Content-Type': 'multipart/form-data' }
    });
    return response.data;
  }

  async getDocuments(propertyId: string): Promise<AssetDocument[]> {
    console.log('API: Fetching documents for', propertyId);
    const response = await axios.get(`${API_BASE_URL}/assets/${propertyId}/documents`);
    console.log('API: Documents response', response.data);
    return response.data;
  }

  async deleteDocument(documentId: string): Promise<void> {
    await axios.delete(`${API_BASE_URL}/assets/documents/${documentId}`);
  }

  async downloadDocument(documentId: string): Promise<Blob> {
    const response = await axios.get(`${API_BASE_URL}/assets/documents/${documentId}/download`, {
      responseType: 'blob'
    });
    return response.data;
  }

  // Transaction APIs
  async addTransaction(propertyId: string, data: Omit<AssetTransaction, 'id' | 'property_id' | 'created_at'>): Promise<AssetTransaction> {
    const response = await axios.post(`${API_BASE_URL}/assets/${propertyId}/transactions`, data);
    return response.data;
  }

  async getTransactions(propertyId: string): Promise<AssetTransaction[]> {
    console.log('API: Fetching transactions for', propertyId);
    const response = await axios.get(`${API_BASE_URL}/assets/${propertyId}/transactions`);
    console.log('API: Transactions response', response.data);
    return response.data;
  }

  async deleteTransaction(transactionId: string): Promise<void> {
    await axios.delete(`${API_BASE_URL}/assets/transactions/${transactionId}`);
  }

  async getFinancialSummary(propertyId: string): Promise<FinancialSummary> {
    const response = await axios.get(`${API_BASE_URL}/assets/${propertyId}/financials`);
    return response.data;
  }
}

export const assetAPI = new AssetAPI();
