import { api } from '../api/client';

export interface AuditEvent {
  operation: string;
  resource_type: string;
  resource_id?: string;
  event_data?: Record<string, any>;
  additional_context?: Record<string, any>;
}

export interface AuditLogEntry {
  id: number;
  operation: string;
  resource_type: string;
  resource_id?: string;
  user_id?: number;
  session_id?: string;
  ip_address?: string;
  user_agent?: string;
  request_method: string;
  request_path: string;
  request_data?: Record<string, any>;
  response_status: number;
  response_data?: Record<string, any>;
  execution_time_ms: number;
  success: boolean;
  error_message?: string;
  additional_context?: Record<string, any>;
  created_at: string;
}

export interface AuditLogFilter {
  operation?: string;
  resource_type?: string;
  user_id?: number;
  success?: boolean;
  start_date?: string;
  end_date?: string;
  limit?: number;
  offset?: number;
}

export interface UserActivitySummary {
  user_id: number;
  period_days: number;
  total_operations: number;
  total_successful: number;
  total_failed: number;
  success_rate: number;
  operations: Array<{
    operation: string;
    count: number;
    successful: number;
    failed: number;
    avg_execution_time: number;
  }>;
}

class AuditService {
  private sessionId: string;
  private isEnabled: boolean = true;

  constructor() {
    this.sessionId = this.getOrCreateSessionId();
  }

  private getOrCreateSessionId(): string {
    let sessionId = sessionStorage.getItem('audit_session_id');
    if (!sessionId) {
      sessionId = crypto.randomUUID();
      sessionStorage.setItem('audit_session_id', sessionId);
    }
    return sessionId;
  }

  /**
   * Log a security-related event on the frontend
   */
  async logSecurityEvent(
    operation: string,
    resource_type: string,
    resource_id?: string,
    event_data?: Record<string, any>
  ): Promise<void> {
    if (!this.isEnabled) return;

    try {
      const auditEvent: AuditEvent = {
        operation,
        resource_type,
        resource_id,
        event_data: {
          ...event_data,
          session_id: this.sessionId,
          timestamp: new Date().toISOString(),
          user_agent: navigator.userAgent,
          page_url: window.location.href,
          referrer: document.referrer
        }
      };

      console.log('🔒 Security Event:', operation, auditEvent);
      this.storeLocalAuditEvent(auditEvent);

    } catch (error) {
      console.warn('❌ Failed to log security event:', error);
    }
  }

  /**
   * Log authentication events
   */
  async logAuthEvent(
    operation: 'LOGIN_ATTEMPT' | 'LOGIN_SUCCESS' | 'LOGIN_FAILURE' | 'LOGOUT' | 'SESSION_EXPIRED',
    event_data?: Record<string, any>
  ): Promise<void> {
    await this.logSecurityEvent(operation, 'AUTH', undefined, event_data);
  }

  /**
   * Log document access events
   */
  async logDocumentEvent(
    operation: 'DOCUMENT_VIEW' | 'DOCUMENT_DOWNLOAD' | 'DOCUMENT_UPLOAD_ATTEMPT',
    document_id: string,
    event_data?: Record<string, any>
  ): Promise<void> {
    await this.logSecurityEvent(operation, 'DOCUMENT', document_id, event_data);
  }

  /**
   * Log property access events
   */
  async logPropertyEvent(
    operation: 'PROPERTY_VIEW' | 'PROPERTY_INQUIRY' | 'PROPERTY_SAVE',
    property_id: string,
    event_data?: Record<string, any>
  ): Promise<void> {
    await this.logSecurityEvent(operation, 'PROPERTY', property_id, event_data);
  }

  /**
   * Log payment events
   */
  async logPaymentEvent(
    operation: 'PAYMENT_INITIATED' | 'PAYMENT_COMPLETED' | 'PAYMENT_FAILED',
    payment_id?: string,
    event_data?: Record<string, any>
  ): Promise<void> {
    await this.logSecurityEvent(operation, 'PAYMENT', payment_id, event_data);
  }

  /**
   * Log data export events
   */
  async logDataExportEvent(
    operation: 'DATA_EXPORT_REQUEST' | 'DATA_EXPORT_DOWNLOAD',
    export_type: string,
    event_data?: Record<string, any>
  ): Promise<void> {
    await this.logSecurityEvent(operation, 'DATA', export_type, event_data);
  }

  /**
   * Store audit event locally (for offline capability)
   */
  private storeLocalAuditEvent(event: AuditEvent): void {
    try {
      const stored = localStorage.getItem('pending_audit_events');
      const events = stored ? JSON.parse(stored) : [];
      events.push({
        ...event,
        timestamp: new Date().toISOString()
      });
      
      if (events.length > 100) {
        events.splice(0, events.length - 100);
      }
      
      localStorage.setItem('pending_audit_events', JSON.stringify(events));
    } catch (error) {
      console.warn('Failed to store audit event locally:', error);
    }
  }

  /**
   * Get audit logs (admin only)
   */
  async getAuditLogs(filters?: AuditLogFilter): Promise<AuditLogEntry[]> {
    try {
      const params = new URLSearchParams();
      
      if (filters) {
        Object.entries(filters).forEach(([key, value]) => {
          if (value !== undefined && value !== null) {
            params.append(key, value.toString());
          }
        });
      }

      const response = await api.get(`/audit-logs/logs?${params.toString()}`);
      return response.data;
    } catch (error) {
      console.error('Failed to fetch audit logs:', error);
      throw error;
    }
  }

  /**
   * Get user activity summary (admin only)
   */
  async getUserActivitySummary(userId: number, days: number = 30): Promise<UserActivitySummary> {
    try {
      const response = await api.get(`/audit-logs/user/${userId}/activity?days=${days}`);
      return response.data;
    } catch (error) {
      console.error('Failed to fetch user activity summary:', error);
      throw error;
    }
  }

  /**
   * Get current user's activity summary
   */
  async getMyActivitySummary(days: number = 30): Promise<UserActivitySummary> {
    try {
      const response = await api.get(`/audit-logs/my-activity?days=${days}`);
      return response.data;
    } catch (error) {
      console.error('Failed to fetch my activity summary:', error);
      throw error;
    }
  }

  /**
   * Clean up old audit logs (admin only)
   */
  async cleanupOldLogs(retentionDays: number = 90): Promise<{ deleted_count: number }> {
    try {
      const response = await api.post(`/audit-logs/cleanup?retention_days=${retentionDays}`);
      return response.data;
    } catch (error) {
      console.error('Failed to cleanup audit logs:', error);
      throw error;
    }
  }

  /**
   * Seed audit log data (admin only)
   */
  async seedAuditData(): Promise<any> {
    try {
      const response = await api.post('/audit-logs/seed-data');
      return response.data;
    } catch (error) {
      console.error('Failed to seed audit data:', error);
      throw error;
    }
  }

  /**
   * Clear all audit logs (admin only)
   */
  async clearAllAuditLogs(): Promise<any> {
    try {
      const response = await api.delete('/audit-logs/clear-all');
      return response.data;
    } catch (error) {
      console.error('Failed to clear audit logs:', error);
      throw error;
    }
  }

  /**
   * Get suspicious activities (admin only)
   */
  async getSuspiciousActivities(): Promise<any> {
    try {
      // Return mock data since this endpoint doesn't exist yet
      return {
        alerts: [
          {
            type: "Multiple Failed Login Attempts",
            description: "User attempted to login 5 times with wrong credentials",
            severity: "high",
            action: "Block User"
          },
          {
            type: "Unusual Access Pattern",
            description: "Access from new geographic location",
            severity: "medium", 
            action: "Verify Identity"
          }
        ]
      };
    } catch (error) {
      console.error('Failed to fetch suspicious activities:', error);
      throw error;
    }
  }

  /**
   * Get security status (admin only)
   */
  async getSecurityStatus(): Promise<any> {
    try {
      // Return mock data since this endpoint doesn't exist yet
      return {
        security_score: 95
      };
    } catch (error) {
      console.error('Failed to fetch security status:', error);
      throw error;
    }
  }

  /**
   * Get security settings (admin only)
   */
  async getSecuritySettings(): Promise<any> {
    try {
      // Return mock data since this endpoint doesn't exist yet
      return {
        two_factor_auth: { enabled: true },
        session_timeout: { minutes: 30 },
        ip_whitelist: { status: 'partial' }
      };
    } catch (error) {
      console.error('Failed to fetch security settings:', error);
      throw error;
    }
  }

  /**
   * Export audit logs to CSV (admin only)
   */
  async exportAuditLogsCSV(filters?: AuditLogFilter): Promise<void> {
    try {
      const params = new URLSearchParams();
      
      if (filters) {
        Object.entries(filters).forEach(([key, value]) => {
          if (value !== undefined && value !== null) {
            params.append(key, value.toString());
          }
        });
      }

      const response = await api.get(`/audit-logs/export?${params.toString()}`, {
        responseType: 'blob'
      });
      
      const blob = new Blob([response.data], {
        type: 'text/csv'
      });
      
      const url = window.URL.createObjectURL(blob);
      const link = document.createElement('a');
      link.href = url;
      
      const contentDisposition = response.headers['content-disposition'];
      let filename = 'audit_logs.csv';
      if (contentDisposition) {
        const filenameMatch = contentDisposition.match(/filename="?([^"]+)"?/);
        if (filenameMatch) {
          filename = filenameMatch[1];
        }
      }
      
      link.download = filename;
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      window.URL.revokeObjectURL(url);
      
      await this.logDataExportEvent('DATA_EXPORT_DOWNLOAD', 'audit_logs_csv', {
        filters,
        filename
      });
      
    } catch (error) {
      console.error('Failed to export audit logs to CSV:', error);
      throw error;
    }
  }

  /**
   * Export audit logs to Excel (admin only)
   */
  async exportAuditLogsExcel(filters?: AuditLogFilter): Promise<void> {
    try {
      const params = new URLSearchParams();
      
      if (filters) {
        Object.entries(filters).forEach(([key, value]) => {
          if (value !== undefined && value !== null) {
            params.append(key, value.toString());
          }
        });
      }

      const response = await api.get(`/audit-logs/export-excel?${params.toString()}`, {
        responseType: 'blob'
      });
      
      const blob = new Blob([response.data], {
        type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
      });
      
      const url = window.URL.createObjectURL(blob);
      const link = document.createElement('a');
      link.href = url;
      
      const contentDisposition = response.headers['content-disposition'];
      let filename = 'audit_logs.xlsx';
      if (contentDisposition) {
        const filenameMatch = contentDisposition.match(/filename="?([^"]+)"?/);
        if (filenameMatch) {
          filename = filenameMatch[1];
        }
      }
      
      link.download = filename;
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      window.URL.revokeObjectURL(url);
      
      await this.logDataExportEvent('DATA_EXPORT_DOWNLOAD', 'audit_logs_excel', {
        filters,
        filename
      });
      
    } catch (error) {
      console.error('Failed to export audit logs to Excel:', error);
      throw error;
    }
  }

  enable(): void {
    this.isEnabled = true;
  }

  disable(): void {
    this.isEnabled = false;
  }

  isAuditEnabled(): boolean {
    return this.isEnabled;
  }
}

export const auditService = new AuditService();