import axios from 'axios';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8090/api';

export interface EventData {
  event_type: string;
  data: any;
  user_id?: string;
  target_users?: string[];
}

export interface WebSocketStatus {
  connected_users_count: number;
  connected_users: string[];
}

class EventBusService {
  private baseURL: string;

  constructor(baseURL: string = API_BASE_URL) {
    this.baseURL = baseURL;
  }

  /**
   * Publish an event to the event bus
   */
  async publishEvent(eventData: EventData): Promise<{ status: string; message: string }> {
    try {
      const response = await axios.post(`${this.baseURL}/websocket/events/publish`, eventData);
      return response.data;
    } catch (error) {
      console.error('Error publishing event:', error);
      throw error;
    }
  }

  /**
   * Get WebSocket connection status
   */
  async getWebSocketStatus(): Promise<WebSocketStatus> {
    try {
      const response = await axios.get(`${this.baseURL}/websocket/status`);
      return response.data;
    } catch (error) {
      console.error('Error getting WebSocket status:', error);
      throw error;
    }
  }

  /**
   * Send a direct message to a specific user
   */
  async sendMessageToUser(userId: string, message: any): Promise<{ status: string; message: string }> {
    try {
      const response = await axios.post(`${this.baseURL}/websocket/send`, {
        user_id: userId,
        message
      });
      return response.data;
    } catch (error) {
      console.error('Error sending message to user:', error);
      throw error;
    }
  }

  /**
   * Test event for debugging
   */
  async sendTestEvent(userId: string, message: string = 'Test notification'): Promise<any> {
    return this.publishEvent({
      event_type: 'notification_created',
      data: {
        title: 'Test Notification',
        message,
        type: 'test'
      },
      user_id: userId
    });
  }

  /**
   * Publish property status change event
   */
  async publishPropertyStatusEvent(
    userId: string, 
    propertyId: string, 
    propertyTitle: string, 
    status: 'approved' | 'rejected',
    reason?: string
  ): Promise<any> {
    const eventType = status === 'approved' ? 'property_approved' : 'property_rejected';
    
    return this.publishEvent({
      event_type: eventType,
      data: {
        property_id: propertyId,
        property_title: propertyTitle,
        status,
        reason
      },
      user_id: userId
    });
  }

  /**
   * Publish visit status change event
   */
  async publishVisitStatusEvent(
    userId: string,
    visitId: string,
    propertyTitle: string,
    status: 'approved' | 'rejected',
    reason?: string
  ): Promise<any> {
    const eventType = status === 'approved' ? 'visit_approved' : 'visit_rejected';
    
    return this.publishEvent({
      event_type: eventType,
      data: {
        visit_id: visitId,
        property_title: propertyTitle,
        status,
        reason
      },
      user_id: userId
    });
  }

  /**
   * Publish auction bid event
   */
  async publishAuctionBidEvent(
    propertyId: string,
    propertyTitle: string,
    bidAmount: number,
    bidderName: string,
    targetUsers: string[]
  ): Promise<any> {
    return this.publishEvent({
      event_type: 'auction_bid',
      data: {
        property_id: propertyId,
        property_title: propertyTitle,
        amount: bidAmount,
        bidder_name: bidderName
      },
      target_users: targetUsers
    });
  }

  /**
   * Publish auction winner event
   */
  async publishAuctionWinnerEvent(
    userId: string,
    propertyId: string,
    propertyTitle: string,
    winningAmount: number
  ): Promise<any> {
    return this.publishEvent({
      event_type: 'auction_won',
      data: {
        property_id: propertyId,
        property_title: propertyTitle,
        winning_amount: winningAmount
      },
      user_id: userId
    });
  }

  /**
   * Publish inquiry received event
   */
  async publishInquiryEvent(
    userId: string,
    inquiryId: string,
    propertyTitle: string,
    buyerName: string
  ): Promise<any> {
    return this.publishEvent({
      event_type: 'inquiry_received',
      data: {
        inquiry_id: inquiryId,
        property_title: propertyTitle,
        buyer_name: buyerName
      },
      user_id: userId
    });
  }

  /**
   * Publish document verification event
   */
  async publishDocumentVerificationEvent(
    userId: string,
    documentId: string,
    documentType: string,
    status: 'verified' | 'rejected'
  ): Promise<any> {
    return this.publishEvent({
      event_type: 'document_verified',
      data: {
        document_id: documentId,
        document_type: documentType,
        status
      },
      user_id: userId
    });
  }
}

// Create and export a singleton instance
export const eventBusService = new EventBusService();
export default eventBusService;