// Google Earth Engine Service for Flood Risk Analysis

import { FloodRiskData, GEEFloodResponse, CachedFloodData, UrbanGrowthData, InfrastructureData } from '../types/gee';
import { formatFloodData, getFloodCacheKey, isCacheValid } from '../utils/floodRiskCalculator';
import { formatUrbanGrowthData, getGrowthCacheKey } from '../utils/urbanGrowthCalculator';
import { 
  calculateAccessibilityScore, 
  calculateTransportScore, 
  calculateAmenitiesScore, 
  calculateEmploymentScore,
  classifyConnectivityLevel,
  formatInfrastructureData 
} from '../utils/infrastructureCalculator';

class GoogleEarthEngineService {
  private cache: CachedFloodData = {};
  private apiKey: string;
  private baseUrl: string;

  constructor() {
    this.apiKey = import.meta.env.VITE_GEE_API_KEY || '';
    this.baseUrl = 'https://earthengine.googleapis.com/v1';
    this.loadCacheFromStorage();
  }

  /**
   * Get flood risk data for a location
   * Uses cache first, then GEE API, falls back to mock data
   */
  async getFloodRisk(lat: number, lng: number): Promise<FloodRiskData> {
    const cacheKey = getFloodCacheKey(lat, lng);
    
    // Check cache first
    const cached = this.cache[cacheKey];
    if (cached && isCacheValid(cached.timestamp)) {
      console.log('✅ Using cached flood data for', lat, lng);
      return cached.data;
    }

    try {
      // Try GEE API
      const geeData = await this.fetchFromGEE(lat, lng);
      
      if (geeData.success && geeData.data) {
        const floodData = formatFloodData(
          geeData.data.elevation,
          geeData.data.waterDistance,
          geeData.data.floodHistory,
          geeData.data.waterOccurrence
        );
        
        this.cacheData(cacheKey, floodData);
        return floodData;
      }
    } catch (error) {
      console.warn('⚠️ GEE API failed, using mock data:', error);
    }

    // Fallback to mock data
    return this.getMockFloodData(lat, lng);
  }

  /**
   * Fetch data from Google Earth Engine API
   * NOTE: This requires GEE account and authentication
   */
  private async fetchFromGEE(lat: number, lng: number): Promise<GEEFloodResponse> {
    // TODO: Implement actual GEE API call when credentials are ready
    // For now, return mock data structure
    
    if (!this.apiKey) {
      throw new Error('GEE API key not configured');
    }

    // Simulated API call structure (replace with actual implementation)
    /*
    const response = await fetch(`${this.baseUrl}/projects/YOUR_PROJECT/assets:compute`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${this.apiKey}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        expression: {
          // GEE Earth Engine expression for flood analysis
          functionName: 'Image.sample',
          arguments: {
            geometry: { type: 'Point', coordinates: [lng, lat] },
            scale: 30
          }
        }
      })
    });
    
    const data = await response.json();
    return { success: true, data: this.parseGEEResponse(data) };
    */

    throw new Error('GEE API not implemented yet');
  }

  /**
   * Generate realistic mock flood data based on location
   * Uses heuristics to simulate real data patterns
   */
  private getMockFloodData(lat: number, lng: number): FloodRiskData {
    // Simulate elevation based on latitude (rough approximation)
    const baseElevation = Math.abs(lat - 20) * 10 + Math.random() * 50;
    
    // Simulate water proximity (coastal areas closer to water)
    const isCoastal = Math.abs(lng - 72.8) < 5 || Math.abs(lng - 80.2) < 5;
    const waterDistance = isCoastal 
      ? Math.random() * 2000 + 500 
      : Math.random() * 5000 + 2000;
    
    // Simulate historical floods (coastal and low elevation = more floods)
    const historicalFloods = isCoastal && baseElevation < 20 
      ? Math.floor(Math.random() * 4) + 1 
      : Math.floor(Math.random() * 2);
    
    // Simulate water occurrence (based on proximity)
    const waterOccurrence = waterDistance < 1000 
      ? Math.random() * 30 + 10 
      : Math.random() * 10;

    return formatFloodData(
      baseElevation,
      waterDistance,
      historicalFloods,
      waterOccurrence
    );
  }

  /**
   * Batch fetch flood data for multiple properties
   */
  async batchGetFloodRisk(locations: Array<{ lat: number; lng: number }>): Promise<FloodRiskData[]> {
    const batchSize = 10;
    const results: FloodRiskData[] = [];

    for (let i = 0; i < locations.length; i += batchSize) {
      const batch = locations.slice(i, i + batchSize);
      const batchResults = await Promise.all(
        batch.map(loc => this.getFloodRisk(loc.lat, loc.lng))
      );
      results.push(...batchResults);
      
      // Small delay to avoid rate limiting
      if (i + batchSize < locations.length) {
        await new Promise(resolve => setTimeout(resolve, 100));
      }
    }

    return results;
  }

  /**
   * Cache flood data
   */
  private cacheData(key: string, data: FloodRiskData): void {
    this.cache[key] = {
      data,
      timestamp: Date.now()
    };
    this.saveCacheToStorage();
  }

  /**
   * Load cache from localStorage
   */
  private loadCacheFromStorage(): void {
    try {
      const stored = localStorage.getItem('gee_flood_cache');
      if (stored) {
        this.cache = JSON.parse(stored);
      }
    } catch (error) {
      console.warn('Failed to load flood cache:', error);
    }
  }

  /**
   * Save cache to localStorage
   */
  private saveCacheToStorage(): void {
    try {
      localStorage.setItem('gee_flood_cache', JSON.stringify(this.cache));
    } catch (error) {
      console.warn('Failed to save flood cache:', error);
    }
  }

  /**
   * Clear cache
   */
  clearCache(): void {
    this.cache = {};
    localStorage.removeItem('gee_flood_cache');
  }

  /**
   * Get urban growth data for a location
   */
  async getUrbanGrowth(lat: number, lng: number): Promise<UrbanGrowthData> {
    const cacheKey = getGrowthCacheKey(lat, lng);
    
    // Check cache
    const cached = this.cache[cacheKey];
    if (cached && isCacheValid(cached.timestamp)) {
      console.log('✅ Using cached growth data for', lat, lng);
      return cached.data as UrbanGrowthData;
    }

    try {
      // Try GEE API (not implemented yet)
      throw new Error('GEE API not implemented');
    } catch (error) {
      console.warn('⚠️ GEE API failed, using mock data:', error);
    }

    // Fallback to mock data
    return this.getMockUrbanGrowthData(lat, lng);
  }

  /**
   * Generate realistic mock urban growth data
   */
  private getMockUrbanGrowthData(lat: number, lng: number): UrbanGrowthData {
    // Simulate growth based on location
    const cityHash = Math.abs(Math.floor(lat * lng * 1000)) % 100;
    
    // Growth rate: 2-12% annually
    const growthRate = 2 + (cityHash % 10);
    
    // Infrastructure score: 3-9
    const infrastructureScore = 3 + (cityHash % 7);
    
    // Built-up density: 20-80%
    const builtUpDensity = 20 + (cityHash % 60);
    
    // Generate timeline data (2015-2024)
    const timelineData = [];
    let baseArea = 50 + (cityHash % 50); // Base urban area in sq km
    
    for (let year = 2015; year <= 2024; year++) {
      const yearGrowth = 1 + (growthRate / 100);
      baseArea *= yearGrowth;
      const yearDensity = builtUpDensity * (year - 2014) / 10;
      
      timelineData.push({
        year,
        urbanArea: Math.round(baseArea * 10) / 10,
        builtUpDensity: Math.min(90, Math.round(yearDensity))
      });
    }
    
    const growthData = formatUrbanGrowthData(
      growthRate,
      infrastructureScore,
      builtUpDensity,
      timelineData
    );
    
    // Cache the result
    this.cache[cacheKey] = {
      data: growthData,
      timestamp: Date.now()
    };
    this.saveCacheToStorage();
    
    return growthData;
  }

  /**
   * Batch fetch urban growth data
   */
  async batchGetUrbanGrowth(locations: Array<{ lat: number; lng: number }>): Promise<UrbanGrowthData[]> {
    const batchSize = 10;
    const results: UrbanGrowthData[] = [];

    for (let i = 0; i < locations.length; i += batchSize) {
      const batch = locations.slice(i, i + batchSize);
      const batchResults = await Promise.all(
        batch.map(loc => this.getUrbanGrowth(loc.lat, loc.lng))
      );
      results.push(...batchResults);
      
      if (i + batchSize < locations.length) {
        await new Promise(resolve => setTimeout(resolve, 100));
      }
    }

    return results;
  }

  /**
   * Get cache statistics
   */
  getCacheStats(): { size: number; oldestEntry: number | null } {
    const entries = Object.values(this.cache);
    return {
      size: entries.length,
      oldestEntry: entries.length > 0 
        ? Math.min(...entries.map(e => e.timestamp)) 
        : null
    };
  }

  /**
   * Get infrastructure & accessibility data for a location
   */
  async getInfrastructure(lat: number, lng: number, city: string): Promise<InfrastructureData> {
    const cacheKey = `infra_${lat.toFixed(4)}_${lng.toFixed(4)}`;
    
    // Check cache
    const cached = this.cache[cacheKey];
    if (cached && isCacheValid(cached.timestamp)) {
      console.log('✅ Using cached infrastructure data for', lat, lng);
      return cached.data as InfrastructureData;
    }

    try {
      // Try GEE API (not implemented yet)
      throw new Error('GEE API not implemented');
    } catch (error) {
      console.warn('⚠️ GEE API failed, using mock data:', error);
    }

    // Fallback to mock data
    return this.getMockInfrastructureData(lat, lng, city);
  }

  /**
   * Generate realistic mock infrastructure data
   */
  private getMockInfrastructureData(lat: number, lng: number, city: string): InfrastructureData {
    // City-specific infrastructure
    const cityData: Record<string, any> = {
      'Mumbai': { hasMetro: true, airports: ['Chhatrapati Shivaji'], itParks: ['BKC', 'Andheri SEEPZ'] },
      'Bangalore': { hasMetro: true, airports: ['Kempegowda'], itParks: ['Electronic City', 'Whitefield'] },
      'Delhi': { hasMetro: true, airports: ['Indira Gandhi'], itParks: ['Cyber City', 'Noida'] },
      'Hyderabad': { hasMetro: true, airports: ['Rajiv Gandhi'], itParks: ['HITEC City', 'Gachibowli'] },
      'Pune': { hasMetro: true, airports: ['Pune Airport'], itParks: ['Hinjewadi', 'Magarpatta'] },
      'Chennai': { hasMetro: true, airports: ['Chennai Airport'], itParks: ['OMR', 'Guindy'] },
      'Kolkata': { hasMetro: true, airports: ['Netaji Subhas'], itParks: ['Salt Lake', 'Rajarhat'] }
    };

    const cityInfo = cityData[city] || { hasMetro: false, airports: ['City Airport'], itParks: null };
    const locationHash = Math.abs(Math.floor(lat * lng * 1000)) % 100;

    // Generate distances (km)
    const highwayDist = 0.5 + (locationHash % 10) * 0.3;
    const metroDist = cityInfo.hasMetro ? 0.3 + (locationHash % 15) * 0.2 : null;
    const airportDist = 5 + (locationHash % 30);
    const itParkDist = cityInfo.itParks ? 1 + (locationHash % 20) * 0.5 : null;
    const hospitalDist = 0.5 + (locationHash % 8) * 0.3;
    const schoolDist = 0.2 + (locationHash % 5) * 0.2;

    // Calculate scores
    const accessibilityScore = calculateAccessibilityScore(
      highwayDist, metroDist, airportDist, itParkDist, hospitalDist, schoolDist
    );
    const transportScore = calculateTransportScore(highwayDist, metroDist, airportDist);
    const amenitiesScore = calculateAmenitiesScore(hospitalDist, schoolDist);
    const employmentScore = calculateEmploymentScore(itParkDist, highwayDist);
    const connectivityLevel = classifyConnectivityLevel(accessibilityScore);

    const infraData: InfrastructureData = {
      accessibilityScore,
      connectivityLevel,
      nearestHighway: {
        name: `NH-${Math.floor(locationHash % 50) + 1}`,
        distance: highwayDist
      },
      nearestMetro: metroDist ? {
        name: `Metro Station ${String.fromCharCode(65 + (locationHash % 26))}`,
        distance: metroDist
      } : null,
      nearestAirport: {
        name: cityInfo.airports[0],
        distance: airportDist
      },
      nearestITPark: itParkDist ? {
        name: cityInfo.itParks[locationHash % cityInfo.itParks.length],
        distance: itParkDist
      } : null,
      nearestHospital: {
        name: `${['Apollo', 'Fortis', 'Max', 'Manipal'][locationHash % 4]} Hospital`,
        distance: hospitalDist
      },
      nearestSchool: {
        name: `${['DPS', 'Ryan', 'Kendriya Vidyalaya', 'DAV'][locationHash % 4]} School`,
        distance: schoolDist
      },
      transportScore,
      amenitiesScore,
      employmentScore,
      lastUpdated: new Date().toISOString()
    };

    const formattedData = formatInfrastructureData(infraData);

    // Cache the result
    const cacheKey = `infra_${lat.toFixed(4)}_${lng.toFixed(4)}`;
    this.cache[cacheKey] = {
      data: formattedData,
      timestamp: Date.now()
    };
    this.saveCacheToStorage();

    return formattedData;
  }

  /**
   * Batch fetch infrastructure data
   */
  async batchGetInfrastructure(
    locations: Array<{ lat: number; lng: number; city: string }>
  ): Promise<InfrastructureData[]> {
    const batchSize = 10;
    const results: InfrastructureData[] = [];

    for (let i = 0; i < locations.length; i += batchSize) {
      const batch = locations.slice(i, i + batchSize);
      const batchResults = await Promise.all(
        batch.map(loc => this.getInfrastructure(loc.lat, loc.lng, loc.city))
      );
      results.push(...batchResults);
      
      if (i + batchSize < locations.length) {
        await new Promise(resolve => setTimeout(resolve, 100));
      }
    }

    return results;
  }
}

// Export singleton instance
export const geeService = new GoogleEarthEngineService();
export default geeService;
