// Google Maps API Service
export interface PlaceDetails {
  name: string;
  formatted_address: string;
  geometry: {
    location: {
      lat: number;
      lng: number;
    };
  };
  place_id: string;
  types: string[];
}

export interface NearbyPlace {
  name: string;
  vicinity: string;
  geometry: {
    location: {
      lat: number;
      lng: number;
    };
  };
  rating?: number;
  types: string[];
  distance?: number;
}

class GoogleMapsService {
  private apiKey: string;
  private geocodeCache = new Map<string, PlaceDetails>();
  private placesCache = new Map<string, NearbyPlace[]>();

  constructor() {
    this.apiKey = import.meta.env.VITE_GOOGLE_MAPS_API_KEY || '';
    if (!this.apiKey) {
      console.warn('⚠️ Google Maps API key not found');
    }
  }

  async geocodeLocation(address: string): Promise<PlaceDetails | null> {
    if (this.geocodeCache.has(address)) {
      return this.geocodeCache.get(address)!;
    }

    if (!this.apiKey) {
      console.warn('Google Maps API key not configured');
      return null;
    }

    try {
      const response = await fetch(
        `https://maps.googleapis.com/maps/api/geocode/json?address=${encodeURIComponent(address)}&key=${this.apiKey}`
      );

      if (!response.ok) {
        throw new Error('Geocoding failed');
      }

      const data = await response.json();

      if (data.results && data.results.length > 0) {
        const result = data.results[0];
        const placeDetails: PlaceDetails = {
          name: result.formatted_address.split(',')[0],
          formatted_address: result.formatted_address,
          geometry: {
            location: {
              lat: result.geometry.location.lat,
              lng: result.geometry.location.lng
            }
          },
          place_id: result.place_id,
          types: result.types
        };

        this.geocodeCache.set(address, placeDetails);
        return placeDetails;
      }

      return null;
    } catch (error) {
      console.error('Geocoding error:', error);
      return null;
    }
  }

  async getNearbyPlaces(
    lat: number,
    lng: number,
    type: string,
    radius: number = 2000
  ): Promise<NearbyPlace[]> {
    const cacheKey = `${lat},${lng},${type},${radius}`;
    
    if (this.placesCache.has(cacheKey)) {
      return this.placesCache.get(cacheKey)!;
    }

    if (!this.apiKey) {
      return [];
    }

    try {
      const response = await fetch(
        `https://maps.googleapis.com/maps/api/place/nearbysearch/json?location=${lat},${lng}&radius=${radius}&type=${type}&key=${this.apiKey}`
      );

      if (!response.ok) {
        throw new Error('Places API failed');
      }

      const data = await response.json();

      if (data.results && data.results.length > 0) {
        const places: NearbyPlace[] = data.results.map((place: any) => {
          const distance = this.calculateDistance(
            lat,
            lng,
            place.geometry.location.lat,
            place.geometry.location.lng
          );

          return {
            name: place.name,
            vicinity: place.vicinity,
            geometry: {
              location: {
                lat: place.geometry.location.lat,
                lng: place.geometry.location.lng
              }
            },
            rating: place.rating,
            types: place.types,
            distance
          };
        });

        this.placesCache.set(cacheKey, places);
        return places;
      }

      return [];
    } catch (error) {
      console.error('Places API error:', error);
      return [];
    }
  }

  private calculateDistance(lat1: number, lng1: number, lat2: number, lng2: number): number {
    const R = 6371;
    const dLat = (lat2 - lat1) * Math.PI / 180;
    const dLng = (lng2 - lng1) * Math.PI / 180;
    const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
      Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
      Math.sin(dLng/2) * Math.sin(dLng/2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
    return R * c;
  }

  clearCache(): void {
    this.geocodeCache.clear();
    this.placesCache.clear();
  }
}

export const googleMapsService = new GoogleMapsService();
