// Groq AI Service for Real Estate Intelligence
import { localityCache, cacheUtils } from '../utils/localityCache';
import { validateLocalityData, getRealisticPriceRange, getRealisticGrowth, getRealisticRentalYield, getCityTier, getApproximateCoordinates } from '../utils/realEstateKnowledge';
import { buildLocalityPrompt, buildSentimentPrompt, buildHeatmapPrompt, buildComparisonPrompt } from '../utils/promptBuilder';

interface GroqResponse {
  choices: Array<{
    message: {
      content: string;
    };
  }>;
}

interface LocalityData {
  name: string;
  center: { lat: number; lng: number };
  pricePerSqft: number;
  oneYearGrowth: number;
  fiveYearGrowth: number;
  rentalYield: number;
  avgRent: number;
  ribl: number;
  demand: string;
  overallScore: number;
  safetyIndex: number;
  connectivityScore: number;
  amenitiesScore: number;
  environmentScore: number;
  growthPotential: number;
  heatmapPoints: Array<{ lat: number; lng: number; intensity: number }>;
  priceTrend: Array<{ year: string; price: number }>;
  amenities: Array<{
    type: string;
    name: string;
    distKm: number;
    rating: number;
    travelTime: string;
  }>;
  safety: {
    crimeRatePerMonth: number;
    policeStationDistKm: number;
    nightSafety: number;
    streetLighting: string;
    cctvCoverage: string;
    emergencyResponse: string;
    crimeStats: string;
  };
  connectivity: {
    metroDistance: number;
    busStops: number;
    trafficScore: number;
    roadQuality: string;
    airportDistance: number;
    railwayDistance: number;
    peakTrafficTime: string;
  };
  lifestyle: {
    malls: string[];
    restaurants: string[];
    entertainment: string[];
    parks: string[];
  };
  environment: {
    aqi: number;
    noiseDb: number;
    greenCoverPct: number;
    waterQuality: string;
    airQualityTrend: string;
    pollutionSources: string;
  };
  upcomingDevelopments: Array<{
    project: string;
    timeline: string;
    impact: string;
  }>;
  demographics: {
    avgIncomeLakhs: number;
    renterRatioPct: number;
    ageMedian: number;
  };
}

class GroqService {
  private apiKey: string;
  private baseUrl = 'https://api.groq.com/openai/v1/chat/completions';
  private requestCount = 0;
  private errorCount = 0;

  constructor() {
    this.apiKey = import.meta.env.VITE_GROQ_API_KEY;
    if (!this.apiKey) {
      console.warn('⚠️ Groq API key not found in environment variables');
      console.warn('📝 Please add VITE_GROQ_API_KEY to your .env file');
      console.warn('🔄 Will use emergency fallback data');
    } else {
      console.log('✅ GROQ API key loaded successfully');
    }
  }

  private async makeRequest(prompt: string, seed?: number): Promise<any> {
    if (!this.apiKey) {
      throw new Error('GROQ API key not configured');
    }

    try {
      console.log('🔄 Making GROQ API request...');
      const response = await fetch(this.baseUrl, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${this.apiKey}`,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          model: 'llama-3.3-70b-versatile',
          messages: [
            {
              role: 'system',
              content: `You are an expert Indian real estate market analyst with deep knowledge of property markets across India. 

KEY GUIDELINES:
- Provide REALISTIC data based on actual Indian market conditions
- Use proper Indian city/locality names and geography
- Price ranges: Tier-1 cities (₹8K-25K/sqft), Tier-2 (₹4K-12K/sqft), Tier-3 (₹2K-6K/sqft)
- Growth rates: Realistic 5-15% annually for good areas
- RIBL scores: 6-9 range (be conservative)
- Always return ONLY valid JSON, no explanations
- Be consistent for same location (use seed for determinism)`
            },
            {
              role: 'user',
              content: prompt
            }
          ],
          temperature: 0.2,
          max_tokens: 3000,
          top_p: 0.9,
          seed: seed || this.generateSeed(prompt)
        })
      });

      if (!response.ok) {
        const errorText = await response.text();
        console.error('GROQ API error response:', errorText);
        throw new Error(`Groq API error: ${response.status}`);
      }

      const data: GroqResponse = await response.json();
      const content = data.choices[0]?.message?.content;
      
      if (!content) {
        throw new Error('No content in Groq response');
      }

      const jsonMatch = content.match(/\{[\s\S]*\}/);
      if (jsonMatch) {
        return JSON.parse(jsonMatch[0]);
      }
      
      throw new Error('No valid JSON found in response');
    } catch (error) {
      console.error('❌ Groq API error:', error);
      throw error;
    }
  }

  private generateSeed(text: string): number {
    let hash = 0;
    for (let i = 0; i < text.length; i++) {
      const char = text.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash;
    }
    return Math.abs(hash);
  }

  getApiStats(): { requests: number; errors: number; errorRate: number } {
    const errorRate = this.requestCount > 0 ? (this.errorCount / this.requestCount) * 100 : 0;
    return {
      requests: this.requestCount,
      errors: this.errorCount,
      errorRate: Number(errorRate.toFixed(2))
    };
  }

  async getLocalityData(location: string, reportType: string = 'comprehensive'): Promise<LocalityData> {
    const cachedData = cacheUtils.getLocality(location, reportType);
    if (cachedData) {
      return cachedData;
    }
    
    console.log(`🌐 Fetching fresh data for: ${location}`);
    this.requestCount++;
    const locationSeed = this.generateSeed(location.toLowerCase());
    
    // Use smart prompt builder
    const prompt = buildLocalityPrompt(location, reportType);



    try {
      const data = await this.makeRequest(prompt, locationSeed);
      
      // Validate and correct unrealistic data
      const validatedData = validateLocalityData(data, location);
      
      // Ensure coordinates are set
      if (!validatedData.center || !validatedData.center.lat) {
        validatedData.center = getApproximateCoordinates(location);
      }
      
      cacheUtils.setLocality(location, validatedData, reportType);
      return validatedData;
    } catch (error) {
      this.errorCount++;
      console.error('❌ Error fetching Groq locality data:', error);
      const emergencyData = this.generateEmergencyData(location);
      cacheUtils.setLocality(location, emergencyData, reportType);
      return emergencyData;
    }
  }

  async getLiveStats(): Promise<any> {
    const cachedData = cacheUtils.getLiveStats();
    if (cachedData) {
      return cachedData;
    }
    
    this.requestCount++;
    const prompt = `Generate current live real estate statistics for Indian market in JSON format with fields: urgentSales (thisMonth, avgDiscount, quickSaleTime), agentActivity (activeAgents, listingsAdded, siteVisits), documents (totalFiles, verifiedDocs, recentUploads), marketTrends (priceMovement, demandLevel, inventoryDays).`;

    try {
      const data = await this.makeRequest(prompt);
      cacheUtils.setLiveStats(data);
      return data;
    } catch (error) {
      this.errorCount++;
      const emergencyStats = {
        urgentSales: { thisMonth: 12, avgDiscount: 15, quickSaleTime: "7-14 days" },
        agentActivity: { activeAgents: 25, listingsAdded: 8, siteVisits: 150 },
        documents: { totalFiles: 1500, verifiedDocs: 92, recentUploads: 25 }
      };
      cacheUtils.setLiveStats(emergencyStats);
      return emergencyStats;
    }
  }

  async getSentimentAnalysis(location: string): Promise<any> {
    const cachedData = cacheUtils.getSentiment(location);
    if (cachedData) {
      return cachedData;
    }
    
    this.requestCount++;
    const locationSeed = this.generateSeed(location.toLowerCase());
    
    // Use smart prompt builder
    const prompt = buildSentimentPrompt(location);

    try {
      const data = await this.makeRequest(prompt, locationSeed);
      cacheUtils.setSentiment(location, data);
      return data;
    } catch (error) {
      this.errorCount++;
      const seed = this.generateSeed(location.toLowerCase());
      const score = 60 + (seed % 30);
      const emergencyData = {
        sentimentScore: score,
        timeSeries: [
          { month: "Jan", score: score - 5 },
          { month: "Feb", score: score - 2 },
          { month: "Mar", score: score + 3 },
          { month: "Apr", score: score }
        ],
        walkabilityScore: 65 + (seed % 25),
        walkabilityBreakdown: { pedestrianInfra: 70, sidewalkQuality: 65, crossings: 75, lighting: 68 },
        floodRisk: { riskLevel: "Low", floodZone: "Zone A", lastFloodYear: "N/A" }
      };
      cacheUtils.setSentiment(location, emergencyData);
      return emergencyData;
    }
  }

  async getCompetitiveAnalysis(locations: string[]): Promise<any> {
    const cacheKey = `competitive_${locations.sort().join('_').toLowerCase().replace(/[^a-z0-9_]/g, '_')}`;
    const cachedData = localityCache.get(cacheKey);
    if (cachedData) {
      return cachedData;
    }
    
    this.requestCount++;
    
    // Use smart prompt builder
    const prompt = buildComparisonPrompt(locations);

    try {
      const data = await this.makeRequest(prompt);
      localityCache.set(cacheKey, data);
      return data;
    } catch (error) {
      this.errorCount++;
      throw error;
    }
  }

  async getMarketTrends(location: string): Promise<any> {
    const cachedData = cacheUtils.getMarketTrends(location);
    if (cachedData) {
      return cachedData;
    }
    
    this.requestCount++;
    const prompt = `Analyze market trends for "${location}" in JSON format with fields: priceMovement (direction, percentage, timeframe), demandSupply (demandLevel, supplyLevel, absorptionRate, inventoryMonths), buyerProfile (primaryBuyers, ageGroup, budgetRange, preferredPropertyType), futureOutlook (nextQuarter, nextYear, keyDrivers).`;

    try {
      const data = await this.makeRequest(prompt);
      cacheUtils.setMarketTrends(location, data);
      return data;
    } catch (error) {
      this.errorCount++;
      throw error;
    }
  }

  async getLocalitySuggestions(query: string): Promise<string[]> {
    if (!query || query.length < 2) return [];
    
    this.requestCount++;
    const prompt = `Suggest 8 real Indian localities matching "${query}" in JSON array format: ["Area Name, City", ...]. Focus on major cities, IT hubs, residential areas.`;

    try {
      const data = await this.makeRequest(prompt);
      if (Array.isArray(data)) {
        return data.slice(0, 8);
      }
      return [];
    } catch (error) {
      this.errorCount++;
      return [];
    }
  }

  async getNearbyLocalities(location: string): Promise<LocalityData[]> {
    return this.generateEmergencyNearbyLocalities(location, this.generateSeed(location.toLowerCase()));
  }

  async getHeatmapData(location: string, center: { lat: number; lng: number }): Promise<any> {
    const cacheKey = `heatmap_${location.toLowerCase().replace(/[^a-z0-9]/g, '_')}`;
    const cachedData = localityCache.get(cacheKey);
    if (cachedData) {
      return cachedData;
    }
    
    this.requestCount++;
    const locationSeed = this.generateSeed(location.toLowerCase());
    
    // Use smart prompt builder
    const prompt = buildHeatmapPrompt(location, center);

    try {
      const data = await this.makeRequest(prompt, locationSeed);
      localityCache.set(cacheKey, data);
      return data;
    } catch (error) {
      this.errorCount++;
      console.warn('⚠️ Using generated heatmap data');
      const generatedData = this.generateHeatmapLayers(center);
      localityCache.set(cacheKey, generatedData);
      return generatedData;
    }
  }

  private generateHeatmapLayers(center: { lat: number; lng: number }) {
    const generateLayer = (type: string, count: number = 60) => {
      const data = [];
      const radius = 0.02;
      
      for (let i = 0; i < count; i++) {
        const angle = Math.random() * 2 * Math.PI;
        const distance = Math.random() * radius;
        const lat = center.lat + distance * Math.cos(angle);
        const lng = center.lng + distance * Math.sin(angle);
        
        let value = Math.random();
        switch(type) {
          case 'price':
            value = 0.3 + value * 0.7;
            break;
          case 'demand':
            value = 0.4 + value * 0.6;
            break;
          case 'rental':
            value = 0.2 + value * 0.8;
            break;
          case 'crime':
            value = value * 0.4;
            break;
          case 'flood':
            value = value * 0.3;
            break;
        }
        
        data.push({ lat, lng, value });
      }
      
      return data;
    };

    return {
      price: generateLayer('price'),
      demand: generateLayer('demand'),
      rental: generateLayer('rental'),
      crime: generateLayer('crime'),
      flood: generateLayer('flood')
    };
  }

  private generateEmergencyNearbyLocalities(location: string, seed: number): LocalityData[] {
    const baseNames = ['North', 'South', 'East'];
    const cityMatch = location.match(/,\s*([^,]+)$/); 
    const city = cityMatch ? cityMatch[1] : 'Bangalore';
    
    return baseNames.map((dir, i) => {
      const localSeed = seed + i * 1000;
      const random = (min: number, max: number) => {
        const x = Math.sin(localSeed + min) * 10000;
        return Math.floor((x - Math.floor(x)) * (max - min + 1)) + min;
      };
      
      return {
        name: `${dir} ${city}`,
        center: { lat: 12.9716 + (random(-50, 50) / 1000), lng: 77.5946 + (random(-50, 50) / 1000) },
        pricePerSqft: random(7000, 12000),
        oneYearGrowth: random(6, 12),
        fiveYearGrowth: random(35, 70),
        rentalYield: random(28, 38) / 10,
        avgRent: random(18, 35),
        ribl: random(65, 85) / 10,
        demand: ['High', 'Very High', 'Medium'][i % 3],
        overallScore: random(68, 88),
        safetyIndex: random(70, 85),
        connectivityScore: random(65, 85),
        amenitiesScore: random(68, 88),
        environmentScore: random(62, 82),
        growthPotential: random(70, 90),
        transitScore: random(60, 85),
        transitBreakdown: {
          metro: random(60, 90),
          bus: random(70, 90),
          auto: random(75, 95),
          frequency: random(55, 80)
        },
        heatmapPoints: [],
        priceTrend: Array.from({length: 5}, (_, j) => ({
          year: (2020 + j).toString(),
          price: random(6000, 10000) + (j * 400)
        })),
        amenities: [],
        safety: {
          crimeRatePerMonth: random(3, 7),
          policeStationDistKm: random(8, 22) / 10,
          nightSafety: random(65, 85) / 10,
          streetLighting: 'Good',
          cctvCoverage: `${random(55, 85)}%`,
          emergencyResponse: `${random(4, 9)} min`,
          crimeStats: 'Moderate'
        },
        connectivity: {
          metroDistance: random(8, 28) / 10,
          busStops: random(6, 14),
          trafficScore: random(55, 75) / 10,
          roadQuality: 'Good',
          airportDistance: random(22, 45),
          railwayDistance: random(6, 18),
          peakTrafficTime: `${random(28, 42)} min`
        },
        lifestyle: {
          malls: ['Local Mall'],
          restaurants: ['Restaurants'],
          entertainment: ['Cinema'],
          parks: ['Park']
        },
        environment: {
          aqi: random(55, 85),
          noiseDb: random(52, 62),
          greenCoverPct: random(18, 35),
          waterQuality: 'Good',
          airQualityTrend: 'Moderate',
          pollutionSources: 'Traffic'
        },
        upcomingDevelopments: [],
        demographics: {
          avgIncomeLakhs: random(12, 22),
          renterRatioPct: random(45, 75),
          ageMedian: random(32, 38)
        }
      };
    });
  }

  clearCache(): void {
    localityCache.clear();
  }

  getCacheStats(): any {
    return localityCache.getStats();
  }

  getFullStats(): any {
    return {
      cache: localityCache.getStats(),
      api: this.getApiStats()
    };
  }

  private generateEmergencyData(location: string): LocalityData {
    const seed = this.generateSeed(location.toLowerCase());
    const random = (min: number, max: number) => {
      const x = Math.sin(seed) * 10000;
      return Math.floor((x - Math.floor(x)) * (max - min + 1)) + min;
    };
    
    const center = { lat: 12.9716 + (random(-100, 100) / 1000), lng: 77.5946 + (random(-100, 100) / 1000) };
    
    return {
      name: location,
      center: center,
      pricePerSqft: random(8000, 15000),
      oneYearGrowth: random(5, 15),
      fiveYearGrowth: random(40, 90),
      rentalYield: random(25, 45) / 10,
      avgRent: random(20, 50),
      ribl: random(70, 95) / 10,
      demand: ["High", "Very High", "Medium"][random(0, 2)],
      overallScore: random(70, 95),
      safetyIndex: random(75, 90),
      connectivityScore: random(65, 90),
      amenitiesScore: random(70, 90),
      environmentScore: random(60, 85),
      growthPotential: random(70, 95),
      transitScore: random(60, 90),
      transitBreakdown: {
        metro: random(65, 90),
        bus: random(70, 95),
        auto: random(75, 95),
        frequency: random(60, 85)
      },
      heatmapPoints: [],
      priceTrend: Array.from({length: 5}, (_, i) => ({
        year: (2020 + i).toString(),
        price: random(7000, 12000) + (i * 500)
      })),
      amenities: [
        { type: "School", name: "Local School", distKm: random(5, 20) / 10, rating: random(35, 50) / 10, travelTime: `${random(5, 15)} min` },
        { type: "Hospital", name: "Medical Center", distKm: random(10, 30) / 10, rating: random(35, 50) / 10, travelTime: `${random(8, 20)} min` }
      ],
      safety: {
        crimeRatePerMonth: random(2, 8),
        policeStationDistKm: random(5, 25) / 10,
        nightSafety: random(70, 90) / 10,
        streetLighting: ["Good", "Excellent", "Average"][random(0, 2)],
        cctvCoverage: `${random(60, 90)}%`,
        emergencyResponse: `${random(3, 8)} min`,
        crimeStats: "Moderate security with regular patrolling"
      },
      connectivity: {
        metroDistance: random(5, 30) / 10,
        busStops: random(5, 15),
        trafficScore: random(50, 80) / 10,
        roadQuality: ["Good", "Excellent", "Average"][random(0, 2)],
        airportDistance: random(25, 50),
        railwayDistance: random(5, 20),
        peakTrafficTime: `${random(25, 45)} min to CBD`
      },
      lifestyle: {
        malls: ["Shopping Center", "Commercial Complex"],
        restaurants: ["Local Restaurants", "Food Courts"],
        entertainment: ["Cinema", "Recreation Center"],
        parks: ["Community Park", "Green Space"]
      },
      environment: {
        aqi: random(50, 90),
        noiseDb: random(50, 65),
        greenCoverPct: random(15, 40),
        waterQuality: ["Good", "Excellent", "Average"][random(0, 2)],
        airQualityTrend: ["Good", "Moderate", "Poor"][random(0, 2)],
        pollutionSources: "Traffic, Urban development"
      },
      upcomingDevelopments: [
        { project: "Infrastructure Development", timeline: "2024-25", impact: "High" }
      ],
      demographics: {
        avgIncomeLakhs: random(10, 25),
        renterRatioPct: random(40, 80),
        ageMedian: random(30, 40)
      }
    };
  }
}

export const groqService = new GroqService();
