/**
 * Real property data service for heatmap - NO FAKE DATA
 */

import { Property, HeatmapPoint } from '../components/heatmap/types';

interface ExtendedProperty extends Property {
  built_up_area?: number;
  carpet_area?: number;
  super_area?: number;
}

interface HeatmapFilters {
  bounds?: {
    north: number;
    south: number;
    east: number;
    west: number;
  };
  propertyType?: string;
  listingType?: string;
  priceRange?: {
    min: number;
    max: number;
  };
}

interface PropertyDataResponse {
  properties: Property[];
  total: number;
  error?: string;
}

class HeatmapDataService {
  private baseUrl: string;
  private cache = new Map<string, { data: Property[]; timestamp: number }>();
  private readonly CACHE_DURATION = 30 * 1000; // 30 seconds

  constructor() {
    this.baseUrl = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  }

  /**
   * Fetch properties with viewport optimization - NO SAMPLE DATA
   */
  async fetchProperties(filters?: HeatmapFilters): Promise<PropertyDataResponse> {
    try {
      const cacheKey = this.getCacheKey(filters);
      const cached = this.cache.get(cacheKey);
      
      // Return cached data if valid
      if (cached && Date.now() - cached.timestamp < this.CACHE_DURATION) {
        return { properties: cached.data, total: cached.data.length };
      }

      const params = new URLSearchParams();
      
      // Viewport bounds filtering removed to show all India properties

      if (filters?.propertyType) {
        params.append('property_type', filters.propertyType);
      }

      if (filters?.listingType) {
        params.append('listing_type', filters.listingType);
      }

      if (filters?.priceRange) {
        params.append('min_price', filters.priceRange.min.toString());
        params.append('max_price', filters.priceRange.max.toString());
      }

      // Set reasonable limit for heatmap
      params.append('limit', '100');

      const response = await fetch(`${this.baseUrl}/properties/?${params.toString()}`);
      
      if (!response.ok) {
        throw new Error(`API Error: ${response.status} ${response.statusText}`);
      }

      const data = await response.json();
      
      // Validate and filter properties
      const validProperties = this.validateProperties(data.properties || []);
      
      if (validProperties.length === 0) {
        return { 
          properties: [], 
          total: 0, 
          error: 'No active properties found with valid coordinates and pricing data in the current area' 
        };
      }

      // Cache the results
      this.cache.set(cacheKey, { 
        data: validProperties, 
        timestamp: Date.now() 
      });

      return { properties: validProperties, total: validProperties.length };

    } catch (error) {
      console.error('Failed to fetch properties:', error);
      return { 
        properties: [], 
        total: 0, 
        error: error instanceof Error ? error.message : 'Unknown error occurred' 
      };
    }
  }

  /**
   * Validate properties - reject invalid data
   */
  private validateProperties(properties: any[]): Property[] {
    return properties
      .filter(this.isValidProperty)
      .map(this.normalizeProperty)
      .filter(Boolean) as Property[];
  }

  /**
   * Strict validation - NO FAKE DATA ALLOWED
   */
  private isValidProperty(property: any): boolean {
    console.log('Validating property:', property.id, property.listing_type, property.property_type);
    
    // Must have valid coordinates
    if (!property.latitude || !property.longitude) {
      console.log('Rejected - missing coordinates:', property.id);
      return false;
    }
    if (isNaN(property.latitude) || isNaN(property.longitude)) {
      console.log('Rejected - invalid coordinates:', property.id);
      return false;
    }
    
    // Coordinates must be within reasonable bounds
    const lat = parseFloat(property.latitude);
    const lng = parseFloat(property.longitude);
    
    if (lat < -90 || lat > 90 || lng < -180 || lng > 180) {
      console.log('Rejected - coordinates out of bounds:', property.id);
      return false;
    }
    
    // Must have valid pricing based on listing type
    let price = 0;
    
    if (property.listing_type === 'sell') {
      price = property.sale_price || 0;
    } else if (property.listing_type === 'rent') {
      price = property.monthly_rent ? property.monthly_rent * 12 : 0;
    } else if (property.listing_type === 'lease') {
      price = property.monthly_lease_amount ? property.monthly_lease_amount * 12 : 0;
    } else if (property.listing_type === 'urgent-sale') {
      price = property.actual_price || property.urgent_pricing || 0;
    } else if (property.listing_type === 'bidding') {
      price = property.starting_bid_price || 0;
    }
    
    console.log('Property price calculated:', property.id, price);
    
    if (price <= 0) {
      console.log('Rejected - no valid price:', property.id, property.listing_type);
      return false;
    }
    
    // Reject obviously fake prices
    if (price < 100000 || price > 1000000000) {
      console.log('Rejected - price out of range:', property.id, price);
      return false;
    }

    // Must have basic property info
    if (!property.id || !property.property_type || !property.listing_type) {
      console.log('Rejected - missing basic info:', property.id, property.property_type, property.listing_type);
      return false;
    }

    console.log('Property accepted:', property.id);
    return true;
  }

  /**
   * Normalize property data
   */
  private normalizeProperty(property: any): Property | null {
    try {
      // Normalize price based on listing type
      let normalizedPrice = 0;
      
      if (property.listing_type === 'sell') {
        normalizedPrice = property.sale_price ? parseFloat(property.sale_price) : 0;
      } else if (property.listing_type === 'rent') {
        normalizedPrice = 0; // Keep monthly_rent separate
      } else if (property.listing_type === 'lease') {
        normalizedPrice = property.monthly_lease_amount ? parseFloat(property.monthly_lease_amount) * 12 : 0;
      } else if (property.listing_type === 'urgent-sale') {
        normalizedPrice = property.actual_price ? parseFloat(property.actual_price) : 
                         property.urgent_pricing ? parseFloat(property.urgent_pricing) : 0;
      } else if (property.listing_type === 'bidding') {
        normalizedPrice = property.starting_bid_price ? parseFloat(property.starting_bid_price) : 0;
      }
      
      return {
        id: property.id,
        latitude: parseFloat(property.latitude),
        longitude: parseFloat(property.longitude),
        sale_price: normalizedPrice,
        monthly_rent: property.monthly_rent ? parseFloat(property.monthly_rent) : 0,
        property_type: property.property_type,
        listing_type: property.listing_type,
        city: property.city || 'Unknown',
        built_up_area: property.built_up_area ? parseFloat(property.built_up_area) : undefined,
        carpet_area: property.carpet_area ? parseFloat(property.carpet_area) : undefined,
        super_area: property.super_area ? parseFloat(property.super_area) : undefined
      };
    } catch (error) {
      console.warn('Failed to normalize property:', property.id, error);
      return null;
    }
  }

  /**
   * Generate cache key for filters
   */
  private getCacheKey(filters?: HeatmapFilters): string {
    if (!filters) return 'default';
    
    return JSON.stringify({
      bounds: filters.bounds,
      propertyType: filters.propertyType,
      listingType: filters.listingType,
      priceRange: filters.priceRange
    });
  }

  /**
   * Clear cache - call when data updates
   */
  clearCache(): void {
    this.cache.clear();
  }

  /**
   * Get cache statistics
   */
  getCacheStats(): { size: number; keys: string[] } {
    return {
      size: this.cache.size,
      keys: Array.from(this.cache.keys())
    };
  }

  /**
   * Fetch properties for specific viewport bounds
   */
  async fetchPropertiesInViewport(
    bounds: { north: number; south: number; east: number; west: number },
    additionalFilters?: Omit<HeatmapFilters, 'bounds'>
  ): Promise<PropertyDataResponse> {
    return this.fetchProperties({
      bounds,
      ...additionalFilters
    });
  }

  /**
   * Get data quality metrics
   */
  getDataQuality(properties: Property[]): {
    totalProperties: number;
    withCoordinates: number;
    withPricing: number;
    withArea: number;
    qualityScore: number;
  } {
    const withCoordinates = properties.filter(p => p.latitude && p.longitude).length;
    const withPricing = properties.filter(p => p.sale_price > 0 || p.monthly_rent > 0).length;
    const withArea = properties.filter(p => 
      p.built_up_area || p.carpet_area || p.super_area
    ).length;
    
    const qualityScore = properties.length > 0 ? 
      ((withCoordinates + withPricing + withArea) / (properties.length * 3)) * 100 : 0;
    
    return {
      totalProperties: properties.length,
      withCoordinates,
      withPricing,
      withArea,
      qualityScore: Math.round(qualityScore)
    };
  }
}

export const heatmapDataService = new HeatmapDataService();
export default heatmapDataService;