// Image Search Service for Property Similarity Matching
import { api } from '../utils/api';

export interface ImageSearchResult {
  success: boolean;
  similarProperties?: any[];
  error?: string;
  confidence?: number;
}

export interface ImageFeatures {
  dominantColors: string[];
  brightness: number;
  contrast: number;
  textureFeatures: number[];
  objectDetection: string[];
}

class ImageSearchService {
  private canvas: HTMLCanvasElement;
  private ctx: CanvasRenderingContext2D;

  constructor() {
    this.canvas = document.createElement('canvas');
    this.ctx = this.canvas.getContext('2d')!;
  }

  /**
   * Extract basic visual features from an uploaded image
   */
  async extractImageFeatures(file: File): Promise<ImageFeatures> {
    return new Promise((resolve, reject) => {
      const img = new Image();
      const url = URL.createObjectURL(file);
      
      img.onload = () => {
        try {
          // Set canvas size to image size
          this.canvas.width = img.width;
          this.canvas.height = img.height;
          
          // Draw image on canvas
          this.ctx.drawImage(img, 0, 0);
          
          // Extract image data
          const imageData = this.ctx.getImageData(0, 0, img.width, img.height);
          const data = imageData.data;
          
          // Extract features
          const features: ImageFeatures = {
            dominantColors: this.extractDominantColors(data),
            brightness: this.calculateBrightness(data),
            contrast: this.calculateContrast(data),
            textureFeatures: this.extractTextureFeatures(data, img.width, img.height),
            objectDetection: this.detectBasicObjects(data, img.width, img.height)
          };
          
          URL.revokeObjectURL(url);
          resolve(features);
        } catch (error) {
          URL.revokeObjectURL(url);
          reject(error);
        }
      };
      
      img.onerror = () => {
        URL.revokeObjectURL(url);
        reject(new Error('Failed to load image'));
      };
      
      img.src = url;
    });
  }

  /**
   * Extract dominant colors from image data
   */
  private extractDominantColors(data: Uint8ClampedArray): string[] {
    const colorMap = new Map<string, number>();
    
    // Sample every 10th pixel for performance
    for (let i = 0; i < data.length; i += 40) {
      const r = Math.floor(data[i] / 32) * 32;
      const g = Math.floor(data[i + 1] / 32) * 32;
      const b = Math.floor(data[i + 2] / 32) * 32;
      
      const color = `rgb(${r},${g},${b})`;
      colorMap.set(color, (colorMap.get(color) || 0) + 1);
    }
    
    // Get top 5 dominant colors
    return Array.from(colorMap.entries())
      .sort((a, b) => b[1] - a[1])
      .slice(0, 5)
      .map(([color]) => color);
  }

  /**
   * Calculate average brightness
   */
  private calculateBrightness(data: Uint8ClampedArray): number {
    let total = 0;
    let count = 0;
    
    for (let i = 0; i < data.length; i += 4) {
      const brightness = (data[i] + data[i + 1] + data[i + 2]) / 3;
      total += brightness;
      count++;
    }
    
    return total / count;
  }

  /**
   * Calculate contrast using standard deviation
   */
  private calculateContrast(data: Uint8ClampedArray): number {
    const brightness = this.calculateBrightness(data);
    let variance = 0;
    let count = 0;
    
    for (let i = 0; i < data.length; i += 4) {
      const pixelBrightness = (data[i] + data[i + 1] + data[i + 2]) / 3;
      variance += Math.pow(pixelBrightness - brightness, 2);
      count++;
    }
    
    return Math.sqrt(variance / count);
  }

  /**
   * Extract basic texture features using edge detection
   */
  private extractTextureFeatures(data: Uint8ClampedArray, width: number, height: number): number[] {
    const features: number[] = [];
    
    // Simple edge detection (horizontal and vertical gradients)
    let horizontalEdges = 0;
    let verticalEdges = 0;
    
    for (let y = 1; y < height - 1; y++) {
      for (let x = 1; x < width - 1; x++) {
        const idx = (y * width + x) * 4;
        
        // Get grayscale values
        const current = (data[idx] + data[idx + 1] + data[idx + 2]) / 3;
        const right = (data[idx + 4] + data[idx + 5] + data[idx + 6]) / 3;
        const bottom = (data[(y + 1) * width * 4 + x * 4] + 
                       data[(y + 1) * width * 4 + x * 4 + 1] + 
                       data[(y + 1) * width * 4 + x * 4 + 2]) / 3;
        
        horizontalEdges += Math.abs(current - right);
        verticalEdges += Math.abs(current - bottom);
      }
    }
    
    features.push(horizontalEdges / (width * height));
    features.push(verticalEdges / (width * height));
    
    return features;
  }

  /**
   * Basic object detection (simplified)
   */
  private detectBasicObjects(data: Uint8ClampedArray, width: number, height: number): string[] {
    const objects: string[] = [];
    
    // Detect if image is likely a building (high contrast, vertical lines)
    const contrast = this.calculateContrast(data);
    const brightness = this.calculateBrightness(data);
    
    if (contrast > 50) {
      objects.push('building');
    }
    
    if (brightness > 150) {
      objects.push('bright_exterior');
    } else if (brightness < 100) {
      objects.push('interior');
    }
    
    // Check for dominant colors that might indicate property types
    const dominantColors = this.extractDominantColors(data);
    const hasGreen = dominantColors.some(color => 
      color.includes('rgb(0,') || color.includes('rgb(32,64,') || color.includes('rgb(64,96,')
    );
    
    if (hasGreen) {
      objects.push('garden', 'outdoor');
    }
    
    return objects;
  }

  /**
   * Search for similar properties based on image features
   */
  async searchSimilarProperties(file: File): Promise<ImageSearchResult> {
    try {
      console.log('🔍 Starting image-based property search...');
      
      // Extract features from uploaded image
      const imageFeatures = await this.extractImageFeatures(file);
      console.log('📊 Extracted image features:', imageFeatures);
      
      // Get all properties to compare against
      const allProperties = await api.fetchPropertiesWithFilters({});
      console.log('📋 Total properties to compare:', allProperties.length);
      
      if (allProperties.length === 0) {
        return {
          success: false,
          error: 'No properties available for comparison'
        };
      }
      
      // Calculate similarity scores
      const similarityScores = await Promise.all(
        allProperties.map(async (property: any) => {
          const score = await this.calculateSimilarityScore(imageFeatures, property);
          return { property, score };
        })
      );
      
      // Sort by similarity score and get top matches
      const topMatches = similarityScores
        .filter(item => item.score > 0.3) // Minimum similarity threshold
        .sort((a, b) => b.score - a.score)
        .slice(0, 10) // Top 10 matches
        .map(item => ({
          ...item.property,
          similarityScore: item.score
        }));
      
      console.log('🎯 Found similar properties:', topMatches.length);
      console.log('📊 Top similarity scores:', topMatches.map(p => p.similarityScore));
      
      return {
        success: true,
        similarProperties: topMatches,
        confidence: topMatches.length > 0 ? topMatches[0].similarityScore : 0
      };
      
    } catch (error) {
      console.error('❌ Image search error:', error);
      return {
        success: false,
        error: error instanceof Error ? error.message : 'Image search failed'
      };
    }
  }

  /**
   * Calculate similarity score between uploaded image and property
   */
  private async calculateSimilarityScore(imageFeatures: ImageFeatures, property: any): Promise<number> {
    let score = 0;
    let factors = 0;
    
    // If property has images, try to compare visual features
    if (property.imageUrl || property.primary_image_url) {
      try {
        // For now, use property metadata to estimate similarity
        // In a full implementation, you'd extract features from property images too
        
        // Property type matching
        if (imageFeatures.objectDetection.includes('building')) {
          if (property.property_type === 'apartment' || property.property_type === 'villa') {
            score += 0.3;
          }
          factors++;
        }
        
        // Brightness-based matching (interior vs exterior)
        if (imageFeatures.brightness > 150 && property.property_type === 'villa') {
          score += 0.2; // Bright images often match villas/houses
          factors++;
        } else if (imageFeatures.brightness < 100 && property.property_type === 'apartment') {
          score += 0.2; // Darker images often match apartment interiors
          factors++;
        }
        
        // Garden/outdoor detection
        if (imageFeatures.objectDetection.includes('garden') || imageFeatures.objectDetection.includes('outdoor')) {
          if (property.amenities?.includes('Garden') || property.property_type === 'villa') {
            score += 0.25;
            factors++;
          }
        }
        
        // Contrast-based matching (modern vs traditional)
        if (imageFeatures.contrast > 60) {
          // High contrast suggests modern architecture
          if (property.property_age === 'new' || property.property_age === '1-5') {
            score += 0.15;
            factors++;
          }
        }
        
      } catch (error) {
        console.warn('Error calculating visual similarity:', error);
      }
    }
    
    // Normalize score
    return factors > 0 ? score / factors : 0;
  }

  /**
   * Enhanced search with AI-like property matching
   */
  async intelligentPropertySearch(file: File, userPreferences?: any): Promise<ImageSearchResult> {
    try {
      const basicResult = await this.searchSimilarProperties(file);
      
      if (!basicResult.success || !basicResult.similarProperties) {
        return basicResult;
      }
      
      // Apply user preferences if available
      let filteredProperties = basicResult.similarProperties;
      
      if (userPreferences) {
        filteredProperties = filteredProperties.filter(property => {
          let matches = true;
          
          if (userPreferences.maxPrice && property.sale_price > userPreferences.maxPrice) {
            matches = false;
          }
          
          if (userPreferences.minBedrooms && property.bedrooms < userPreferences.minBedrooms) {
            matches = false;
          }
          
          if (userPreferences.city && property.city !== userPreferences.city) {
            matches = false;
          }
          
          if (userPreferences.propertyType && property.property_type !== userPreferences.propertyType) {
            matches = false;
          }
          
          return matches;
        });
      }
      
      return {
        success: true,
        similarProperties: filteredProperties,
        confidence: basicResult.confidence
      };
      
    } catch (error) {
      return {
        success: false,
        error: error instanceof Error ? error.message : 'Intelligent search failed'
      };
    }
  }
}

export const imageSearchService = new ImageSearchService();