import { apiClient } from '@/config/api';

export interface CreateLegalHealthReportRequest {
  property_address: string;
  property_type: string;
  property_id?: string;
}

export interface LegalHealthReport {
  id: string;
  user_id: string;
  property_id?: string;
  report_number: string;
  property_address: string;
  property_type: string;
  report_status: 'initiated' | 'in_progress' | 'completed' | 'failed';
  overall_score: number;
  risk_level: 'low' | 'medium' | 'high' | 'critical' | 'unknown';
  recommendation?: string;
  estimated_completion_date?: string;
  completed_at?: string;
  created_at: string;
  updated_at: string;
}

export interface VerificationModule {
  id: string;
  report_id: string;
  module_type: 'ownership' | 'documents' | 'litigation' | 'encumbrance' | 'zoning' | 'project_approval';
  status: 'pending' | 'in_progress' | 'completed' | 'failed' | 'not_applicable';
  score: number;
  findings: Record<string, any>;
  issues: string[];
  recommendations: string[];
  verified_by?: string;
  verified_at?: string;
  created_at: string;
  updated_at: string;
}

export interface DocumentSubmission {
  id: string;
  report_id: string;
  module_type: string;
  document_type: string;
  document_name: string;
  file_path: string;
  file_size: number;
  mime_type: string;
  verification_status: 'pending' | 'verified' | 'rejected' | 'requires_clarification';
  verification_notes?: string;
  uploaded_at: string;
  verified_at?: string;
}

export interface LegalIssue {
  id: string;
  report_id: string;
  module_type: string;
  issue_type: string;
  severity: 'low' | 'medium' | 'high' | 'critical';
  title: string;
  description: string;
  impact_assessment?: string;
  recommended_action?: string;
  resolution_timeline?: string;
  status: 'open' | 'in_progress' | 'resolved' | 'cannot_resolve';
  created_at: string;
  updated_at: string;
}

export interface ChecklistItem {
  id: string;
  report_id: string;
  module_type: string;
  check_item: string;
  is_required: boolean;
  status: 'pending' | 'passed' | 'failed' | 'not_applicable';
  notes?: string;
  verified_by?: string;
  verified_at?: string;
  created_at: string;
}

export interface ReportTimeline {
  id: string;
  report_id: string;
  event_type: string;
  event_title: string;
  event_description?: string;
  event_data: Record<string, any>;
  created_by?: string;
  created_at: string;
}

export interface LegalHealthReportDetail {
  report: LegalHealthReport;
  modules: VerificationModule[];
  documents: DocumentSubmission[];
  issues: LegalIssue[];
  checklist: ChecklistItem[];
  timeline: ReportTimeline[];
}

export interface DashboardStats {
  total_reports: number;
  completed_reports: number;
  in_progress_reports: number;
  average_completion_time?: number;
  risk_distribution: Record<string, number>;
  recent_reports: LegalHealthReport[];
}

class LegalHealthAPI {
  private baseURL = '/api/legal-health';

  async createReport(data: CreateLegalHealthReportRequest) {
    const response = await apiClient.post(`${this.baseURL}/reports`, data);
    return response.data;
  }

  async getUserReports(limit = 10, offset = 0) {
    const response = await apiClient.get(`${this.baseURL}/reports`, {
      params: { limit, offset }
    });
    return response.data;
  }

  async getReportDetail(reportId: string) {
    const response = await apiClient.get(`${this.baseURL}/reports/${reportId}`);
    return response.data;
  }

  async uploadDocument(reportId: string, moduleType: string, documentType: string, file: File) {
    const formData = new FormData();
    formData.append('file', file);
    formData.append('module_type', moduleType);
    formData.append('document_type', documentType);

    const response = await apiClient.post(
      `${this.baseURL}/reports/${reportId}/documents`,
      formData,
      {
        headers: {
          'Content-Type': 'multipart/form-data',
        },
      }
    );
    return response.data;
  }

  async updateVerificationModule(
    reportId: string, 
    moduleType: string, 
    data: {
      status: string;
      score?: number;
      findings?: Record<string, any>;
      issues?: string[];
      recommendations?: string[];
      verified_by?: string;
    }
  ) {
    const response = await apiClient.put(
      `${this.baseURL}/reports/${reportId}/modules/${moduleType}`,
      data
    );
    return response.data;
  }

  async createLegalIssue(reportId: string, data: {
    module_type: string;
    issue_type: string;
    severity: string;
    title: string;
    description: string;
    impact_assessment?: string;
    recommended_action?: string;
    resolution_timeline?: string;
  }) {
    const response = await apiClient.post(`${this.baseURL}/reports/${reportId}/issues`, data);
    return response.data;
  }

  async updateChecklistItem(checklistId: string, data: {
    status: string;
    notes?: string;
    verified_by?: string;
  }) {
    const response = await apiClient.put(`${this.baseURL}/checklist/${checklistId}`, data);
    return response.data;
  }

  async getDashboardStats() {
    const response = await apiClient.get(`${this.baseURL}/dashboard`);
    return response.data;
  }

  async getModuleStats(reportId: string, moduleType: string) {
    const response = await apiClient.get(
      `${this.baseURL}/reports/${reportId}/modules/${moduleType}/stats`
    );
    return response.data;
  }

  async generateCertificate(reportId: string) {
    const response = await apiClient.get(`${this.baseURL}/reports/${reportId}/certificate`);
    return response.data;
  }
}

export const legalHealthAPI = new LegalHealthAPI();