import axios from 'axios';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8090/api';

export interface ForgotPasswordRequest {
  email: string;
}

export interface ValidateTokenRequest {
  token: string;
}

export interface ResetPasswordRequest {
  token: string;
  new_password: string;
  confirm_password: string;
}

export interface PasswordResetResponse {
  success: boolean;
  message: string;
  expires_in?: number;
}

export interface ValidateTokenResponse {
  success: boolean;
  message: string;
  user_type?: string;
  email?: string;
}

export interface ResetPasswordResponse {
  success: boolean;
  message: string;
}

class PasswordResetService {
  // Users/Buyers Password Reset
  async forgotPassword(email: string): Promise<PasswordResetResponse> {
    const response = await axios.post(`${API_BASE_URL}/auth/forgot-password`, {
      email
    });
    return response.data;
  }

  async validateResetToken(token: string): Promise<ValidateTokenResponse> {
    const response = await axios.post(`${API_BASE_URL}/auth/validate-reset-token`, {
      token
    });
    return response.data;
  }

  async resetPassword(token: string, newPassword: string, confirmPassword: string): Promise<ResetPasswordResponse> {
    const response = await axios.post(`${API_BASE_URL}/auth/reset-password`, {
      token,
      new_password: newPassword,
      confirm_password: confirmPassword
    });
    return response.data;
  }

  // Seller/Agent Password Reset
  async sellerAgentForgotPassword(email: string): Promise<PasswordResetResponse> {
    const response = await axios.post(`${API_BASE_URL}/seller-agent-auth/forgot-password`, {
      email
    });
    return response.data;
  }

  async sellerAgentValidateToken(token: string): Promise<ValidateTokenResponse> {
    const response = await axios.post(`${API_BASE_URL}/seller-agent-auth/validate-token`, {
      token
    });
    return response.data;
  }

  async sellerAgentResetPassword(token: string, newPassword: string, confirmPassword: string): Promise<ResetPasswordResponse> {
    const response = await axios.post(`${API_BASE_URL}/seller-agent-auth/reset-password`, {
      token,
      new_password: newPassword,
      confirm_password: confirmPassword
    });
    return response.data;
  }
}

export const passwordResetService = new PasswordResetService();