// reCAPTCHA v3 service for frontend
class RecaptchaService {
  private siteKey: string = '6LdtgjUsAAAAAPpAIX98_TkHUT14kEWGekD9fmFo';
  private enabled: boolean = false; // Disabled reCAPTCHA
  private scriptLoaded: boolean = false;

  async initialize(): Promise<void> {
    try {
      if (this.enabled && this.siteKey) {
        await this.loadScript();
      }
    } catch (error) {
      console.warn('Failed to load reCAPTCHA script:', error);
      this.enabled = false;
    }
  }

  private async loadScript(): Promise<void> {
    if (this.scriptLoaded || !this.siteKey) return;

    return new Promise((resolve, reject) => {
      const script = document.createElement('script');
      script.src = `https://www.google.com/recaptcha/api.js?render=${this.siteKey}`;
      script.async = true;
      script.defer = true;
      
      script.onload = () => {
        this.scriptLoaded = true;
        resolve();
      };
      
      script.onerror = () => {
        reject(new Error('Failed to load reCAPTCHA script'));
      };
      
      document.head.appendChild(script);
    });
  }

  async getToken(action: string): Promise<string | null> {
    if (!this.enabled || !this.siteKey) {
      console.warn('reCAPTCHA not enabled or site key missing');
      return null;
    }

    if (!this.scriptLoaded) {
      console.warn('reCAPTCHA script not loaded yet, attempting to load...');
      try {
        await this.loadScript();
      } catch (error) {
        console.error('Failed to load reCAPTCHA script:', error);
        return null;
      }
    }

    if (!window.grecaptcha) {
      console.warn('reCAPTCHA not available on window object');
      return null;
    }

    try {
      const token = await window.grecaptcha.execute(this.siteKey, { action });
      console.log('✅ reCAPTCHA token generated for action:', action);
      return token;
    } catch (error) {
      console.error('reCAPTCHA token generation failed:', error);
      return null;
    }
  }

  isEnabled(): boolean {
    return this.enabled;
  }

  getSiteKey(): string {
    return this.siteKey;
  }
}

// Global reCAPTCHA service instance
export const recaptchaService = new RecaptchaService();

// Initialize on module load
recaptchaService.initialize();

// Extend window interface for TypeScript
declare global {
  interface Window {
    grecaptcha: {
      execute: (siteKey: string, options: { action: string }) => Promise<string>;
      ready: (callback: () => void) => void;
      render: (container: HTMLElement, options: {
        sitekey: string;
        callback: (token: string) => void;
        'expired-callback'?: () => void;
        'error-callback'?: () => void;
      }) => void;
    };
  }
}