import jsPDF from 'jspdf';
import autoTable from 'jspdf-autotable';
import { AdminUserStats, AdminPropertyStats } from '../api/admin';

// Extend jsPDF type
declare module 'jspdf' {
  interface jsPDF {
    lastAutoTable: { finalY: number };
  }
}

export const reportService = {
  async downloadDashboardReport(userStats?: AdminUserStats, propertyStats?: AdminPropertyStats) {
    try {
      const doc = new jsPDF();
      const date = new Date().toLocaleDateString('en-GB');
      
      // Load logo
      const logoImg = new Image();
      logoImg.src = '/favicon.png';
      
      await new Promise((resolve) => {
        logoImg.onload = () => {
          // Blue Header Background
          doc.setFillColor(43, 37, 109);
          doc.rect(0, 0, 210, 50, 'F');
          
          // Add Logo
          doc.addImage(logoImg, 'PNG', 14, 10, 30, 30);
          
          // Header Text
          doc.setTextColor(255, 255, 255);
          doc.setFontSize(24);
          doc.setFont('helvetica', 'bold');
          doc.text('ADMIN DASHBOARD REPORT', 115, 22, { align: 'center' });
          
          doc.setFontSize(12);
          doc.setFont('helvetica', 'normal');
          doc.text('Real-time Platform Statistics and Insights', 105, 32, { align: 'center' });
          
          doc.setFontSize(10);
          doc.text(`Generated: ${date}`, 105, 42, { align: 'center' });
          
          resolve(true);
        };
        logoImg.onerror = () => resolve(true);
      });

      let currentY = 65;

      // User Statistics Section
      if (userStats) {
        doc.setFontSize(16);
        doc.setTextColor(0, 0, 0);
        doc.setFont('helvetica', 'bold');
        doc.text('USER STATISTICS', 14, currentY);
        
        autoTable(doc, {
          startY: currentY + 5,
          head: [['Metric', 'Value']],
          body: [
            ['Total Users', userStats.total_users.toLocaleString()],
            ['Active Users', userStats.active_users.toLocaleString()],
            ['Verified Users', `${userStats.verified_users?.toLocaleString() || 0} (${userStats.total_users > 0 ? Math.round((userStats.verified_users || 0) / userStats.total_users * 100) : 0}%)`],
            ['Complete Profiles', `${userStats.users_with_complete_profile?.toLocaleString() || 0} (${userStats.total_users > 0 ? Math.round((userStats.users_with_complete_profile || 0) / userStats.total_users * 100) : 0}%)`],
            ['New Users Today', userStats.new_users_today.toString()],
            ['New Users This Week', userStats.new_users_this_week?.toString() || '0'],
            ['New Users This Month', userStats.new_users_this_month.toString()]
          ],
          theme: 'striped',
          headStyles: { fillColor: [43, 37, 109] as any, textColor: [255, 255, 255], fontStyle: 'bold' },
          styles: { fontSize: 11, cellPadding: 4 }
        });

        currentY = doc.lastAutoTable.finalY + 15;

        // Users by Role
        if (userStats.users_by_role && Object.keys(userStats.users_by_role).length > 0) {
          doc.setFontSize(14);
          doc.setFont('helvetica', 'bold');
          doc.text('Users by Role', 14, currentY);
          
          autoTable(doc, {
            startY: currentY + 5,
            head: [['Role', 'Count']],
            body: Object.entries(userStats.users_by_role).map(([role, count]) => [
              role.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase()),
              count.toString()
            ]),
            theme: 'striped',
            headStyles: { fillColor: [43, 37, 109] as any, textColor: [255, 255, 255], fontStyle: 'bold' },
            styles: { fontSize: 10, cellPadding: 3 }
          });
          currentY = doc.lastAutoTable.finalY + 15;
        }
      }

      // Property Statistics Section
      if (propertyStats) {
        doc.setFontSize(16);
        doc.setFont('helvetica', 'bold');
        doc.text('PROPERTY STATISTICS', 14, currentY);
        
        autoTable(doc, {
          startY: currentY + 5,
          head: [['Metric', 'Value']],
          body: [
            ['Total Properties', propertyStats.total_properties.toLocaleString()],
            ['Active Properties', propertyStats.active_properties.toLocaleString()],
            ['Sold Properties', `${propertyStats.sold_properties.toLocaleString()} (${propertyStats.total_properties > 0 ? Math.round(propertyStats.sold_properties / propertyStats.total_properties * 100) : 0}%)`],
            ['Rented Properties', `${propertyStats.rented_properties.toLocaleString()} (${propertyStats.total_properties > 0 ? Math.round(propertyStats.rented_properties / propertyStats.total_properties * 100) : 0}%)`],
            ['Average Price', propertyStats.average_price ? `₹${(propertyStats.average_price / 100000).toFixed(1)}L` : '₹0'],
            ['New Properties Today', propertyStats.new_properties_today?.toString() || '0'],
            ['New Properties This Week', propertyStats.new_properties_this_week?.toString() || '0'],
            ['New Properties This Month', propertyStats.new_properties_this_month?.toString() || '0']
          ],
          theme: 'striped',
          headStyles: { fillColor: [43, 37, 109] as any, textColor: [255, 255, 255], fontStyle: 'bold' },
          styles: { fontSize: 11, cellPadding: 4 }
        });

        currentY = doc.lastAutoTable.finalY + 15;

        // Properties by Type
        if (propertyStats.properties_by_type && Object.keys(propertyStats.properties_by_type).length > 0) {
          doc.setFontSize(14);
          doc.setFont('helvetica', 'bold');
          doc.text('Properties by Type', 14, currentY);
          
          autoTable(doc, {
            startY: currentY + 5,
            head: [['Type', 'Count']],
            body: Object.entries(propertyStats.properties_by_type).map(([type, count]) => [
              type.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase()),
              count.toString()
            ]),
            theme: 'striped',
            headStyles: { fillColor: [43, 37, 109] as any, textColor: [255, 255, 255], fontStyle: 'bold' },
            styles: { fontSize: 10, cellPadding: 3 }
          });
          currentY = doc.lastAutoTable.finalY + 15;
        }

        // Properties by City
        if (propertyStats.properties_by_city && Object.keys(propertyStats.properties_by_city).length > 0) {
          doc.setFontSize(14);
          doc.setFont('helvetica', 'bold');
          doc.text('Properties by City', 14, currentY);
          
          autoTable(doc, {
            startY: currentY + 5,
            head: [['City', 'Count']],
            body: Object.entries(propertyStats.properties_by_city)
              .sort(([, a], [, b]) => b - a)
              .slice(0, 10)
              .map(([city, count]) => [city, count.toString()]),
            theme: 'striped',
            headStyles: { fillColor: [43, 37, 109] as any, textColor: [255, 255, 255], fontStyle: 'bold' },
            styles: { fontSize: 10, cellPadding: 3 }
          });
        }
      }

      // Footer
      const totalPages = doc.internal.getNumberOfPages();
      for (let i = 1; i <= totalPages; i++) {
        doc.setPage(i);
        doc.setFillColor(43, 37, 109);
        doc.rect(0, 275, 210, 22, 'F');
        doc.setTextColor(255, 255, 255);
        doc.setFontSize(10);
        doc.setFont('helvetica', 'bold');
        doc.text('NAL INDIA - New Age Land', 14, 285);
        doc.setFont('helvetica', 'normal');
        doc.setFontSize(8);
        doc.text('www.nalindia.com | Professional Real Estate Platform', 14, 292);
        doc.setFontSize(8);
        doc.text(`Page ${i} of ${totalPages}`, 196, 285, { align: 'right' });
      }

      const filename = `admin-dashboard-report-${new Date().toISOString().split('T')[0]}.pdf`;
      doc.save(filename);
      return filename;
    } catch (error: any) {
      console.error('Report generation error:', error);
      throw new Error(error.message || 'Failed to generate report');
    }
  },

  exportToCSV(userStats?: AdminUserStats, propertyStats?: AdminPropertyStats) {
    try {
      let csvContent = 'data:text/csv;charset=utf-8,';
      
      // User Statistics
      if (userStats) {
        csvContent += 'USER STATISTICS\n';
        csvContent += 'Metric,Value\n';
        csvContent += `Total Users,${userStats.total_users}\n`;
        csvContent += `Active Users,${userStats.active_users}\n`;
        csvContent += `Verified Users,${userStats.verified_users || 0}\n`;
        csvContent += `Complete Profiles,${userStats.users_with_complete_profile || 0}\n`;
        csvContent += `New Users Today,${userStats.new_users_today}\n`;
        csvContent += `New Users This Week,${userStats.new_users_this_week || 0}\n`;
        csvContent += `New Users This Month,${userStats.new_users_this_month}\n`;
        csvContent += '\n';

        if (userStats.users_by_role && Object.keys(userStats.users_by_role).length > 0) {
          csvContent += 'USERS BY ROLE\n';
          csvContent += 'Role,Count\n';
          Object.entries(userStats.users_by_role).forEach(([role, count]) => {
            csvContent += `${role},${count}\n`;
          });
          csvContent += '\n';
        }
      }

      // Property Statistics
      if (propertyStats) {
        csvContent += 'PROPERTY STATISTICS\n';
        csvContent += 'Metric,Value\n';
        csvContent += `Total Properties,${propertyStats.total_properties}\n`;
        csvContent += `Active Properties,${propertyStats.active_properties}\n`;
        csvContent += `Sold Properties,${propertyStats.sold_properties}\n`;
        csvContent += `Rented Properties,${propertyStats.rented_properties}\n`;
        csvContent += `Average Price,${propertyStats.average_price || 0}\n`;
        csvContent += `New Properties Today,${propertyStats.new_properties_today || 0}\n`;
        csvContent += `New Properties This Week,${propertyStats.new_properties_this_week || 0}\n`;
        csvContent += `New Properties This Month,${propertyStats.new_properties_this_month || 0}\n`;
        csvContent += '\n';

        if (propertyStats.properties_by_type && Object.keys(propertyStats.properties_by_type).length > 0) {
          csvContent += 'PROPERTIES BY TYPE\n';
          csvContent += 'Type,Count\n';
          Object.entries(propertyStats.properties_by_type).forEach(([type, count]) => {
            csvContent += `${type},${count}\n`;
          });
          csvContent += '\n';
        }

        if (propertyStats.properties_by_city && Object.keys(propertyStats.properties_by_city).length > 0) {
          csvContent += 'PROPERTIES BY CITY\n';
          csvContent += 'City,Count\n';
          Object.entries(propertyStats.properties_by_city)
            .sort(([, a], [, b]) => b - a)
            .forEach(([city, count]) => {
              csvContent += `${city},${count}\n`;
            });
        }
      }

      const encodedUri = encodeURI(csvContent);
      const link = document.createElement('a');
      link.setAttribute('href', encodedUri);
      link.setAttribute('download', `admin-dashboard-export-${new Date().toISOString().split('T')[0]}.csv`);
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      
      return `admin-dashboard-export-${new Date().toISOString().split('T')[0]}.csv`;
    } catch (error: any) {
      console.error('CSV export error:', error);
      throw new Error(error.message || 'Failed to export data');
    }
  }
};
