export interface School {
  id: string;
  name: string;
  lat: number;
  lon: number;
  distance: number;
  driveTime: string;
  address?: string;
  schoolType?: string;
}

interface SchoolsResponse {
  schools: School[];
  methodUsed: 'geoapify';
}

const GEOAPIFY_API_KEY = '820cfeab3e7b43018d00e835d3df6bc3';



export const fetchNearbySchools = async (lat: number, lon: number, radiusKm: number = 5): Promise<SchoolsResponse> => {
  console.log('🏫 Fetching schools for:', lat, lon);
  
  // Use fallback coordinates if invalid
  let searchLat = lat || 12.9716;
  let searchLon = lon || 77.5946;
  
  // Try direct Geoapify API first
  try {
    console.log('🌐 Trying direct Geoapify API...');
    const directUrl = `https://api.geoapify.com/v2/places?categories=education.school&filter=circle:${searchLon},${searchLat},${radiusKm * 1000}&limit=20&apiKey=${GEOAPIFY_API_KEY}`;
    console.log('🔗 API URL:', directUrl);
    
    const response = await fetch(directUrl);
    console.log('📊 Response status:', response.status, response.statusText);
    
    if (response.ok) {
      const data = await response.json();
      console.log('📊 Direct API returned:', data.features?.length || 0, 'schools');
      console.log('📊 Full API response:', data);
      
      if (data.features && data.features.length > 0) {
        const schools = data.features.map((feature: any) => {
          const coords = feature.geometry.coordinates;
          const props = feature.properties;
          const distance = calculateDistance(searchLat, searchLon, coords[1], coords[0]);
          
          return {
            id: feature.properties.place_id || `school-${Math.random()}`,
            name: props.name || 'School',
            lat: coords[1],
            lon: coords[0],
            distance: Math.round(distance * 10) / 10,
            driveTime: estimateDriveTime(distance),
            address: buildGeoapifyAddress(props),
            schoolType: 'School'
          };
        });
        
        console.log('✅ Real schools from API:', schools.map((s: School) => s.name));
        return { schools, methodUsed: 'geoapify' };
      } else {
        console.log('⚠️ API returned no schools');
      }
    } else {
      console.log('❌ API response not OK:', response.status, response.statusText);
    }
  } catch (error) {
    console.log('❌ Direct API failed:', error);
    console.log('❌ Error details:', error instanceof Error ? error.message : 'Unknown error', error instanceof Error ? error.name : 'Error');
  }
  
  // Try backend proxy as fallback
  try {
    console.log('🌐 Trying backend proxy...');
    const backendUrl = `http://localhost:8090/api/schools/nearby?lat=${searchLat}&lon=${searchLon}&radius=${radiusKm}`;
    
    const response = await fetch(backendUrl);
    
    if (response.ok) {
      const data = await response.json();
      console.log('📊 Backend returned:', data.schools?.length || 0, 'schools');
      
      if (data.schools && data.schools.length > 0) {
        console.log('✅ Schools from backend:', data.schools.map((s: any) => s.name));
        return { schools: data.schools, methodUsed: 'geoapify' };
      }
    }
  } catch (error) {
    console.log('⚠️ Backend proxy failed:', error);
  }
  
  // Final fallback to mock data
  console.log('❌❌ ALL METHODS FAILED - Using mock data as final fallback');
  return generateMockSchools(searchLat, searchLon);
};

// Generate mock schools for testing when API fails
const generateMockSchools = (lat: number, lon: number): SchoolsResponse => {
  console.log('🏫 Generating mock schools for testing');
  
  const mockSchools: School[] = [
    {
      id: 'mock-1',
      name: 'Delhi Public School',
      lat: lat + 0.01,
      lon: lon + 0.01,
      distance: 1.2,
      driveTime: '3 min drive',
      address: 'Near Property Location',
      schoolType: 'Private School'
    },
    {
      id: 'mock-2', 
      name: 'Kendriya Vidyalaya',
      lat: lat - 0.015,
      lon: lon + 0.02,
      distance: 2.1,
      driveTime: '5 min drive',
      address: 'Government School Area',
      schoolType: 'Government School'
    },
    {
      id: 'mock-3',
      name: 'International School',
      lat: lat + 0.02,
      lon: lon - 0.01,
      distance: 2.8,
      driveTime: '7 min drive', 
      address: 'International Education Hub',
      schoolType: 'International School'
    },
    {
      id: 'mock-4',
      name: 'St. Mary\'s Convent School',
      lat: lat - 0.01,
      lon: lon - 0.02,
      distance: 3.5,
      driveTime: '9 min drive',
      address: 'Convent Road',
      schoolType: 'Convent School'
    },
    {
      id: 'mock-5',
      name: 'Ryan International School',
      lat: lat + 0.025,
      lon: lon + 0.015,
      distance: 4.2,
      driveTime: '11 min drive',
      address: 'Education District',
      schoolType: 'Private School'
    }
  ];
  
  console.log('✅ Generated', mockSchools.length, 'mock schools for testing');
  return { schools: mockSchools, methodUsed: 'geoapify' };
};

const buildGeoapifyAddress = (props: any): string => {
  const parts = [];
  if (props.housenumber) parts.push(props.housenumber);
  if (props.street) parts.push(props.street);
  if (props.suburb) parts.push(props.suburb);
  if (props.city) parts.push(props.city);
  if (props.state) parts.push(props.state);
  if (props.postcode) parts.push(props.postcode);
  if (props.country) parts.push(props.country);
  
  return parts.length > 0 ? parts.join(', ') : props.formatted || 'Address not available';
};

const calculateDistance = (lat1: number, lon1: number, lat2: number, lon2: number): number => {
  const R = 6371;
  const dLat = (lat2 - lat1) * Math.PI / 180;
  const dLon = (lon2 - lon1) * Math.PI / 180;
  const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
    Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
    Math.sin(dLon/2) * Math.sin(dLon/2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
  return R * c;
};

const estimateDriveTime = (distanceKm: number): string => {
  const avgSpeedKmh = 25;
  const timeHours = distanceKm / avgSpeedKmh;
  const timeMinutes = Math.round(timeHours * 60);
  return `${timeMinutes} min drive`;
};