// Title Search API Service
import { API_BASE_URL } from '../config/api';

export interface TitleSearchRequest {
  property_id: string;
  survey_number?: string;
  owner_name?: string;
  address?: string;
  search_type: 'basic' | 'comprehensive';
}

export interface PropertyHistoryRequest {
  property_id: string;
  include_disputes?: boolean;
  include_encumbrances?: boolean;
  timeline_years?: number;
}

export interface TitleSearchResult {
  success: boolean;
  search_id: string;
  title_integrity_score: number;
  risk_level: 'low' | 'medium' | 'high' | 'critical';
  title_status: 'clear' | 'disputed' | 'encumbered' | 'pending';
  report_data: any;
}

export interface PropertyHistoryResult {
  success: boolean;
  property_id: string;
  ownership_transfers: OwnershipTransfer[];
  transaction_history: Transaction[];
  legal_disputes: LegalDispute[];
  verification_metrics: VerificationMetrics;
  timeline_analysis: TimelineAnalysis;
}

export interface OwnershipTransfer {
  document_id: string;
  document_type: string;
  transfer_date: string;
  from_owner: string;
  to_owner: string;
  property_identifiers: string[];
  survey_numbers: string[];
  verification_status: string;
  authenticity_score: number;
}

export interface Transaction {
  document_id: string;
  document_type: string;
  transaction_date: string;
  registration_numbers: string[];
  transaction_type: string;
  compliance_status: string;
  verification_score: number;
}

export interface LegalDispute {
  document_id: string;
  document_type: string;
  court_references: string[];
  legal_notices: string[];
  dispute_status: string;
  identified_date: string;
  severity: 'low' | 'medium' | 'high';
}

export interface VerificationMetrics {
  records_verified: number;
  total_records: number;
  verification_percentage: number;
  ownership_transfers_verified: number;
  transactions_verified: number;
  legal_disputes_count: number;
}

export interface TimelineAnalysis {
  total_transfers: number;
  total_transactions: number;
  timeline_span_years: number;
  ownership_gaps: OwnershipGap[];
  transaction_patterns: TransactionPatterns;
  risk_indicators: string[];
}

export interface OwnershipGap {
  gap_type: 'ownership_mismatch' | 'time_gap';
  current_owner?: string;
  expected_owner?: string;
  gap_days?: number;
  gap_years?: number;
  gap_position: number;
  severity: 'low' | 'medium' | 'high';
}

export interface TransactionPatterns {
  frequent_transactions: boolean;
  recent_activity: boolean;
  compliance_issues: number;
  registration_consistency: boolean;
}

class TitleSearchService {
  private baseUrl = `${API_BASE_URL}/title-search`;

  private getAuthHeaders(): HeadersInit {
    const token = 
      localStorage.getItem('access_token') ||
      localStorage.getItem('token') ||
      localStorage.getItem('auth_token');
    
    // Check for buyer authentication first (access_token)
    const accessToken = localStorage.getItem('access_token');
    if (accessToken) {
      return {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${accessToken}`
      };
    }

    // Fallback to auth_token for backward compatibility
    const authToken = localStorage.getItem('auth_token');
    if (authToken) {
      return {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${authToken}`
      };
    }

    // Check for seller authentication
    const sellerToken = localStorage.getItem('sellerToken');
    if (sellerToken) {
      return {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${sellerToken}`
      };
    }

    // Check for agent authentication
    const agentToken = localStorage.getItem('agentToken');
    if (agentToken) {
      return {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${agentToken}`
      };
    }

    return {
      'Content-Type': 'application/json'
    };
  }

  async initiateTitleSearch(request: TitleSearchRequest): Promise<{ success: boolean; search_id: string; status: string; message: string }> {
    try {
      // Debug logging
      const token = localStorage.getItem('access_token') || localStorage.getItem('token') || localStorage.getItem('auth_token');
      console.log('🔍 Debug Info:');
      console.log('- API URL:', `${this.baseUrl}/initiate`);
      console.log('- Token found:', !!token);
      console.log('- Token value:', token ? token.substring(0, 20) + '...' : 'null');
      console.log('- Request data:', request);
      
      const response = await fetch(`${this.baseUrl}/initiate`, {
        method: 'POST',
        headers: this.getAuthHeaders(),
        body: JSON.stringify(request)
      });

      console.log('- Response status:', response.status);
      console.log('- Response ok:', response.ok);

      if (!response.ok) {
        const errorText = await response.text();
        console.log('- Error response:', errorText);
        throw new Error(`HTTP error! status: ${response.status}, body: ${errorText}`);
      }

      const result = await response.json();
      console.log('- Success result:', result);
      return result;
    } catch (error) {
      console.error('❌ Title search failed:', error);
      throw error;
    }
  }

  async uploadTitleDocuments(searchId: string, sessionId: string): Promise<any> {
    try {
      const formData = new FormData();
      formData.append('session_id', sessionId);

      const authHeaders = this.getAuthHeaders() as Record<string, string>;
      // Let the browser set multipart boundary for FormData.
      if (authHeaders['Content-Type']) {
        delete authHeaders['Content-Type'];
      }

      const response = await fetch(`${this.baseUrl}/upload-documents/${searchId}`, {
        method: 'POST',
        headers: {
          ...authHeaders,
          'Authorization': `Bearer ${localStorage.getItem('access_token') || localStorage.getItem('token') || localStorage.getItem('auth_token')}`
        },
        body: formData
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Failed to upload title documents:', error);
      throw error;
    }
  }

  async performTitleAnalysis(searchId: string): Promise<TitleSearchResult> {
    try {
      const response = await fetch(`${this.baseUrl}/analyze/${searchId}`, {
        method: 'POST',
        headers: this.getAuthHeaders()
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Failed to perform title analysis:', error);
      throw error;
    }
  }

  async getTitleSearchResults(searchId: string): Promise<TitleSearchResult> {
    try {
      const response = await fetch(`${this.baseUrl}/results/${searchId}`, {
        method: 'GET',
        headers: this.getAuthHeaders()
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Failed to get title search results:', error);
      throw error;
    }
  }

  async analyzePropertyHistory(request: PropertyHistoryRequest): Promise<PropertyHistoryResult> {
    try {
      const response = await fetch(`${this.baseUrl}/property-history/analyze`, {
        method: 'POST',
        headers: this.getAuthHeaders(),
        body: JSON.stringify(request)
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Failed to analyze property history:', error);
      throw error;
    }
  }

  async getPropertyHistory(propertyId: string): Promise<PropertyHistoryResult> {
    try {
      const response = await fetch(`${this.baseUrl}/property-history/${propertyId}`, {
        method: 'GET',
        headers: this.getAuthHeaders()
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Failed to get property history:', error);
      throw error;
    }
  }

  async instantTitleVerification(
    propertyId: string,
    titleDeed: File,
    saleDeed: File,
    encumbranceCert?: File
  ): Promise<any> {
    try {
      const formData = new FormData();
      formData.append('property_id', propertyId);
      formData.append('title_deed', titleDeed);
      formData.append('sale_deed', saleDeed);
      
      if (encumbranceCert) {
        formData.append('encumbrance_cert', encumbranceCert);
      }

      const authHeaders = this.getAuthHeaders() as Record<string, string>;
      if (authHeaders['Content-Type']) {
        delete authHeaders['Content-Type'];
      }

      const response = await fetch(`${this.baseUrl}/instant-verification`, {
        method: 'POST',
        headers: {
          ...authHeaders,
          'Authorization': `Bearer ${localStorage.getItem('access_token') || localStorage.getItem('token') || localStorage.getItem('auth_token')}`
        },
        body: formData
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Failed to perform instant title verification:', error);
      throw error;
    }
  }

  async instantTitleVerificationBySession(sessionId: string): Promise<any> {
    try {
      const formData = new FormData();
      formData.append('session_id', sessionId);
      
      // Get user_id from localStorage for all user types
      const userStr = localStorage.getItem('user');
      const sellerId = localStorage.getItem('sellerId');
      const agentId = localStorage.getItem('agentId');
      
      let userId = 'guest';
      if (userStr) {
        try {
          const user = JSON.parse(userStr);
          userId = user?.user_id || user?.id || 'guest';
        } catch (e) {
          console.error('Error parsing user data:', e);
        }
      } else if (sellerId) {
        userId = sellerId;
      } else if (agentId) {
        userId = agentId;
      }
      
      formData.append('user_id', userId);

      const authHeaders = this.getAuthHeaders() as Record<string, string>;
      if (authHeaders['Content-Type']) {
        delete authHeaders['Content-Type'];
      }

      const response = await fetch(`${this.baseUrl}/instant-verification`, {
        method: 'POST',
        headers: {
          ...authHeaders,
          ...(localStorage.getItem('access_token') ||
            localStorage.getItem('token') ||
            localStorage.getItem('auth_token'))
            ? {
                'Authorization': `Bearer ${localStorage.getItem('access_token') || localStorage.getItem('token') || localStorage.getItem('auth_token')}`
              }
            : {}
        },
        body: formData
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.detail || `HTTP error! status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Failed to perform instant title verification by session:', error);
      throw error;
    }
  }

  async getUserTitleSearches(): Promise<any> {
    try {
      const response = await fetch(`${this.baseUrl}/search-history`, {
        method: 'GET',
        headers: this.getAuthHeaders()
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Failed to get user title searches:', error);
      throw error;
    }
  }

  // Utility methods for UI components
  getRiskLevelColor(riskLevel: string): string {
    switch (riskLevel) {
      case 'low': return 'text-green-600 bg-green-50';
      case 'medium': return 'text-yellow-600 bg-yellow-50';
      case 'high': return 'text-orange-600 bg-orange-50';
      case 'critical': return 'text-red-600 bg-red-50';
      default: return 'text-gray-600 bg-gray-50';
    }
  }

  getTitleStatusColor(titleStatus: string): string {
    switch (titleStatus) {
      case 'clear': return 'text-green-600 bg-green-50';
      case 'pending': return 'text-yellow-600 bg-yellow-50';
      case 'encumbered': return 'text-orange-600 bg-orange-50';
      case 'disputed': return 'text-red-600 bg-red-50';
      default: return 'text-gray-600 bg-gray-50';
    }
  }

  getVerificationStatusIcon(status: string): string {
    switch (status) {
      case 'completed': return '✅';
      case 'pending': return '⏳';
      case 'failed': return '❌';
      default: return '❓';
    }
  }

  formatScore(score: number): string {
    return `${score}%`;
  }

  formatDate(dateString: string): string {
    try {
      return new Date(dateString).toLocaleDateString('en-IN', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
      });
    } catch {
      return dateString;
    }
  }

  generateRecommendation(integrityScore: number, riskLevel: string): string {
    if (integrityScore >= 85) {
      return "Excellent title integrity. Proceed with confidence - all documents verified successfully with minimal legal risks.";
    } else if (integrityScore >= 70) {
      return "Good title integrity with minor issues. Consider additional verification for complete assurance before proceeding.";
    } else if (integrityScore >= 50) {
      return "Moderate title integrity with several concerns. Professional legal review strongly recommended before transaction.";
    } else {
      return "Poor title integrity with significant issues. Do not proceed without comprehensive legal consultation and remediation.";
    }
  }
}

export const titleSearchService = new TitleSearchService();
