import { ValuationRequest, ValuationResult, Comparable } from '../types/valuation';

const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

const getSessionId = (): string => {
  let sessionId = sessionStorage.getItem('valuation_session_id');
  if (!sessionId) {
    sessionId = `session_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    sessionStorage.setItem('valuation_session_id', sessionId);
  }
  return sessionId;
};

export const valuationApi = {
  // Calculate valuation using backend API
  calculateValuation: async (data: ValuationRequest): Promise<{ success: boolean; data?: any; error?: string }> => {
    try {
      const token = localStorage.getItem('sellerToken');
      const headers: Record<string, string> = {
        'Content-Type': 'application/json',
        'X-Session-Id': getSessionId()
      };
      
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }
      
      const response = await fetch(`${API_BASE_URL}/valuations/calculate`, {
        method: 'POST',
        headers,
        body: JSON.stringify(data)
      });
      
      const result = await response.json();
      return result;
    } catch (error) {
      console.error('Valuation API error:', error);
      return { success: false, error: 'Failed to calculate valuation' };
    }
  },

  // Get valuation result by ID
  getValuation: async (requestId: string): Promise<ValuationResult | null> => {
    try {
      const response = await fetch(`${API_BASE_URL}/valuations/${requestId}`);
      const result = await response.json();
      
      if (result.success && result.data) {
        return result.data;
      }
      return null;
    } catch (error) {
      console.error('Get valuation error:', error);
      return null;
    }
  },

  // Get user's valuation history
  getHistory: async (page: number = 1, limit: number = 20): Promise<any> => {
    try {
      const token = localStorage.getItem('access_token');
      if (!token) {
        return { valuations: [], total_count: 0, page, limit };
      }
      
      const response = await fetch(`${API_BASE_URL}/valuations/history/me?page=${page}&limit=${limit}`, {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });
      
      return await response.json();
    } catch (error) {
      console.error('Get history error:', error);
      return { valuations: [], total_count: 0, page, limit };
    }
  }
};

export default valuationApi;