interface PropertyData {
  propertyType: 'apartment' | 'villa' | 'plot' | 'commercial';
  location: string;
  area: number;
  bedrooms: string;
  bathrooms?: number;
  age?: number;
  floor?: number;
  totalFloors?: number;
  furnishing?: string;
  facing?: string;
  amenities?: string[];
}

interface ValuationResult {
  estimatedValue: number;
  pricePerSqft: number;
  confidence: number;
  range: { min: number; max: number };
  methodology: string;
  comparables: Array<{
    address: string;
    price: number;
    pricePerSqft: number;
    distance: number;
  }>;
  marketTrends: {
    appreciation: number;
    rentalYield: number;
    demandSupply: 'high' | 'medium' | 'low';
  };
}

// Comprehensive market data - Updated with latest rates
const CITY_RATES: Record<string, Record<string, number>> = {
  bangalore: {
    koramangala: 13500, indiranagar: 12800, whitefield: 9200, electronic_city: 7800,
    hebbal: 9800, jayanagar: 11200, btm_layout: 10500, marathahalli: 8800,
    sarjapur: 8200, yelahanka: 7200, hsr_layout: 12500, jp_nagar: 10800,
    banashankari: 9500, rajajinagar: 10200, malleshwaram: 11500, bellandur: 9800,
    kormangala: 13500, mg_road: 15000, brigade_road: 14500, cunningham_road: 13800,
    gkvk: 6500
  },
  mumbai: {
    bandra: 38000, andheri: 27000, powai: 24000, thane: 16500, navi_mumbai: 13500,
    malad: 19500, goregaon: 22000, kandivali: 17500, borivali: 15500, vikhroli: 20500,
    juhu: 42000, worli: 45000, lower_parel: 35000, dadar: 28000, kurla: 18000,
    mulund: 16000, ghatkopar: 19000, chembur: 21000, santacruz: 26000, khar: 36000
  },
  delhi: {
    gurgaon: 13500, noida: 9000, dwarka: 11000, rohini: 10000, lajpat_nagar: 16500,
    south_delhi: 28000, cp: 35000, karol_bagh: 19500, janakpuri: 12000, vasant_kunj: 22000,
    greater_noida: 7500, faridabad: 8500, saket: 24000, hauz_khas: 26000, nehru_place: 18000,
    pitampura: 11500, mayur_vihar: 10500, laxmi_nagar: 9500, rajouri_garden: 14000, punjabi_bagh: 13000
  },
  pune: {
    koregaon_park: 11000, hinjewadi: 7500, wakad: 7000, baner: 8500, aundh: 9000,
    kharadi: 8000, viman_nagar: 9500, hadapsar: 7200, pimpri: 6500, chinchwad: 6800,
    magarpatta: 10500, kalyani_nagar: 10000, shivaji_nagar: 12000, deccan: 11500, camp: 10800
  },
  hyderabad: {
    hitech_city: 9500, gachibowli: 9000, madhapur: 8800, kondapur: 8200, kukatpally: 7500,
    miyapur: 7000, secunderabad: 8500, banjara_hills: 12000, jubilee_hills: 13500, begumpet: 9200,
    ameerpet: 8000, lb_nagar: 6800, uppal: 7200, kompally: 7800, manikonda: 8500
  },
  chennai: {
    anna_nagar: 9000, t_nagar: 10500, adyar: 11000, velachery: 8500, tambaram: 6500,
    omr: 8000, ecr: 9500, porur: 7800, chrompet: 7000, perungudi: 8200,
    mylapore: 10000, nungambakkam: 11500, guindy: 8800, sholinganallur: 8500, pallavaram: 7200
  },
  kolkata: {
    salt_lake: 7500, new_town: 7000, rajarhat: 6800, ballygunge: 9000, alipore: 10500,
    park_street: 11000, em_bypass: 7200, howrah: 6000, dum_dum: 6500, behala: 6200,
    jadavpur: 7800, garia: 6800, barasat: 5800, baranagar: 6500, tollygunge: 8000
  },
  ahmedabad: {
    sg_highway: 6500, satellite: 7000, vastrapur: 7500, bodakdev: 8000, prahlad_nagar: 7200,
    chandkheda: 5500, maninagar: 6000, naranpura: 6800, thaltej: 7800, bopal: 6200,
    gota: 5800, science_city: 6500, ambawadi: 7500, navrangpura: 8500, ellis_bridge: 9000
  }
};

const PROPERTY_MULTIPLIERS = {
  apartment: 1.0,
  villa: 1.4,
  plot: 0.65,
  commercial: 2.0,
  penthouse: 1.6,
  studio: 0.85,
  duplex: 1.25,
  farmhouse: 1.5,
  warehouse: 1.3,
  office: 1.9,
  retail: 2.1,
  industrial: 1.2
};

const BEDROOM_MULTIPLIERS = {
  '0': 0.75,  // Studio
  '1': 0.88,
  '2': 1.0,
  '3': 1.18,
  '4': 1.42,
  '5': 1.65,
  '6': 1.85
};

const BATHROOM_MULTIPLIERS = {
  1: 0.95,
  2: 1.0,
  3: 1.08,
  4: 1.15,
  5: 1.20
};

const AGE_MULTIPLIERS = {
  0: 1.15,   // New construction
  3: 1.05,   // 3-5 years
  7: 1.0,    // 5-10 years (base)
  12: 0.92,  // 10-15 years
  20: 0.85   // 15+ years
};

const FLOOR_MULTIPLIERS = {
  0: 0.95,   // Ground floor
  1: 0.98,   // 1st floor
  2: 1.0,    // 2nd floor (base)
  3: 1.02,   // 3rd floor
  4: 1.03,   // 4th floor
  5: 1.04,   // 5th floor
  6: 1.05,   // 6th floor
  7: 1.05,   // 7th floor
  8: 1.04,   // 8th floor
  9: 1.03,   // 9th floor
  10: 1.02   // 10th+ floor
};

const FURNISHING_MULTIPLIERS = {
  'unfurnished': 1.0,
  'semi_furnished': 1.08,
  'fully_furnished': 1.15
};

const FACING_MULTIPLIERS = {
  'north': 1.05,
  'northeast': 1.08,
  'east': 1.03,
  'southeast': 1.02,
  'south': 1.0,
  'southwest': 0.98,
  'west': 0.96,
  'northwest': 1.01
};

const AMENITY_VALUES = {
  'Parking': 0.08,
  'Gym': 0.03,
  'Swimming Pool': 0.05,
  'Security': 0.04,
  'Power Backup': 0.02,
  'Lift': 0.06,
  'Garden': 0.03,
  'Club House': 0.04,
  'Play Area': 0.02,
  'Intercom': 0.01,
  'Gas Pipeline': 0.02,
  'Water Supply': 0.02,
  'Maintenance Staff': 0.02
};

class ValuationEngine {
  private normalizeLocation(location: string): { city: string; area: string } {
    const normalized = location.toLowerCase().replace(/[^a-z\s]/g, '').trim();
    
    // First, try to find area match across all cities (more specific)
    for (const [city, areas] of Object.entries(CITY_RATES)) {
      for (const area of Object.keys(areas)) {
        const areaName = area.replace(/_/g, ' ');
        // Check for exact match or if normalized contains the area name
        if (normalized.includes(areaName) || areaName.includes(normalized.split(' ')[0])) {
          return { city, area };
        }
      }
    }
    
    // Then try city match
    for (const [city, areas] of Object.entries(CITY_RATES)) {
      if (normalized.includes(city)) {
        return { city, area: Object.keys(areas)[0] };
      }
    }
    
    // Default fallback
    return { city: 'bangalore', area: 'koramangala' };
  }

  private calculateBasePrice(propertyData: PropertyData): number {
    const { city, area } = this.normalizeLocation(propertyData.location);
    const baseRate = CITY_RATES[city]?.[area] || 7000;
    
    const propertyMultiplier = PROPERTY_MULTIPLIERS[propertyData.propertyType] || 1.0;
    const bedroomMultiplier = BEDROOM_MULTIPLIERS[propertyData.bedrooms as keyof typeof BEDROOM_MULTIPLIERS] || 1.0;
    
    // Additional multipliers
    const bathroomMultiplier = propertyData.bathrooms ? (BATHROOM_MULTIPLIERS[propertyData.bathrooms] || 1.0) : 1.0;
    const ageMultiplier = propertyData.age !== undefined ? (AGE_MULTIPLIERS[propertyData.age as keyof typeof AGE_MULTIPLIERS] || 1.0) : 1.0;
    const floorMultiplier = propertyData.floor !== undefined ? (FLOOR_MULTIPLIERS[Math.min(propertyData.floor, 10)] || 1.0) : 1.0;
    const furnishingMultiplier = propertyData.furnishing ? (FURNISHING_MULTIPLIERS[propertyData.furnishing as keyof typeof FURNISHING_MULTIPLIERS] || 1.0) : 1.0;
    const facingMultiplier = propertyData.facing ? (FACING_MULTIPLIERS[propertyData.facing as keyof typeof FACING_MULTIPLIERS] || 1.0) : 1.0;
    
    // Amenities bonus (additive)
    let amenityBonus = 0;
    if (propertyData.amenities && propertyData.amenities.length > 0) {
      amenityBonus = propertyData.amenities.reduce((total, amenity) => {
        return total + (AMENITY_VALUES[amenity as keyof typeof AMENITY_VALUES] || 0);
      }, 0);
      amenityBonus = Math.min(amenityBonus, 0.25); // Cap at 25% bonus
    }
    
    return baseRate * propertyMultiplier * bedroomMultiplier * bathroomMultiplier * ageMultiplier * floorMultiplier * furnishingMultiplier * facingMultiplier * (1 + amenityBonus);
  }

  private generateComparables(location: string, pricePerSqft: number, propertyType: string): Array<{
    address: string;
    price: number;
    pricePerSqft: number;
    distance: number;
  }> {
    const comparableLabels = ['Comparable 1', 'Comparable 2', 'Comparable 3'];
    const distances = [0.5, 0.8, 1.1];
    
    return comparableLabels.map((label, i) => {
      const priceVariance = (i === 0 ? -0.08 : i === 1 ? 0.00 : 0.075);
      const adjustedPricePerSqft = Math.round(pricePerSqft * (1 + priceVariance));
      
      return {
        address: `${location} - ${label}`,
        price: adjustedPricePerSqft * 1000,
        pricePerSqft: adjustedPricePerSqft,
        distance: distances[i]
      };
    });
  }

  private calculateConfidence(propertyData: PropertyData): number {
    const { city, area } = this.normalizeLocation(propertyData.location);
    let confidence = 65; // Base confidence
    
    // Tier 1 cities: +20%
    if (['bangalore', 'mumbai', 'delhi', 'pune', 'hyderabad', 'chennai'].includes(city)) {
      confidence += 20;
    }
    // Tier 2 cities: +12%
    else if (['kolkata', 'ahmedabad', 'jaipur', 'lucknow', 'kochi'].includes(city)) {
      confidence += 12;
    }
    // Other cities: +5%
    else {
      confidence += 5;
    }
    
    // Known area within city: +8%
    if (CITY_RATES[city]?.[area]) confidence += 8;
    
    // Standard property types: +10%
    if (['apartment', 'villa', 'commercial'].includes(propertyData.propertyType)) {
      confidence += 10;
    }
    
    // Optimal area range: +7%
    if (propertyData.area >= 600 && propertyData.area <= 2500) {
      confidence += 7;
    }
    // Acceptable range: +3%
    else if (propertyData.area >= 400 && propertyData.area <= 5000) {
      confidence += 3;
    }
    
    // Standard bedroom count: +5%
    if (['2', '3', '4'].includes(propertyData.bedrooms)) {
      confidence += 5;
    }
    
    // Additional details boost confidence
    if (propertyData.bathrooms) confidence += 2;
    if (propertyData.age !== undefined) confidence += 3;
    if (propertyData.floor !== undefined) confidence += 2;
    if (propertyData.furnishing) confidence += 2;
    if (propertyData.facing) confidence += 2;
    if (propertyData.amenities && propertyData.amenities.length > 0) {
      confidence += Math.min(propertyData.amenities.length * 0.5, 4); // Max 4% for amenities
    }
    
    return Math.min(confidence, 95); // Cap at 95%
  }

  async calculateValuation(propertyData: PropertyData): Promise<ValuationResult> {
    // Input validation
    if (!propertyData.location?.trim()) {
      throw new Error('Property location is required for valuation');
    }
    
    if (!propertyData.area || propertyData.area <= 0) {
      throw new Error('Valid property area is required for valuation');
    }
    
    if (propertyData.area > 50000) {
      throw new Error('Property area seems unusually large. Please verify the input.');
    }
    
    try {
      // Simulate API delay with timeout
      await Promise.race([
        new Promise(resolve => setTimeout(resolve, 1500)),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error('Valuation timeout')), 10000)
        )
      ]);
      
      const pricePerSqft = this.calculateBasePrice(propertyData);
      
      if (pricePerSqft <= 0) {
        throw new Error('Unable to determine market rates for this location');
      }
      
      const estimatedValue = Math.round(pricePerSqft * propertyData.area);
      const confidence = this.calculateConfidence(propertyData);
      
      // Calculate range based on confidence (tighter ranges for higher confidence)
      let varianceFactor = 0.25; // Base 25%
      if (confidence >= 85) varianceFactor = 0.12; // ±12% for high confidence
      else if (confidence >= 75) varianceFactor = 0.18; // ±18% for good confidence
      else if (confidence >= 65) varianceFactor = 0.22; // ±22% for medium confidence
      
      const variance = (100 - confidence) / 100 * varianceFactor;
      const minValue = Math.round(estimatedValue * (1 - variance));
      const maxValue = Math.round(estimatedValue * (1 + variance));
      
      // Format location for display (capitalize first letter of each word)
      const displayLocation = propertyData.location
        .split(/[,\s]+/)
        .filter(word => word.trim())
        .map(word => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
        .join(', ');
      
      const comparables = this.generateComparables(displayLocation, pricePerSqft, propertyData.propertyType);
      console.log('Generated comparables:', comparables);
      console.log('Price per sqft:', pricePerSqft);
      console.log('Estimated value:', estimatedValue);
      
      // Calculate market trends based on city tier and property type
      const { city } = this.normalizeLocation(propertyData.location);
      
      // Appreciation rates by city tier
      let baseAppreciation = 6;
      if (['bangalore', 'mumbai', 'pune', 'hyderabad'].includes(city)) {
        baseAppreciation = 8.5; // Tier 1 IT hubs
      } else if (['delhi', 'chennai', 'kolkata'].includes(city)) {
        baseAppreciation = 7.2; // Tier 1 metros
      } else if (['ahmedabad', 'jaipur', 'kochi'].includes(city)) {
        baseAppreciation = 6.5; // Tier 2
      }
      
      const appreciation = Math.max(3, Math.min(18, Math.round((baseAppreciation + Math.random() * 4 - 2) * 10) / 10));
      
      // Rental yield by property type
      let baseYield = 3.5;
      if (propertyData.propertyType === 'commercial') baseYield = 6.5;
      else if (propertyData.propertyType === 'plot') baseYield = 0;
      else if (['apartment', 'villa'].includes(propertyData.propertyType)) baseYield = 3.2;
      
      const rentalYield = propertyData.propertyType === 'plot' ? 0 : 
        Math.max(1.5, Math.min(9, Math.round((baseYield + Math.random() * 2 - 1) * 10) / 10));
      
      // Demand-supply based on confidence and city
      let demandSupply: 'high' | 'medium' | 'low';
      if (confidence >= 85 && ['bangalore', 'mumbai', 'pune'].includes(city)) {
        demandSupply = 'high';
      } else if (confidence >= 75) {
        demandSupply = 'medium';
      } else {
        demandSupply = 'low';
      }
      
      return {
        estimatedValue,
        pricePerSqft,
        confidence,
        range: { min: minValue, max: maxValue },
        methodology: 'Comparative Market Analysis with location-based pricing models',
        comparables,
        marketTrends: {
          appreciation,
          rentalYield,
          demandSupply: demandSupply as 'high' | 'medium' | 'low'
        }
      };
    } catch (error) {
      if (error instanceof Error) {
        // Re-throw validation errors as-is
        if (error.message.includes('required') || 
            error.message.includes('location') || 
            error.message.includes('area')) {
          throw error;
        }
        
        // Handle specific error types
        if (error.message.includes('timeout')) {
          throw new Error('Valuation service is taking too long. Please try again.');
        }
      }
      
      // Generic error for unexpected issues
      throw new Error('Valuation calculation failed. Our team has been notified.');
    }
  }
}

export const valuationEngine = new ValuationEngine();
export type { PropertyData, ValuationResult };

// Export validation utilities
export const validatePropertyData = (data: Partial<PropertyData>): string[] => {
  const errors: string[] = [];
  
  if (!data.propertyType) errors.push('Property type is required');
  if (!data.location?.trim()) errors.push('Location is required');
  if (!data.area || data.area <= 0) errors.push('Valid area is required');
  if (data.area && data.area > 50000) errors.push('Area seems unusually large');
  if (!data.bedrooms) errors.push('Bedrooms selection is required');
  
  return errors;
};