import React, { useState, useEffect } from 'react';
import { 
  FaHeartbeat, FaMicrochip, FaMemory, FaHdd, FaDatabase, FaCheckCircle, 
  FaExclamationTriangle, FaChartLine, FaServer, FaList, FaShieldAlt,
  FaBell, FaEye, FaDownload, FaSyncAlt, FaClock, FaWifi, FaThermometerHalf
} from 'react-icons/fa';

const ComprehensiveSystemHealth: React.FC = () => {
  const [activeTab, setActiveTab] = useState('overview');
  const [currentTime, setCurrentTime] = useState(new Date());
  const [metrics, setMetrics] = useState({
    cpu: 0, memory: 0, disk: 0, network: 0, temperature: 0
  });
  const [alerts, setAlerts] = useState([
    { id: 1, type: 'warning', message: 'High CPU usage detected', time: '2 min ago', severity: 'medium' },
    { id: 2, type: 'info', message: 'System backup completed', time: '15 min ago', severity: 'low' },
    { id: 3, type: 'error', message: 'Database connection timeout', time: '1 hour ago', severity: 'high' }
  ]);
  const [logs, setLogs] = useState([
    { id: 1, level: 'INFO', message: 'User authentication successful', timestamp: '2024-01-15 23:30:15', source: 'auth-service' },
    { id: 2, level: 'WARN', message: 'Memory usage above 80%', timestamp: '2024-01-15 23:28:42', source: 'system-monitor' },
    { id: 3, level: 'ERROR', message: 'Failed to connect to external API', timestamp: '2024-01-15 23:25:10', source: 'api-gateway' },
    { id: 4, level: 'INFO', message: 'Scheduled backup initiated', timestamp: '2024-01-15 23:20:00', source: 'backup-service' }
  ]);
  const [services, setServices] = useState([
    { name: 'Database', status: 'healthy', responseTime: 12, uptime: '99.9%' },
    { name: 'API Gateway', status: 'healthy', responseTime: 8, uptime: '99.8%' },
    { name: 'Cache Server', status: 'degraded', responseTime: 45, uptime: '98.5%' },
    { name: 'File Storage', status: 'healthy', responseTime: 15, uptime: '99.7%' }
  ]);

  useEffect(() => {
    const timer = setInterval(() => {
      setCurrentTime(new Date());
      setMetrics({
        cpu: Math.floor(Math.random() * 40) + 20,
        memory: Math.floor(Math.random() * 50) + 30,
        disk: Math.floor(Math.random() * 30) + 15,
        network: Math.floor(Math.random() * 100) + 50,
        temperature: Math.floor(Math.random() * 20) + 35
      });
    }, 3000);
    return () => clearInterval(timer);
  }, []);

  const getStatusColor = (value: number, thresholds = { warning: 70, critical: 90 }) => {
    if (value < thresholds.warning) return '#27ae60';
    if (value < thresholds.critical) return '#f39c12';
    return '#e74c3c';
  };

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case 'high': return '#e74c3c';
      case 'medium': return '#f39c12';
      case 'low': return '#3498db';
      default: return '#95a5a6';
    }
  };

  const getLogLevelColor = (level: string) => {
    switch (level) {
      case 'ERROR': return '#e74c3c';
      case 'WARN': return '#f39c12';
      case 'INFO': return '#3498db';
      default: return '#95a5a6';
    }
  };

  const renderOverview = () => (
    <div>
      {/* Metrics Grid */}
      <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(250px, 1fr))', gap: '20px', marginBottom: '30px' }}>
        {[
          { icon: FaMicrochip, label: 'CPU Usage', value: metrics.cpu, unit: '%', color: '#3498db' },
          { icon: FaMemory, label: 'Memory Usage', value: metrics.memory, unit: '%', color: '#e74c3c' },
          { icon: FaHdd, label: 'Disk Usage', value: metrics.disk, unit: '%', color: '#f39c12' },
          { icon: FaWifi, label: 'Network I/O', value: metrics.network, unit: 'MB/s', color: '#9b59b6' },
          { icon: FaThermometerHalf, label: 'Temperature', value: metrics.temperature, unit: '°C', color: '#e67e22' }
        ].map((metric, index) => (
          <div key={index} style={{ 
            padding: '20px', backgroundColor: '#f8f9fa', borderRadius: '12px', 
            border: '2px solid #e9ecef', position: 'relative', overflow: 'hidden'
          }}>
            <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', marginBottom: '15px' }}>
              <div style={{ display: 'flex', alignItems: 'center' }}>
                <metric.icon style={{ fontSize: '24px', color: metric.color, marginRight: '10px' }} />
                <span style={{ fontSize: '16px', fontWeight: '600', color: '#2c3e50' }}>{metric.label}</span>
              </div>
              <div style={{ 
                width: '12px', height: '12px', borderRadius: '50%', 
                backgroundColor: getStatusColor(metric.value)
              }} />
            </div>
            <div style={{ fontSize: '32px', fontWeight: 'bold', color: getStatusColor(metric.value) }}>
              {metric.value}{metric.unit}
            </div>
            <div style={{ 
              position: 'absolute', bottom: 0, left: 0, 
              width: `${metric.value}%`, height: '4px', 
              backgroundColor: getStatusColor(metric.value), 
              transition: 'width 0.5s ease'
            }} />
          </div>
        ))}
      </div>

      {/* Quick Stats */}
      <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(200px, 1fr))', gap: '15px' }}>
        <div style={{ padding: '15px', backgroundColor: '#d4edda', borderRadius: '8px', textAlign: 'center' }}>
          <div style={{ fontSize: '24px', fontWeight: 'bold', color: '#155724' }}>4/4</div>
          <div style={{ fontSize: '14px', color: '#155724' }}>Services Online</div>
        </div>
        <div style={{ padding: '15px', backgroundColor: '#fff3cd', borderRadius: '8px', textAlign: 'center' }}>
          <div style={{ fontSize: '24px', fontWeight: 'bold', color: '#856404' }}>{alerts.length}</div>
          <div style={{ fontSize: '14px', color: '#856404' }}>Active Alerts</div>
        </div>
        <div style={{ padding: '15px', backgroundColor: '#cce5ff', borderRadius: '8px', textAlign: 'center' }}>
          <div style={{ fontSize: '24px', fontWeight: 'bold', color: '#004085' }}>99.8%</div>
          <div style={{ fontSize: '14px', color: '#004085' }}>Uptime</div>
        </div>
        <div style={{ padding: '15px', backgroundColor: '#f8d7da', borderRadius: '8px', textAlign: 'center' }}>
          <div style={{ fontSize: '24px', fontWeight: 'bold', color: '#721c24' }}>15ms</div>
          <div style={{ fontSize: '14px', color: '#721c24' }}>Avg Response</div>
        </div>
      </div>
    </div>
  );

  const renderAlerts = () => (
    <div>
      <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '20px' }}>
        <h3 style={{ margin: 0, color: '#2c3e50' }}>System Alerts</h3>
        <button style={{ 
          padding: '8px 16px', backgroundColor: '#3498db', color: 'white', 
          border: 'none', borderRadius: '6px', cursor: 'pointer'
        }}>
          <FaSyncAlt style={{ marginRight: '5px' }} /> Refresh
        </button>
      </div>
      {alerts.map(alert => (
        <div key={alert.id} style={{ 
          padding: '15px', marginBottom: '10px', backgroundColor: 'white', 
          borderRadius: '8px', border: `2px solid ${getSeverityColor(alert.severity)}`,
          display: 'flex', alignItems: 'center', justifyContent: 'space-between'
        }}>
          <div style={{ display: 'flex', alignItems: 'center' }}>
            <div style={{ 
              width: '12px', height: '12px', borderRadius: '50%', 
              backgroundColor: getSeverityColor(alert.severity), marginRight: '15px'
            }} />
            <div>
              <div style={{ fontWeight: '600', color: '#2c3e50', marginBottom: '5px' }}>
                {alert.message}
              </div>
              <div style={{ fontSize: '12px', color: '#7f8c8d' }}>
                {alert.time} • {alert.severity.toUpperCase()}
              </div>
            </div>
          </div>
          <div style={{ display: 'flex', gap: '10px' }}>
            <button style={{ 
              padding: '5px 10px', backgroundColor: '#27ae60', color: 'white', 
              border: 'none', borderRadius: '4px', fontSize: '12px', cursor: 'pointer'
            }}>
              Acknowledge
            </button>
            <button style={{ 
              padding: '5px 10px', backgroundColor: '#e74c3c', color: 'white', 
              border: 'none', borderRadius: '4px', fontSize: '12px', cursor: 'pointer'
            }}>
              Dismiss
            </button>
          </div>
        </div>
      ))}
    </div>
  );

  const renderLogs = () => (
    <div>
      <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '20px' }}>
        <h3 style={{ margin: 0, color: '#2c3e50' }}>System Logs</h3>
        <div style={{ display: 'flex', gap: '10px' }}>
          <select style={{ padding: '5px 10px', borderRadius: '4px', border: '1px solid #ddd' }}>
            <option>All Levels</option>
            <option>ERROR</option>
            <option>WARN</option>
            <option>INFO</option>
          </select>
          <button style={{ 
            padding: '5px 10px', backgroundColor: '#27ae60', color: 'white', 
            border: 'none', borderRadius: '4px', cursor: 'pointer'
          }}>
            <FaDownload /> Export
          </button>
        </div>
      </div>
      <div style={{ backgroundColor: '#2c3e50', borderRadius: '8px', padding: '15px', fontFamily: 'monospace' }}>
        {logs.map(log => (
          <div key={log.id} style={{ 
            marginBottom: '8px', fontSize: '13px', display: 'flex', alignItems: 'center'
          }}>
            <span style={{ color: '#95a5a6', marginRight: '10px' }}>{log.timestamp}</span>
            <span style={{ 
              color: getLogLevelColor(log.level), fontWeight: 'bold', 
              marginRight: '10px', minWidth: '50px'
            }}>
              [{log.level}]
            </span>
            <span style={{ color: '#ecf0f1', marginRight: '10px' }}>{log.source}:</span>
            <span style={{ color: '#bdc3c7' }}>{log.message}</span>
          </div>
        ))}
      </div>
    </div>
  );

  const renderServices = () => (
    <div>
      <h3 style={{ margin: '0 0 20px 0', color: '#2c3e50' }}>Service Health Checks</h3>
      <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(300px, 1fr))', gap: '15px' }}>
        {services.map((service, index) => (
          <div key={index} style={{ 
            padding: '20px', backgroundColor: 'white', borderRadius: '8px', 
            border: '2px solid #e9ecef', display: 'flex', alignItems: 'center', justifyContent: 'space-between'
          }}>
            <div style={{ display: 'flex', alignItems: 'center' }}>
              <div style={{ 
                width: '16px', height: '16px', borderRadius: '50%', marginRight: '15px',
                backgroundColor: service.status === 'healthy' ? '#27ae60' : '#f39c12'
              }} />
              <div>
                <div style={{ fontWeight: '600', color: '#2c3e50', marginBottom: '5px' }}>
                  {service.name}
                </div>
                <div style={{ fontSize: '12px', color: '#7f8c8d' }}>
                  {service.responseTime}ms • {service.uptime} uptime
                </div>
              </div>
            </div>
            <div style={{ 
              padding: '5px 10px', borderRadius: '15px', fontSize: '12px', fontWeight: 'bold',
              backgroundColor: service.status === 'healthy' ? '#d4edda' : '#fff3cd',
              color: service.status === 'healthy' ? '#155724' : '#856404'
            }}>
              {service.status.toUpperCase()}
            </div>
          </div>
        ))}
      </div>
    </div>
  );

  return (
    <div style={{ padding: '24px', backgroundColor: '#f8f9fa', minHeight: '100vh' }}>
      {/* Header */}
      <div style={{ 
        display: 'flex', alignItems: 'center', justifyContent: 'space-between',
        marginBottom: '30px', padding: '20px', backgroundColor: 'white', 
        borderRadius: '12px', boxShadow: '0 2px 10px rgba(0,0,0,0.1)'
      }}>
        <div style={{ display: 'flex', alignItems: 'center' }}>
          <FaHeartbeat style={{ 
            color: '#27ae60', marginRight: '15px', fontSize: '36px',
            animation: 'pulse 2s infinite'
          }} />
          <div>
            <h1 style={{ margin: 0, color: '#2c3e50', fontSize: '32px', fontWeight: 'bold' }}>
              System Health Dashboard
            </h1>
            <p style={{ margin: '5px 0 0 0', color: '#7f8c8d', fontSize: '16px' }}>
              Real-time monitoring and analytics • Last updated: {currentTime.toLocaleTimeString()}
            </p>
          </div>
        </div>
        <div style={{ 
          backgroundColor: '#27ae60', color: 'white', padding: '15px 25px', 
          borderRadius: '25px', fontSize: '18px', fontWeight: 'bold',
          display: 'flex', alignItems: 'center'
        }}>
          <FaCheckCircle style={{ marginRight: '8px' }} />
          OPERATIONAL
        </div>
      </div>

      {/* Navigation Tabs */}
      <div style={{ 
        display: 'flex', marginBottom: '25px', backgroundColor: 'white', 
        borderRadius: '12px', padding: '5px', boxShadow: '0 2px 10px rgba(0,0,0,0.1)'
      }}>
        {[
          { id: 'overview', label: 'Overview', icon: FaChartLine },
          { id: 'alerts', label: 'Alerts', icon: FaBell },
          { id: 'logs', label: 'Logs', icon: FaList },
          { id: 'services', label: 'Services', icon: FaShieldAlt }
        ].map(tab => (
          <button
            key={tab.id}
            onClick={() => setActiveTab(tab.id)}
            style={{
              flex: 1, padding: '15px 20px', border: 'none', borderRadius: '8px',
              backgroundColor: activeTab === tab.id ? '#3498db' : 'transparent',
              color: activeTab === tab.id ? 'white' : '#7f8c8d',
              cursor: 'pointer', fontSize: '16px', fontWeight: '600',
              display: 'flex', alignItems: 'center', justifyContent: 'center',
              transition: 'all 0.3s ease'
            }}
          >
            <tab.icon style={{ marginRight: '8px' }} />
            {tab.label}
          </button>
        ))}
      </div>

      {/* Content Area */}
      <div style={{ 
        backgroundColor: 'white', borderRadius: '12px', padding: '25px',
        boxShadow: '0 2px 10px rgba(0,0,0,0.1)', minHeight: '500px'
      }}>
        {activeTab === 'overview' && renderOverview()}
        {activeTab === 'alerts' && renderAlerts()}
        {activeTab === 'logs' && renderLogs()}
        {activeTab === 'services' && renderServices()}
      </div>

      <style>{`
        @keyframes pulse {
          0% { transform: scale(1); }
          50% { transform: scale(1.1); }
          100% { transform: scale(1); }
        }
      `}</style>
    </div>
  );
};

export default ComprehensiveSystemHealth;