import React, { useState, useEffect } from 'react';
import { FaMicrochip, FaMemory, FaHdd, FaDatabase, FaSync } from 'react-icons/fa';
import { SystemMonitoringAPI } from '../utils/api';
import { MetricHistoryPoint } from '../types';

const MetricsChart: React.FC = () => {
  const [selectedMetric, setSelectedMetric] = useState('cpu_usage');
  const [timeRange, setTimeRange] = useState(24);
  const [data, setData] = useState<MetricHistoryPoint[]>([]);
  const [loading, setLoading] = useState(false);

  const metrics = [
    { key: 'cpu_usage', label: 'CPU Usage', icon: FaMicrochip, color: '#e74c3c' },
    { key: 'memory_usage', label: 'Memory Usage', icon: FaMemory, color: '#3498db' },
    { key: 'disk_usage', label: 'Disk Usage', icon: FaHdd, color: '#f39c12' },
    { key: 'response_time', label: 'Response Time', icon: FaDatabase, color: '#27ae60' }
  ];

  const fetchMetrics = async () => {
    try {
      setLoading(true);
      const result = await SystemMonitoringAPI.getMetricsHistory(selectedMetric, timeRange);
      setData(result);
    } catch (error) {
      console.error('Error fetching metrics:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchMetrics();
  }, [selectedMetric, timeRange]);

  const currentMetric = metrics.find(m => m.key === selectedMetric);
  const maxValue = Math.max(...data.map(d => d.value));
  const minValue = Math.min(...data.map(d => d.value));

  return (
    <div className="metrics-chart">
      <div className="chart-header">
        <div className="metric-selector">
          {metrics.map((metric) => {
            const IconComponent = metric.icon;
            return (
              <button
                key={metric.key}
                className={`metric-btn ${selectedMetric === metric.key ? 'active' : ''}`}
                onClick={() => setSelectedMetric(metric.key)}
              >
                <IconComponent />
                <span>{metric.label}</span>
              </button>
            );
          })}
        </div>

        <div className="chart-controls">
          <select 
            value={timeRange} 
            onChange={(e) => setTimeRange(Number(e.target.value))}
          >
            <option value={1}>Last Hour</option>
            <option value={6}>Last 6 Hours</option>
            <option value={24}>Last 24 Hours</option>
            <option value={168}>Last Week</option>
          </select>
          <button onClick={fetchMetrics} className="refresh-btn">
            <FaSync className={loading ? 'spinning' : ''} />
          </button>
        </div>
      </div>

      <div className="chart-content">
        {loading ? (
          <div className="chart-loading">Loading metrics...</div>
        ) : data.length === 0 ? (
          <div className="chart-empty">No data available</div>
        ) : (
          <>
            <div className="chart-stats">
              <div className="stat-item">
                <span className="stat-label">Current</span>
                <span className="stat-value">
                  {data[data.length - 1]?.value.toFixed(1)} {data[0]?.unit}
                </span>
              </div>
              <div className="stat-item">
                <span className="stat-label">Max</span>
                <span className="stat-value">
                  {maxValue.toFixed(1)} {data[0]?.unit}
                </span>
              </div>
              <div className="stat-item">
                <span className="stat-label">Min</span>
                <span className="stat-value">
                  {minValue.toFixed(1)} {data[0]?.unit}
                </span>
              </div>
            </div>

            <div className="simple-chart">
              {data.map((point, index) => (
                <div key={index} className="chart-bar">
                  <div 
                    className="bar-fill"
                    style={{ 
                      height: `${((point.value - minValue) / (maxValue - minValue)) * 100}%`,
                      backgroundColor: currentMetric?.color 
                    }}
                  ></div>
                  <span className="bar-value">{point.value.toFixed(1)}</span>
                </div>
              ))}
            </div>
          </>
        )}
      </div>
    </div>
  );
};

export default MetricsChart;