import React, { useState, useEffect } from 'react';
import { FaHeartbeat, FaExclamationTriangle, FaCheckCircle } from 'react-icons/fa';

const SystemHealthDebug: React.FC = () => {
  const [apiStatus, setApiStatus] = useState('Testing...');
  const [healthData, setHealthData] = useState<any>(null);
  const [statsData, setStatsData] = useState<any>(null);
  const [error, setError] = useState<string | null>(null);
  const [logs, setLogs] = useState<string[]>([]);

  const addLog = (message: string) => {
    setLogs(prev => [...prev, `${new Date().toLocaleTimeString()}: ${message}`]);
  };

  const testAPI = async () => {
    addLog('Starting API connection test...');
    
    try {
      // Test basic connectivity with auth token
      addLog('Testing basic connectivity to http://localhost:8090');
      const token = localStorage.getItem('access_token');
      addLog(`Using auth token: ${token ? 'Found' : 'Not found'}`);
      
      // First try without auth to see if endpoint exists
      let response = await fetch('http://localhost:8090/api/system_monitoring/health', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json'
        },
      });
      
      addLog(`Response without auth: ${response.status}`);
      
      // If 401, try with auth token
      if (response.status === 401 && token) {
        addLog('Trying with auth token...');
        response = await fetch('http://localhost:8090/api/system_monitoring/health', {
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`
          },
        });
      }
      
      addLog(`Response status: ${response.status}`);
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }
      
      const data = await response.json();
      addLog('✅ Successfully received health data');
      setHealthData(data);
      setApiStatus('Connected');
      
      // Test dashboard stats with auth
      addLog('Fetching dashboard stats...');
      const statsResponse = await fetch('http://localhost:8090/api/system_monitoring/dashboard/stats', {
        headers: {
          'Content-Type': 'application/json',
          ...(token && { 'Authorization': `Bearer ${token}` })
        }
      });
      const statsResult = await statsResponse.json();
      addLog('✅ Successfully received stats data');
      setStatsData(statsResult);
      
    } catch (err: any) {
      addLog(`❌ API Error: ${err.message}`);
      setError(err.message);
      setApiStatus('Failed');
    }
  };

  useEffect(() => {
    testAPI();
    const interval = setInterval(testAPI, 15000); // Test every 15 seconds
    return () => clearInterval(interval);
  }, []);

  return (
    <div style={{ padding: '20px', fontFamily: 'monospace' }}>
      <div style={{ 
        display: 'flex', alignItems: 'center', marginBottom: '20px',
        padding: '15px', backgroundColor: '#f8f9fa', borderRadius: '8px'
      }}>
        <FaHeartbeat style={{ marginRight: '10px', fontSize: '24px', color: '#e74c3c' }} />
        <h2 style={{ margin: 0 }}>System Health API Debug</h2>
      </div>

      {/* API Status */}
      <div style={{ 
        padding: '15px', marginBottom: '20px', borderRadius: '8px',
        backgroundColor: apiStatus === 'Connected' ? '#d4edda' : apiStatus === 'Failed' ? '#f8d7da' : '#fff3cd',
        border: `1px solid ${apiStatus === 'Connected' ? '#c3e6cb' : apiStatus === 'Failed' ? '#f5c6cb' : '#ffeaa7'}`
      }}>
        <div style={{ display: 'flex', alignItems: 'center' }}>
          {apiStatus === 'Connected' ? (
            <FaCheckCircle style={{ color: '#155724', marginRight: '10px' }} />
          ) : (
            <FaExclamationTriangle style={{ color: '#721c24', marginRight: '10px' }} />
          )}
          <strong>API Status: {apiStatus}</strong>
        </div>
        {error && <div style={{ marginTop: '10px', color: '#721c24' }}>Error: {error}</div>}
      </div>

      {/* Real Data Display */}
      {healthData && (
        <div style={{ 
          padding: '15px', marginBottom: '20px', backgroundColor: '#d4edda', 
          borderRadius: '8px', border: '1px solid #c3e6cb'
        }}>
          <h3 style={{ margin: '0 0 10px 0', color: '#155724' }}>✅ Real Health Data Received:</h3>
          <div style={{ backgroundColor: '#155724', color: 'white', padding: '10px', borderRadius: '4px' }}>
            <pre>{JSON.stringify(healthData, null, 2)}</pre>
          </div>
        </div>
      )}

      {statsData && (
        <div style={{ 
          padding: '15px', marginBottom: '20px', backgroundColor: '#d4edda', 
          borderRadius: '8px', border: '1px solid #c3e6cb'
        }}>
          <h3 style={{ margin: '0 0 10px 0', color: '#155724' }}>✅ Real Stats Data Received:</h3>
          <div style={{ backgroundColor: '#155724', color: 'white', padding: '10px', borderRadius: '4px' }}>
            <pre>{JSON.stringify(statsData, null, 2)}</pre>
          </div>
        </div>
      )}

      {/* Debug Logs */}
      <div style={{ 
        padding: '15px', backgroundColor: '#2c3e50', borderRadius: '8px',
        color: '#ecf0f1', maxHeight: '300px', overflowY: 'auto'
      }}>
        <h3 style={{ margin: '0 0 10px 0', color: '#ecf0f1' }}>Debug Log:</h3>
        {logs.map((log, index) => (
          <div key={index} style={{ marginBottom: '5px', fontSize: '12px' }}>
            {log}
          </div>
        ))}
      </div>

      {/* Quick Actions */}
      <div style={{ marginTop: '20px', display: 'flex', gap: '10px' }}>
        <button 
          onClick={testAPI}
          style={{ 
            padding: '10px 20px', backgroundColor: '#3498db', color: 'white',
            border: 'none', borderRadius: '4px', cursor: 'pointer'
          }}
        >
          Test API Again
        </button>
        <button 
          onClick={() => window.open('http://localhost:8090/docs', '_blank')}
          style={{ 
            padding: '10px 20px', backgroundColor: '#27ae60', color: 'white',
            border: 'none', borderRadius: '4px', cursor: 'pointer'
          }}
        >
          Open API Docs
        </button>
      </div>

      {/* Instructions */}
      <div style={{ 
        marginTop: '20px', padding: '15px', backgroundColor: '#fff3cd', 
        borderRadius: '8px', border: '1px solid #ffeaa7'
      }}>
        <h4 style={{ margin: '0 0 10px 0', color: '#856404' }}>Troubleshooting:</h4>
        <ol style={{ margin: 0, color: '#856404' }}>
          <li>Make sure backend is running: <code>cd NAL-BE-sindhuranihr && python -m src</code></li>
          <li>Check if port 8090 is accessible: <code>http://localhost:8090/docs</code></li>
          <li>Verify CORS settings allow localhost:5173</li>
          <li>Check browser console for network errors</li>
        </ol>
      </div>
    </div>
  );
};

export default SystemHealthDebug;